# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
import hashlib
import itertools
import re
import six

import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2

import sandbox.projects.common.error_handlers as eh
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.helpers.events_helper as events_helper
import sandbox.projects.release_machine.changelogs as rm_ch
import sandbox.projects.release_machine.helpers.wiki_helper as wh
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.tasks.base_task as rm_bt
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common import time_utils
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.client import RMClient
from sandbox.projects.release_machine.components.configs.release_machine_test import ReleaseMachineTestCfg
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine.helpers import solomon_helper


@rm_notify.notify2()
class ReleaseMonitorCrawlReleases(binary_task.LastBinaryTaskRelease, rm_bt.ComponentErrorReporter, sdk2.Task):

    _result_path = "tmp_results.txt"

    class Requirements(task_env.StartrekRequirements):
        disk_space = 1024  # 1 Gb

    class Parameters(rm_params.BaseReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        debug = sdk2.parameters.Bool("Debug", default_value=False)
        component_name_filter = sdk2.parameters.String("Component name filter")

    class Context(sdk2.Task.Context):
        fail_on_any_error = True
        current_component_name = ReleaseMachineTestCfg.name

    @decorators.memoized_property
    def rm_model_client(self):
        from release_machine.release_machine.services.release_engine.services.Model import ModelClient
        return ModelClient.from_address(rm_const.Urls.RM_HOST)

    @decorators.memoized_property
    def rm_event_helper(self):
        return events_helper.EventHelper()

    @decorators.memoized_property
    def rm_token(self):
        return rm_sec.get_rm_token(self)

    @decorators.memoized_property
    def st_helper(self):
        return STHelper(self.rm_token)

    @decorators.memoized_property
    def rm_http_custom_client(self):
        return RMClient()

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        failed_components = []
        tokens = {}

        for component_name in self._get_component_names():

            logging.info("\nCurrent component: %s", component_name)

            try:

                c_info = rmc.get_component(component_name)
                self.Context.current_component_name = component_name

                logging.info("c_info obtained successfully")

                token = self._get_token_for_deploy(c_info, tokens)
                deploy_data_list = c_info.get_last_deploy(token)

                logging.info("c_info.get_last_deploy response: %s", deploy_data_list)

                if deploy_data_list:
                    for deploy_data in deploy_data_list:
                        self._process_deploy_data(c_info, deploy_data)
                    self.Context.tm_message = None
                else:
                    logging.info("[%s] No deploy data found", component_name)

                self.process_deploy_proto(c_info, token)

            except Exception as exc:
                self._report_error(component_name, 'Failed component processing', exc)
                failed_components.append(component_name)
            finally:
                logging.info("%s - processing finished\n")

        eh.ensure(
            not failed_components,
            'Failed components: {}'.format(', '.join(failed_components))
        )

    def process_deploy_proto(self, c_info, token):
        """New way of processing deploy info. RMDEV-78"""

        logging.info("%s.process_deploy_proto", self.__class__.__name__)

        if not c_info.releases_cfg__releasable_items:
            logging.info("%s does not define any releasable items. No deploy proto data to process", c_info.name)
            return

        new_component_versions = c_info.get_last_deploy_proto(token)

        if not new_component_versions:
            logging.info("New RM component versions not found, skip processing.")
            return

        logging.info("Got new RM component versions:\n%s", new_component_versions)

        old_component_versions = self._get_rm_component_versions(c_info.name)
        logging.info("Got old RM component versions:\n%s", old_component_versions)

        if old_component_versions != new_component_versions:
            logging.info("[%s] Found diff in versions", c_info.name)
            self._send_component_versions_change_event(c_info, old_component_versions, new_component_versions)
        else:
            logging.debug("[%s] Versions are the same", c_info.name)

    def _get_rm_component_versions(self, component_name):
        # import grpc
        from release_machine.release_machine.proto.structures import message_pb2

        try:
            return self.rm_model_client.get_component_versions(
                message_pb2.ComponentRequest(component_name=component_name)
            )
        except Exception:  # grpc.RpcError:
            logging.exception("Unable to get component versions for component %s", component_name)

    def _send_component_versions_change_event(self, c_info, old_versions, new_versions):
        from release_machine.common_proto import events_pb2

        logging.info(
            "Send component versions change event:\n  OLD: %s\n  NEW: %s",
            old_versions,
            new_versions,
        )

        try:
            version_change_event = events_pb2.EventData(
                general_data=events_pb2.EventGeneralData(
                    hash=hashlib.md5("{}{}{}".format(c_info.name, old_versions, new_versions)).hexdigest(),
                    component_name=c_info.name,
                    referrer="sandbox_task:{}".format(self.id),
                ),
                task_data=events_pb2.EventSandboxTaskData(
                    task_id=self.id,
                    status=self.status,
                    created_at=self.created.isoformat(),
                    updated_at=self.updated.isoformat(),
                ),
                component_versions_change_data=events_pb2.ComponentVersionsChangeData(
                    old_versions=old_versions,
                    new_versions=new_versions,
                ),
            )
            logging.debug("Versions change event:\n%s", version_change_event)
            self.rm_event_helper.post_grpc_events([version_change_event])
        except Exception as e:  # grpc.RpcError:
            eh.log_exception("Unable to send event abound versions change for component {}".format(c_info.name), e)

    def _get_component_names(self):
        """Get filtered component names"""
        name_filter_re = re.compile(self.Parameters.component_name_filter or ".*", re.IGNORECASE)
        return filter(name_filter_re.match, rmc.get_component_names())

    def _get_token_for_deploy(self, c_info, tokens):
        key = "{}_{}".format(c_info.releases_cfg__token_vault_owner, c_info.releases_cfg__token_vault_name)
        if key not in tokens:
            logging.debug("Request for token for key: %s", key)
            try:
                tokens[key] = sdk2.Vault.data(
                    c_info.releases_cfg__token_vault_owner,
                    c_info.releases_cfg__token_vault_name
                )
            except Exception as e:
                logging.exception("Unable to get token for key %s: %s", key, e)
                logging.info("Set default token for key %s", key)
                tokens[key] = self.rm_token
        return tokens[key]

    def _process_deploy_data(self, c_info, deploy_data):
        """
        :param c_info: component info
        :param deploy_data: rm_core.DeployedResource
        :return:
        """
        logging.info("Processing deploy data %s", deploy_data)

        if not isinstance(deploy_data, rm_core.ReleasedResource):
            self._report_error(
                c_info.name,
                'Wrong type of release data: {}. Should be `ReleasedResource`'.format(type(deploy_data)),
            )
            return
        if not deploy_data.major_release:
            # Do not process component, if scope_number cannot be detected
            logging.warning('[%s] Cannot get major_release from data: %s', c_info.name, deploy_data)
            return
        try:
            curr_version = self.rm_http_custom_client.get_current_version(
                c_info.name, deploy_data.status, deploy_data.resource_name
            )
            diff = c_info.release_diff(curr_version, deploy_data)
            if diff.position is rm_core.ReleaseDiff.Position.same:
                logging.info("[%s] No diff found, continue", c_info.name)
                return  # No need to send event in this case, it will be a duplicate
            else:
                self._report_success(c_info.name, '{} `{}` release found: {}'.format(
                    rm_core.ReleaseDiff.POS[diff.position], diff.staging, deploy_data
                ))
                if diff.position is rm_core.ReleaseDiff.Position.new:
                    self.post_deploy_process(c_info, deploy_data, diff.release_type)
            if c_info.is_branched_or_ci:
                tag_number = deploy_data.minor_release
            else:
                tag_number = deploy_data.major_release
            self._post_new_style_release_deployed_event(
                component_name=c_info.name,
                scope_number=deploy_data.major_release,
                tag_number=tag_number,
                deploy_time=deploy_data.timestamp,
                deploy_level=deploy_data.status,
                build_task_id=deploy_data.build_task_id,
                resource_id=deploy_data.id,
                nanny_ticket=deploy_data.info,
                task_status="SUCCESS",
                released_resource_name=deploy_data.resource_name,
                is_major=(diff.release_type == rm_core.ReleaseDiff.Type.major),
            )
        except Exception as exc:
            self._report_error(c_info.name, 'Failed to process release data', exc)

    def _post_new_style_release_deployed_event(
        self,
        component_name,
        scope_number,
        tag_number,
        deploy_time,
        deploy_level,
        build_task_id,
        resource_id,
        nanny_ticket,
        task_status,
        released_resource_name,
        is_major=False,
    ):
        from release_machine.common_proto import events_pb2

        event = events_pb2.EventData(
            general_data=events_pb2.EventGeneralData(
                hash=six.text_type(hash((
                    component_name,
                    'ReleaseDeployed',
                    "sandbox_task:{}".format(self.id),
                    scope_number or 0,
                    tag_number or 0,
                    resource_id or 0,
                    build_task_id or 0,
                    deploy_level or "",
                    deploy_time or 0,
                    released_resource_name or "",
                ))),
                component_name=component_name,
                referrer="sandbox_task:{}".format(self.id),
            ),
            task_data=events_pb2.EventSandboxTaskData(
                task_id=self.id,
                status=task_status,
                created_at=self.created.isoformat(),
                updated_at=time_utils.datetime_utc_iso(),
            ),
            release_deployed_data=events_pb2.ReleaseDeployedData(
                scope_number=str(scope_number),
                tag_number=str(tag_number),
                deploy_timestamp=int(deploy_time),
                deploy_level=deploy_level,
                released_resource_info=events_pb2.ReleasedResourceInfo(
                    resource_id=str(resource_id) if resource_id else "",
                    resource_type=self.server.resource[resource_id].read()['type'] if resource_id else "",
                    resource_name=released_resource_name or "",
                    nanny_ticket_key=nanny_ticket,
                    build_task_id=str(build_task_id) if build_task_id else "",
                ),
                is_major=is_major,
            ),
        )
        if self.Parameters.debug:
            logging.debug("Debug mode, do not send event")
            return
        return events_helper.post_proto_events(event_messages=[event])

    @staticmethod
    def _get_release_message(c_info, release_type, release_name, release_status, tags=""):
        return (
            "[{component_name}] {tags}\n"
            "New {release_type} release {release_name} is "
            "deployed to {release_status} services.".format(
                component_name=c_info.name,
                tags=tags,
                release_type=release_type,
                release_name=release_name,
                release_status=release_status,
            )
        )

    @staticmethod
    def get_changes_from_changelog(c_info, current_release):
        """
        :param c_info: component info
        :param current_release: instance of ReleasedResource
        :return: list of `changes`, each `change` is a dict:
            - revision: int
            - commit_author: str
            - commit_message: str
            - review_ids: List[int]
            - added: bool
            - startrek_tickets: List[str]
            - te_problem_owner: List[?]
            - commit_importance: int
            - summary: str
            - reasons: List[str]
            - te_problems: dict
            - revision_paths: List[str]

        """
        full_changelog_res = rm_ch.get_changelog_resource(
            c_info.name, current_release.major_release, current_release.minor_release
        )
        if full_changelog_res is not None:
            logging.info("Got full changelog: %s", full_changelog_res.id)
            changes = {}
            full_changelog = rm_ch.get_rm_changelog(full_changelog_res)
            for all_changes in full_changelog.get("all_changes", []):
                # todo: get only one change part, related to current_release
                for change in all_changes.get("changes", []):
                    if change["revision"] not in changes:
                        changes[change["revision"]] = change
            return changes.values()
        return []

    def notify_on_post_deploy(self, c_info, current_release, release_type, added_changes):
        """
        Post a deploy notification comment to feature tickets (if required)

        :param c_info: component info
        :param current_release: instance of ReleasedResource
        :param release_type: (str) major or minor
        :param added_changes: list of (added) changes, each change is a dict (see get_changes_from_changelog doc)
        :return: None
        """
        if not c_info.notify_cfg__st__notify_on_deploy_to_feature_st_tickets:
            return

        if release_type == rm_core.ReleaseDiff.Type.minor:
            return

        if (
            not isinstance(c_info.notify_cfg__st__notify_on_deploy_to_feature_st_tickets, bool) and
            current_release.status not in c_info.notify_cfg__st__notify_on_deploy_to_feature_st_tickets
        ):
            logging.info(
                "SKIP. The given release status (%s) does not belong to the set of stages accepted for deployment "
                "notifications provided in %s's config: %s",
                current_release.status,
                c_info.name,
                c_info.notify_cfg__st__notify_on_deploy_to_feature_st_tickets,
            )
            return

        try:
            release_ticket_issue = self.st_helper.find_ticket_by_release_number(current_release.major_release, c_info)
            issues = []

            for tickets, changes in itertools.groupby(sorted(added_changes, key=wh.st_tickets), key=wh.st_tickets):

                if tickets == [wh.NO_ISSUE]:
                    logging.info("No tickets found for changes %s", changes)
                    continue

                revs_in_changes = [c["revision"] for c in changes]
                plural = len(revs_in_changes) > 1

                for ticket in c_info.filter_tickets(changes):

                    if "IGNIETFERRO" in ticket:
                        # SPIKE!!! Remove it after RMDEV-2551
                        continue

                    issue = self.st_helper.get_ticket_by_key(ticket)
                    issues.append(issue)

                    message = self._get_ticket_deploy_comment_message(
                        c_info=c_info,
                        plural=plural,
                        revs_in_changes=revs_in_changes,
                        current_release=current_release,
                        release_ticket_issue=release_ticket_issue,
                    )

                    logging.debug("Ticket %s\nMessage %s", ticket, message)

                    self.st_helper.create_comment(
                        issue, message,
                        notify=c_info.notify_cfg__st__notify_on_robot_comments_to_tickets
                    )

            self._report_success(
                c_info.name,
                "Added comments to {}".format(
                    ", ".join(lb.st_link(i.key) for i in issues)
                ),
            )

        except Exception as exc:
            self._report_error(c_info.name, 'Failed post-deploy notification', exc)

    def _get_ticket_deploy_comment_message(
        self,
        c_info,
        plural,
        revs_in_changes,
        current_release,
        release_ticket_issue,
    ):
        return (
            "Revision{s} {revs} {was} deployed to {stage} from component {component} "
            "{release_ticket}. Checked by sandbox task: {task}. If you find this comment wrong, "
            "check our {docs} and after that come to RM Support {chat}".format(
                s='s' if plural else '',
                revs=','.join([str(rev) for rev in revs_in_changes]),
                was='were' if plural else 'was',
                stage=current_release.status,
                component=c_info.name,
                release_ticket=(
                    lb.st_wiki_link(release_ticket_issue.key) if release_ticket_issue else ''
                ),
                task=lb.task_wiki_link(self.id),
                docs=lb.WIKILINK_TO_ITEM.format(
                    link="https://nda.ya.ru/t/cOsVGQTe3ctYFr", name="documentation",
                ),
                chat=lb.WIKILINK_TO_ITEM.format(link="https://nda.ya.ru/3UXG4F", name="chat"),
            )
        )

    def post_deploy_process(self, c_info, deploy_data, release_type):
        """

        :param c_info: instance of ComponentInfoGeneral
        :param deploy_data: instance of ReleasedResource
        :param release_type: (str) major or minor
        :return:
        """
        if self.Parameters.debug:
            logging.debug("Debug mode, do not do post deploy processing")
            return
        major_release = int(deploy_data.major_release)
        release_name = c_info.release_id(deploy_data.major_release, deploy_data.minor_release)
        create_notify = True
        changes = self.get_changes_from_changelog(c_info, deploy_data)
        added_changes = [c for c in changes if c["added"]]

        if c_info.notify_cfg__use_startrek:
            notify_message = self._get_release_message(
                c_info,
                release_type,
                release_name,
                deploy_data.status,
            )
            comment, issue = self.st_helper.find_comment(major_release, notify_message, c_info)
            if comment:
                create_notify = False
                if False:
                    # disable warnings, stop spam (RMDEV-1143)
                    rm_notify.send_tm_message(
                        self,
                        (
                            "WARNING! Multiple deploy notifications for component: [{component}]\n"
                            "SB: {task_link} ST: {st_link}"
                        ).format(
                            component=c_info.name,
                            task_link=lb.task_link(self.id),
                            st_link=lb.st_link(issue.key),
                        ),
                        [rm_const.RM_USERS["rm_maintainers"].telegram],
                    )
            else:
                if deploy_data.status == rm_const.ReleaseStatus.stable:
                    c_info.change_ticket_on_post_deploy(major_release, self.st_helper, self)
                    c_info.cleanup_on_post_deploy(major_release, self.st_helper)
                self.notify_on_post_deploy(c_info, deploy_data, release_type, added_changes)
                notify_message += " SB: {sb_link}\n".format(sb_link=lb.task_wiki_link(self.id))
                release_ticket = self.st_helper.comment(major_release, notify_message, c_info, fail=False)
                if not release_ticket:
                    logging.warning(
                        "Unable to find Startrek issue for major release = %s, so notify message '%s' not posted",
                        major_release,
                        notify_message,
                    )
                if c_info.notify_cfg__st__close_prev_tickets_stage == rm_const.PipelineStage.deploy:
                    self.st_helper.close_prev_tickets(
                        c_info, major_release,
                        "RM crawler found new deployed items. Closing previous tickets"
                    )
        # New release is deployed - check if new branch exists for autobranched components

        if added_changes:
            solomon_api = solomon_helper.SolomonApi(token=self.rm_token)
            response = solomon_api.post_number_commits_in_release_to_solomon(len(added_changes), c_info.name)
            logging.debug("Got response from Solomon: [%s] '%s'", response.status_code, response.content)

        if create_notify:
            notify_message = self._get_release_message(
                c_info,
                release_type,
                release_name,
                release_status=deploy_data.status,
                tags="#release #r{}".format(major_release),
            ) + " SB: {sb_link}\n".format(sb_link=lb.task_link(self.id))
            self.Context.tm_message = notify_message
            self.send_tm_notify()
