# -*- coding: utf-8 -*-
import datetime as dt
import json
import logging

import sandbox.sdk2 as sdk2
import sandbox.projects.common.binary_task as binary_task
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.common.gsid_parser as gsid_parser
import sandbox.projects.common.time_utils as time_utils
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.components.components_info as rm_comp
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.events_helper as events_helper
import sandbox.projects.release_machine.helpers.startrek_helper as st_helper
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.security as rm_sec
import sandbox.projects.release_machine.rm_notify as rm_notify

from sandbox.projects.release_machine.components.configs.release_machine_test import ReleaseMachineTestCfg


@rm_notify.notify2()
class ReleaseMonitorCrawlTickets(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Collect last 10 acceptance ticket history and send events on them."""

    class Requirements(task_env.StartrekRequirements):
        pass

    class Parameters(rm_params.BaseReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

    class Context(sdk2.Task.Context):
        fail_on_any_error = True
        component_name = None

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        rm_token = rm_sec.get_rm_token(self)
        sth = st_helper.STHelper(rm_token)
        for component_name in rmc.get_component_names():
            logging.info('Current component: %s', component_name)

            if component_name == ReleaseMachineTestCfg.name:
                logging.info('Skip `%s` component', component_name)
                continue

            c_info = rmc.COMPONENTS[component_name]()
            if isinstance(c_info, rm_comp.mixin.Startreked):
                logging.info('Process component: %s', component_name)
                self.Context.component_name = component_name

                try:
                    proto_events = list(self.get_ticket_history_proto_events(c_info, sth))
                    if proto_events:
                        events_helper.post_proto_events(proto_events)
                except Exception as e:
                    eh.log_exception("Unable to send proto events", e)

                self.update_st_assignee(c_info, sth)
            else:
                logging.warning("Skip component {}, it is not Startreked".format(component_name))

    @property
    def is_binary_run(self):
        return self.Parameters.binary_executor_release_type != 'none'

    @staticmethod
    def update_st_assignee(c_info, sth):
        try:
            if c_info.notify_cfg__st__time_to_update_assignee:
                sth.change_assignee(c_info, c_info.last_scope_num)
        except Exception as e:
            logging.exception("Unable to update st assignee: %s", e, exc_info=True)

    def get_ticket_history_proto_events(self, c_info, sth):
        """
        Build and yield RM new-style events for the given component's tickets.

        :param c_info: component info instance
        :param sth: Startrek helper instance
        """
        if not self.is_binary_run:
            return

        from release_machine.common_proto import events_pb2
        from release_machine.public import events as events_public

        issues = self._get_issues(c_info, sth)
        for issue in issues:
            processed_changelog = self._process_issue(issue)

            release_num = c_info.get_release_num_from_st_summary(issue.summary)

            if not release_num:
                continue

            specific_data = events_pb2.TicketHistoryData(
                ticket_key=issue.key,
                scope_number=str(release_num),
                job_name=gsid_parser.get_job_name_from_gsid(self.Context.__values__.get("__GSID")),
                ticket_history_latest_status=processed_changelog[-1][0],
            )

            specific_data.items.extend([events_pb2.TicketHistoryDataItem(
                status=status.lower(),
                start_timestamp=int(start_time or 0),
                end_timestamp=int(end_time or 0),
            ) for status, start_time, end_time in processed_changelog])

            ticket_status_log = [
                {
                    "status": status.lower(),
                    "start_time": start_time or 0,
                    "end_time": end_time or 0,
                } for status, start_time, end_time in processed_changelog
            ]

            ticket_status_log_json = json.dumps(ticket_status_log, sort_keys=True)

            logging.info("Ticket status log (affects event hash): %s", ticket_status_log_json)

            hash_items = (
                c_info.name,
                'TicketHistory',
                issue.key,
                ticket_status_log_json,
            )

            hash_str = events_public.get_event_hash(*hash_items)

            event = events_pb2.EventData(
                general_data=events_pb2.EventGeneralData(
                    component_name=c_info.name,
                    referrer=u"sandbox_task:{}".format(self.id),
                    hash=hash_str,
                ),
                task_data=events_pb2.EventSandboxTaskData(
                    task_id=self.id,
                    status=self.status,
                    created_at=self.created.isoformat(),
                    updated_at=time_utils.datetime_utc_iso(),
                ),
                ticket_history_data=specific_data,
            )

            logging.debug("Got proto event: %s", event)

            yield event

    @staticmethod
    def _get_issues(c_info, sth):
        if not isinstance(c_info.st_tags, (list, tuple)):
            logging.debug("Bad st_tags format in Component Info, st_tags should be a list")
            return []
        since_time = dt.datetime.now() - dt.timedelta(days=1)
        query = " ".join([
            "(Queue: {})".format(c_info.notify_cfg__st__queue),
            " ".join(["AND Tags: {}".format(st_tag) for st_tag in c_info.st_tags]),
            "AND Updated: >=\"{}\"".format(since_time.strftime("%Y-%m-%d %H:%M:%S")),
            "\"Sort By\": Key DESC"
        ])
        logging.info("Process query: {}".format(query))
        issues = list(sth.st_client.issues.find(query))
        return issues[:10]

    @staticmethod
    def _process_issue(issue):
        start_time = time_utils.to_unixtime(issue.createdAt)
        to_status_key = None
        issue_changelog = []
        for dd in issue.changelog.get_all(field="status"):
            end_time = time_utils.to_unixtime(dd.updatedAt)
            for field in dd.fields:
                if field["field"].id == "status":
                    if start_time < end_time - 5:
                        issue_changelog.append([str(field["from"].key), start_time, end_time])
                        start_time = end_time
                    to_status_key = str(field["to"].key)
        if to_status_key and to_status_key != rm_const.Workflow.CLOSED:
            issue_changelog.append([to_status_key.lower(), start_time, None])
        return issue_changelog or [["opened", start_time, None]]
