# -*- coding: utf-8 -*-
import hashlib
import logging
import time
from datetime import datetime

import sandbox.common.types.notification as ctn
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.input_params2 as rm_params
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common import time_utils
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import requests_wrapper
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine.client import RMClient
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine.helpers import events_helper as eh


@rm_notify.notify2()
class ReleaseMonitorKpiAlert(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
        Check release KPI with release monitor api
    """

    class Requirements(task_env.StartrekRequirements):
        disk_space = 5000  # 5G

    class Parameters(rm_params.BaseReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        email_notify = sdk2.parameters.Bool("Send notify by email", default=False)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        rm_client = RMClient()
        event_client = eh.EventHelper()

        for component_name in rmc.get_component_names():

            c_info = rmc.COMPONENTS[component_name]()

            if c_info.kpi_alert_allowed():

                setattr(self.Context, "component_name", component_name)
                logging.info("Current component: %s", component_name)
                scope_data = rm_client.get_scopes(component_name, limit=1).get("branchScopes")
                if not scope_data:
                    self.set_info("No scopes for component: {}".format(component_name))
                    continue
                curr_version = rm_client.get_current_version(component_name)
                if not curr_version or not curr_version.get("scope_number"):
                    self.set_info("There is no releases for '{}' component".format(c_info.name))
                    continue
                scope_data = scope_data[0]
                current_scope = scope_data["scopeNumber"]
                message = ""
                subject = ""
                if self._check_priemka(curr_version["scope_number"], current_scope):
                    message = (
                        '[{component_name}] It\'s time to create new branch.\n'
                        'Release <a href="{rm_url}/component/{component_name}/manage">'
                        '{major_release}-{minor_release}</a> has been deployed to production.\n'
                        'Just press <a href="{testenv_url}/?screen=pre_release_action&amp;database={te_db}">'
                        'Pre-release</a> button please'.format(
                            rm_url=rm_const.Urls.RM_URL,
                            testenv_url=rm_const.Urls.TESTENV,
                            component_name=component_name,
                            major_release=curr_version["scope_number"],
                            minor_release=curr_version.get("tag_number"),
                            te_db=c_info.testenv_cfg__trunk_db,
                        )
                    )
                    subject = "[{}] Time to create new branch".format(component_name)

                    event_client.post_grpc_events(
                        events=[
                            self._get_time_to_create_new_branch_event(
                                component_name=component_name,
                                scope_number=curr_version['scope_number'],
                                tag_number=curr_version['tag_number'],
                                trunk_te_db=c_info.testenv_cfg__trunk_db,
                            ),
                        ]
                    )

                if not message and self._check_last_release(current_scope, curr_version.get("deploy_time"), c_info):
                    subject = "[{}] Time to finish priemka".format(component_name)
                    st_ticket_history = scope_data.get("stTicket")
                    st_ticket = st_ticket_history[0]["ticketKey"] if st_ticket_history else None
                    acceptance_ticket_link = lb.st_link(st_ticket, plain=True) if st_ticket else "TICKET_NOT_FOUND"
                    message = (
                        "[{component_name}] Hurry up! Last release creation time: {last_time}.\n"
                        "It's time to deploy `{component_name}-{branch_num}` release "
                        "from {st_ticket}\n".format(
                            last_time="{}Z".format(
                                datetime.utcfromtimestamp(int(curr_version.get("deploy_time"))).isoformat()
                            ),
                            component_name=component_name,
                            st_ticket=acceptance_ticket_link,
                            branch_num=current_scope,
                        )
                    )

                    event_client.post_grpc_events(events=[
                        self._get_time_to_release_event(
                            component_name=component_name,
                            last_release_time="{}Z".format(
                                datetime.utcfromtimestamp(int(curr_version.get("deploy_time"))).isoformat(),
                            ),
                            scope_number=current_scope,
                            st_ticket=st_ticket,
                        )
                    ]),

                if message:
                    self.Context.message = message
                    setattr(self.Context, rm_notify.TM_MESSAGE, message)
                    if self.Parameters.email_notify:
                        self.Context.email_notification_id = self.server.notification(
                            subject=subject,
                            body=message,
                            recipients=c_info.release_viewer_cfg__kpi_alerts_recipients,
                            transport=ctn.Transport.EMAIL,
                        )["id"]
                    self.set_info("KPI alert was created for '{}'".format(component_name))
                    self.send_tm_notify()

    @staticmethod
    def _check_priemka(last_major_release, current_scope):
        if str(current_scope) == str(last_major_release):
            return True
        return False

    def _check_last_release(self, current_scope, deploy_time, c_info):
        try:
            release_num = int(current_scope)
            st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
            st_issue = st_helper.find_ticket_by_release_number(release_num, c_info)
            if st_issue.status.key in [rm_const.Workflow.DEPLOYING, rm_const.Workflow.CLOSE, rm_const.Workflow.CLOSED]:
                return False
        except Exception:
            pass
        try:
            logging.debug("st issue: {}".format(st_issue))
            if hasattr(st_issue, "key"):
                logging.debug("st issue key: {}".format(st_issue.key))
            if hasattr(st_issue, "status"):
                logging.debug("st issue status: {}".format(st_issue.status))
        except Exception:
            pass

        if deploy_time < (int(time.time()) - int(c_info.release_viewer_cfg__kpi_alert) * 24 * 3600):
            return True
        return False

    @decorators.retries(5)
    def _get_json_response(self, url):
        return requests_wrapper.get(url).json()

    def _get_kpi_alert_event_general_data(self, component_name, events_pb2):

        return events_pb2.EventGeneralData(
            hash=hashlib.md5(u"{}{}{}".format(
                component_name,
                rm_const.EVENT_HASH_SEPARATOR,
                self.updated,
            )).hexdigest(),
            component_name=component_name,
            referrer="sandbox_task:{}".format(self.id),
        )

    def _get_kpi_alert_event_task_data(self, events_pb2):

        return events_pb2.EventSandboxTaskData(
            task_id=self.id,
            status=self.status,
            created_at=self.created.isoformat(),
            updated_at=time_utils.datetime_utc_iso(),
        )

    def _get_time_to_create_new_branch_event(self, component_name, scope_number, tag_number, trunk_te_db):

        from release_machine.common_proto import events_pb2

        return events_pb2.EventData(
            general_data=self._get_kpi_alert_event_general_data(component_name, events_pb2),
            task_data=self._get_kpi_alert_event_task_data(events_pb2),
            kpi_alert_data=events_pb2.KpiAlertData(
                time_to_create_new_branch=events_pb2.KpiAlertTimeToCreateNewBranchInfo(
                    deployed_scope_number=scope_number,
                    deployed_tag_number=tag_number,
                    te_db=trunk_te_db,
                ),
            ),
        )

    def _get_time_to_release_event(self, component_name, last_release_time, scope_number, st_ticket):

        from release_machine.common_proto import events_pb2

        return events_pb2.EventData(
            general_data=self._get_kpi_alert_event_general_data(component_name, events_pb2),
            task_data=self._get_kpi_alert_event_task_data(events_pb2),
            kpi_alert_data=events_pb2.KpiAlertData(
                time_to_release=events_pb2.KpiAlertTimeToReleaseInfo(
                    last_release_time=last_release_time,
                    time_to_deploy_scope_number=scope_number,
                    st_ticket=st_ticket,
                ),
            ),
        )
