# -*- coding: utf-8 -*-
import time
import datetime
import json
import logging

from dateutil.tz import gettz
from collections import defaultdict

import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2

import sandbox.common.rest as cr
import sandbox.common.types.task as ctt

import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.projects.release_machine.components.components_info as rm_comp
import sandbox.projects.release_machine.notify_helper as nh
import sandbox.projects.release_machine.rm_notify as rm_notify

from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import string
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common import time_utils as tu
from sandbox.projects.common import utils2
from sandbox.projects.common.nanny import const as nanny_const
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.release_machine import rm_utils
from sandbox.projects.release_machine.core import ReleasedItem
from sandbox.projects.release_machine.core import Ok, Error
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.common.testenv_client import TEClient
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.components import configs as rm_config
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.release_engine.BuildReleaseEngineUI import BuildReleaseEngineUI  # noqa: UnusedImport


REASON_KEY = 'reason'
BLOCK_OLD_RELEASE_TEMPLATE = (
    'Attempt to release an older release version {release_num} of {release_item_name} to '
    '{where_to_release} blocked since {component_name} does not allow releases from older branches '
    '(allow_old_releases = False).\nCurrent {where_to_release} release number is {major_num}'
)


@rm_notify.notify2()
class ReleaseRmComponent(rm_bt.BaseReleaseMachineTask):
    """
        **Release-machine**

        Task for pressing release button:

        - Automatically press "release" button for sandbox-tasks, which build specified binaries.
        - Can release to testing and to stable.
        - Removes obsolete numbered Yappy-betas (garbage collection).
        - Writes release status to release ticket.
        - Can notify responsible people in case of release fails
    """
    END_RELEASE_STATUSES = ["DEPLOY_SUCCESS", "CLOSED", "CANCELLED", "DEPLOY_FAILED"]
    MAX_WAIT_DEPLOY_NUM = 6  # 6 times * 4 hours = 24 h (KPI from thatguy@)

    class Requirements(task_env.StartrekRequirements):
        disk_space = 2 * 1024  # 2 Gb

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        fail_msg = "unknown error"
        wait_num = 0
        nanny_requests = {}
        release_num = 0
        max_minor_release_num = 0
        release_info = []

    class Parameters(rm_params.ComponentNameResources):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        subject_prefix = sdk2.parameters.String("Subject custom prefix")
        wait_for_deploy = sdk2.parameters.Bool("Wait for deploy")
        with sdk2.parameters.String("Release to", default=rm_const.ReleaseStatus.testing) as where_to_release:
            rm_params.set_choices(where_to_release, [x for x in dir(rm_const.ReleaseStatus) if not x.startswith("__")])
        release_number = sdk2.parameters.Integer("Release number")
        major_release_num = sdk2.parameters.Integer("Major release number")
        minor_release_num = sdk2.parameters.Integer("Minor release number")
        release_notes = sdk2.parameters.String("Additional comment for release", default_value="")
        tasks_to_check = sdk2.parameters.String("Tasks to check before release (divide by comma)")
        do_release = sdk2.parameters.Bool("Do you want to release this?", default=True)
        release_notes_comment = sdk2.parameters.String("Additional comment from other tasks")
        additional_addresses = sdk2.parameters.String("Additional release followers")
        urgent_release = sdk2.parameters.Bool("Urgent release")
        additional_release_parameters = sdk2.parameters.Dict("Additional release parameters")

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        self._is_to_stable = self.Parameters.where_to_release == rm_const.ReleaseStatus.stable
        self._c_info = rmc.COMPONENTS[self.Parameters.component_name]()

        if (
            not self.Context.wait_for_deploy_mark and
            self._is_to_stable and
            not self._c_info.releases_cfg__allow_robots_to_release_stable and
            not utils2.is_human(self.author)
        ):
            self.Context.fail_msg = (
                "Robots aren't allowed to make stable releases for the component: {}. "
                "Add `YourComponentInfo.Releases.allow_robots_to_release_stable = True` to your RM configuration "
                "if you want to allow robots to perform automatic releases ( one can find examples "
                "in https://nda.ya.ru/3UZoNM )".format(self._c_info.name)
            )
            eh.check_failed(self.Context.fail_msg)

        release_num = self.Parameters.release_number
        self._token = rm_sec.get_rm_token(self)
        st_helper = STHelper(self._token)

        if self.Context.wait_for_deploy_mark:
            self.wait_time_for_deploy(st_helper, self.Context.release_num)
            return

        self.check_input_tasks()
        check_result = self._c_info.check_release_blocked(
            self.Parameters.where_to_release,
            self.Parameters.release_number,
            self.Parameters.minor_release_num
        )
        logging.info("Check release block result: %s", check_result)
        if not check_result.ok:
            if check_result.result["fail_task"]:
                eh.check_failed(check_result.result["msg"])
            else:
                self.set_info("No release from this task: {}".format(check_result.result["msg"]))
                return
        prev_release_msg = self._check_release(st_helper)

        if not self.Parameters.do_release and not self.Parameters.urgent_release:
            self.set_info("Parameters.do_release is False, so no need to release")
            return

        logging.info("Component resources are: {}".format(self.Parameters.component_resources))
        if self.Parameters.urgent_release:
            self._nn_client = NannyClient(api_url=rm_const.Urls.NANNY_BASE_URL, oauth_token=self._token)
        errors = []
        services = defaultdict(list)
        released_items = []
        released_tasks = set()
        try:
            st_issue = self.get_st_issue(st_helper, release_num)
        except Exception as ex:
            logging.exception("Problems with Startrek functionality, exception: %s", ex)
            self.set_info("Can't get Startrek issue, see log for details.")
            st_issue = None
        message_processor = MessageProcessor(self, st_helper, st_issue)
        for release_item, service_ids in self._filter_resources():
            if self.Parameters.do_release and release_item.build_task_id not in released_tasks:
                check_numbers = self._check_release_numbers(release_item, release_num)
                if not check_numbers.ok:
                    errors.append(check_numbers.result)
                    continue
                release_num = check_numbers.result
                release_result = self._release_task(release_item, release_num, st_issue, add_msg=prev_release_msg)
                if release_result.ok:
                    released_tasks.add(release_item.build_task_id)
                    if release_result.result.get("release_req_id"):
                        self.Context.nanny_requests[str(release_item.resource.type)] = {
                            "release_req_id": release_result.result["release_req_id"],
                            "state": "OPEN"
                        }
                    if "release_label" in release_result.result:
                        released_items.append({
                            "item": release_item,
                            "label": release_result.result["release_label"],
                            "prev_num": release_result.result.get("prev_req_count")
                        })
                else:
                    errors.append(release_result.result)
            if self.Parameters.urgent_release:
                for service_id in service_ids:
                    services[service_id].append(release_item)

        if self.Parameters.urgent_release:
            for service_id, release_items in services.items():
                self._push_service_configuration(service_id, release_items, release_num)

        for released_item in released_items:
            wait_result = self.wait_after_release(released_item["item"])
            if not wait_result.ok:
                errors.append(wait_result.result)
                continue
            nanny_req = None
            if self._c_info.releases_cfg__deploy_system == rm_const.DeploySystem.nanny:
                nanny_req = self.wait_nanny_release_req(released_item["item"].build_task_id, released_item["prev_num"])
            message_processor.add_message(released_item, nanny_req)
            if self.Parameters.wait_for_deploy and nanny_req:
                self.Context.nanny_requests[str(released_item["item"].resource.type)] = {
                    "release_req_id": nanny_req,
                    "state": "OPEN"
                }

        message_processor.process_messages()

        eh.ensure(not errors, "\n".join(errors))

        if not self.Parameters.do_release:
            self.set_info("Parameters.do_release is False, so no need to release")
            return

        if self._is_to_stable:
            self.check_unresolved_problems(release_num)
            self.cleanup_after_release(release_num, st_helper)
            if isinstance(self._c_info, rm_comp.mixin.Startreked):
                try:
                    self._c_info.st_post_release(self, self._token, release_num)
                except Exception as exc:
                    eh.log_exception("Cannot execute post-release handler", exc)
                    self.set_info("Cannot execute post release code ({}, see logs for details)".format(exc))

        try:
            self.Context.release_info = message_processor.messages
            self.Context.save()
        except Exception:
            logging.exception("Failed to store release_info in context")

        eh.ensure(
            released_items or released_tasks,
            "Nothing was released. Expected resources: {}".format(
                ", ".join([res_info.resource_name for res_info in self._c_info.releases_cfg__resources_info])
            )
        )

        self.send_rm_proto_event(status='SUCCESS')

        if self.Parameters.wait_for_deploy:
            eh.ensure(
                self.Context.nanny_requests,
                "No release requests to wait for, probably they have not been created. Released items: {}".format(
                    ", ".join([item["item"].name for item in released_items])
                )
            )
            self.Context.wait_for_deploy_mark = True
            self.Context.release_num = release_num
            self.set_info(
                "Waiting for deploy for {} min...".format(self._c_info.releases_cfg__wait_for_deploy_time_sec / 60)
            )
            raise sdk2.WaitTime(self._c_info.releases_cfg__wait_for_deploy_time_sec)

    def _filter_resources(self):
        for res_info in self._c_info.releases_cfg__resources_info:
            logging.info("Filter %s:", res_info.resource_name)
            service_ids = []
            if self.Parameters.urgent_release:
                service_ids = self._get_service_ids(res_info)
                if not service_ids:
                    logging.error("Deploy configuration doesn't exist or wrong")
                    continue
            resource_id = self.Parameters.component_resources.get(res_info.resource_name, None)
            if not resource_id:
                logging.info("No id specified for this resource")
                continue
            resource = sdk2.Resource[resource_id]
            if resource.type != res_info.resource_type:
                logging.error("Resource has type %s instead of %s", resource.type, res_info.resource_type)
            logging.info("Take it")
            release_item = ReleasedItem(
                res_info.resource_name,
                resource,
                build_ctx_key=res_info.build_ctx_key,
            )
            yield release_item, service_ids

    def _get_service_ids(self, res_info):
        deploy = getattr(res_info, "deploy", None)
        if not deploy:
            return
        for deploy_info in deploy:
            if not isinstance(deploy_info, rm_config.DeployServicesInfo):
                logging.debug("Deploy configuration has to be instance of DeployServicesInfo")
                return
            if deploy_info.level == self.Parameters.where_to_release:
                return rm_comp.ComponentInfoGeneral.get_deploy_services(deploy_info, self._nn_client)
        return

    @staticmethod
    def filter_sandbox_files(sandbox_files, release_items):
        for i, sandbox_file in enumerate(sandbox_files):
            for release_item in release_items:
                if sandbox_file["resource_type"] != release_item.resource.type:
                    continue
                if sandbox_file["resource_id"] == str(release_item.resource.id):
                    logging.info("Resource %s is already deployed", release_item.resource.id)
                    continue
                yield i, sandbox_file, release_item

    def _push_service_configuration(self, service_id, release_items, release_num):
        logging.info("Push: resources '%s' to '%s'", release_items, service_id)
        resources = self._nn_client.get_service_resources(service_id)
        sandbox_files = resources["content"]["sandbox_files"]
        comment = []
        for i, sandbox_file, release_item in self.filter_sandbox_files(sandbox_files, release_items):
            old_res = sdk2.Resource[sandbox_file["resource_id"]]
            diff = [old_res.id, old_res.description, release_item.resource.id, release_item.resource.description]
            self.set_info("Service {}:\nid: {}\n{}\n>>>\nid: {}\n{}".format(service_id, *diff), do_escape=False)
            if self.Parameters.do_release:
                build_task_info = self.server.task[release_item.build_task_id].read()
                sandbox_files[i].update({
                    "resource_id": str(release_item.resource.id),
                    "task_id": str(release_item.build_task_id),
                    "task_type": str(build_task_info["type"]),
                })
                comment.append(self._c_info.get_stable_release_label(release_item, release_num))
        if self.Parameters.do_release and comment:
            self._nn_client.update_service_resources(service_id, {
                "content": resources["content"],
                "comment": "{}\n(committed by RM directly)".format("\n".join(comment)),
                "snapshot_id": resources["snapshot_id"]
            })

    @sdk2.header()
    def header(self):
        if not self.Parameters.wait_for_deploy:
            return ""
        nanny_reqs = sorted(self.Context.nanny_requests)
        return [{
            "helperName": "",
            "content": {
                u"Nanny release requests": {
                    "header": [
                        {"key": "resource", "title": "Resource"},
                        {"key": "status", "title": "Status"},
                        {"key": "time", "title": "Time"},
                        {"key": "release", "title": "Release"}
                    ],
                    "body": {
                        "resource": [k for k in nanny_reqs],
                        "status": [self.Context.nanny_requests[k]["state"] for k in nanny_reqs],
                        "time": [self.Context.nanny_requests[k].get("end_time", "") for k in nanny_reqs],
                        "release": [lb.nanny_link(self.Context.nanny_requests[k]["release_req_id"]) for k in nanny_reqs]
                    }
                }
            }
        }]

    def cleanup_after_release(self, release_num, st_helper):
        if isinstance(self._c_info, rm_comp.Branched) and self._c_info.name != rm_const.RMNames.RELEASE_MACHINE_TEST:
            # SEARCH-4058
            stop_database_names = self._c_info.testenv_db_stop_names(release_num)
            if stop_database_names:
                success = TEClient.cleanup_db({'stop_database_names': stop_database_names})
                eh.ensure(success, "Cleanup databases failed")

        if isinstance(self._c_info, rm_comp.mixin.Startreked):
            st_helper.execute_transition(self._c_info, release_num, rm_const.Workflow.DEPLOYING, self)
            if self._c_info.notify_cfg__st__close_prev_tickets_stage == rm_const.PipelineStage.release:
                st_helper.close_prev_tickets(
                    self._c_info, release_num,
                    "[cleanup after release] Closing previous tickets after release of {} version".format(release_num)

                )
            if self._c_info.notify_cfg__st__close_linked_tickets:
                st_helper.close_linked_tickets(self._c_info, release_num)

    def check_unresolved_problems(self, release_num):
        if not isinstance(self._c_info, rm_comp.NoTrunkReleased):
            # check only for no-trunk components
            return
        try:
            te_db = self._c_info.testenv_cfg__db_template.format(testenv_db_num=release_num)
            te_problems = TEClient.get_te_problems(te_db, unresolved_only=True)
            logging.info("Unresolved problems: %s", te_problems)
            if te_problems.get("results"):
                nh.email2(
                    self,
                    recipients={p["owner"] for p in te_problems["rows"]} | {self._c_info.get_responsible_for_release()},
                    subject="Warning! Release {} is rolled with unresolved problems".format(
                        self._c_info.scope_path_short(release_num)
                    ),
                    body=[
                        "Unresolved revisions: {}".format(", ".join([str(p["revision"]) for p in te_problems["rows"]])),
                        lb.HREF_TO_ITEM.format(
                            link=rm_const.Urls.te_db_screen(te_db, "unresolved_problems"),
                            name="Problems page"
                        ),
                    ]
                )
                self.set_info(
                    "There are {} unresolved testenv problems for this release!".format(te_problems["results"])
                )
            if int(self.Context.max_minor_release_num) > 0:
                revs = rm_svn.SvnHelper.revisions_merged(self._c_info.full_branch_path(release_num))
                problems_by_tag = {}
                for p in te_problems["rows"]:
                    try:
                        problem_rev = int(p["revision"])
                        ind = revs.index(problem_rev)
                        problems_by_tag.setdefault(ind, []).append(problem_rev)
                    except ValueError:
                        logging.debug("Rev %s in not in merged to branch list", problem_rev)
                self.Context.problems_by_tag = problems_by_tag
            else:
                logging.debug("There is no minor releases for %s", self._c_info.name)

        except Exception as exc:
            eh.log_exception("Cannot get unresolved problems", exc)

    def wait_task_released(self, task_id):
        start_time = int(time.time())
        # wait for build task released status up to 7 minutes RMINCIDENTS-151
        while True:
            task_status = rm_utils.get_task_field(task_id, "status", self.server)
            logging.debug("Current task (%s) status is: %s", task_id, task_status)
            if task_status == ctt.Status.RELEASED:
                return True
            elif task_status == ctt.Status.NOT_RELEASED:
                return False
            if int(time.time()) - start_time > 420:
                return False
            time.sleep(15)

    def wait_resource_released(self, resource_id):
        start_time = int(time.time())
        while True:
            status = sdk2.Resource[resource_id].reload().released
            logging.debug("Current released status for resource %s: %s", resource_id, status)
            if status == self.Parameters.where_to_release:
                return True
            if int(time.time()) - start_time > 420:
                return False
            time.sleep(15)

    def _parse_additional_addresses(self):
        addr_list = (
            str(self.Parameters.additional_addresses)
            .replace(",", " ")
            .replace(";", " ")
            .replace(".", " ")
        ).split(" ")
        addr_list = [addr for addr in addr_list if addr]
        return addr_list

    def _get_comment(self, release_item, release_to, issue, release_num):
        if release_to != rm_const.ReleaseStatus.stable:
            return self._c_info.testing_release_notes(release_item, release_num, self.Parameters.where_to_release)
        return u"Please deploy: '{}'\n{}\nRelease author: {}\n{}".format(
            self._c_info.name,
            self.Parameters.release_notes,
            self.author,
            self._c_info.stable_release_notes(
                item=release_item,
                release_num=release_num,
                st_issue=issue,
                add_info=self.Parameters.release_notes_comment or u"",
                task_id=release_item.build_task_id,
                build_ctx_key=release_item.build_ctx_key,
            )
        )

    def get_release_input(self, release_to, release_item, release_label, release_num, st_issue, add_msg):
        # implicit assertion: st_issue is not None => isinstance(c_info, Startreked)
        if self.Parameters.additional_release_parameters:
            additional_release_parameters = self.Parameters.additional_release_parameters.copy()
        else:
            additional_release_parameters = {}
        if self._c_info.notify_cfg__use_startrek and st_issue:
            self._c_info.notify_cfg__st__add_nanny_reports(
                release_to, additional_release_parameters, st_issue.key
            )
        release_input = {
            "to": [],
            "params": additional_release_parameters,
            "task_id": release_item.build_task_id,
            "cc": self._get_addresses_cc(release_to),
            "message": add_msg + self._get_comment(release_item, release_to, st_issue, release_num),
            "type": release_to,
            "subject": self.Parameters.subject_prefix + " " + release_label,
        }
        logging.info("Release input:\n%s", release_input)
        return release_input

    def _get_addresses_cc(self, release_to):
        if release_to == rm_const.ReleaseStatus.stable:
            additional_addresses = self._parse_additional_addresses()

            addresses_cc = (
                [self._c_info.get_responsible_for_release()] +
                self._c_info.releases_cfg__release_followers_permanent +
                additional_addresses
            )
            if isinstance(self._c_info, rm_comp.mixin.Startreked):
                addresses_cc.append(self._c_info.st_assignee)
                addresses_cc.extend(self._c_info.get_followers(self._token))
                addresses_cc.append(self.author)
            addresses_cc = [i for i in set(addresses_cc) if i]
        else:
            addresses_cc = []
        return addresses_cc

    def get_nanny_release_reqs(self, build_task_id):
        logging.info("Try to get nanny release requests from task: %s", build_task_id)
        reqs = rm_utils.get_ctx_field(build_task_id, "nanny_release_requests", self.server, [])
        if reqs is None:
            reqs = []
        return reqs

    def wait_nanny_release_req(self, build_task_id, prev_requests_count):
        wait_time = 0
        while True:
            release_reqs = self.get_nanny_release_reqs(build_task_id)
            if prev_requests_count < len(release_reqs):
                return release_reqs[-1]
            if wait_time > 150:
                logging.info("TIMEOUT: Waiting for nanny_release_request_id timed out!")
                return
            wait_time += 30
            logging.info("Wait %s seconds for nanny_release_request_id appearing in build task", wait_time)
            time.sleep(wait_time)

    @decorators.retries(5, delay=1)
    def get_st_issue(self, st_helper, release_num):
        return st_helper.find_ticket_by_release_number(release_num, self._c_info, fail=False)

    @decorators.retries(3)
    def _create_web_release(self, release_input):
        try:
            self.server.release(release_input)
        except Exception as e:
            if e.message.endswith(" is already in progress"):  # RMINCIDENTS-162
                self.set_info("Release is already initialized by another task")
            else:
                raise

    def _check_release_numbers(self, release_item, release_num):
        major_num, minor_num = self._c_info.get_release_numbers(release_item)
        if not major_num:
            return Ok(release_num)
        if minor_num:
            minor_num = int(minor_num)
            self.Context.max_minor_release_num = max(minor_num, self.Context.max_minor_release_num)
            self.Context.save()
        major_num = int(major_num)
        if release_num == -1:
            release_num = major_num
            return Ok(release_num)
        error_condition = self._is_to_stable and isinstance(self._c_info, rm_comp.NoTrunkReleased)
        error_condition &= major_num != release_num and major_num != release_item.build_task_id
        if not error_condition:
            return Ok(release_num)
        error_message = BLOCK_OLD_RELEASE_TEMPLATE.format(
            release_num=release_num,
            release_item_name=release_item.name,
            component_name=self.Parameters.component_name,
            where_to_release=self.Parameters.where_to_release,
            major_num=major_num,
        )
        self.set_info(error_message)
        return Error(error_message)

    def _release_task(self, release_item, release_num, st_issue, add_msg=u""):
        released = sdk2.Resource[release_item.resource.id].reload().released == self.Parameters.where_to_release
        if (
            not released
            and rm_utils.get_task_field(release_item.build_task_id, "status", self.server) == ctt.Status.RELEASING
        ):
            released = self.wait_task_released(release_item.build_task_id)
            released = released and self.wait_resource_released(release_item.resource.id)
        if released:
            logging.info("Resource '%s' is already released by another task!", release_item.name)
            reqs = self.get_nanny_release_reqs(release_item.build_task_id) if self.Parameters.wait_for_deploy else None
            if self._is_to_stable:
                return Ok({"release_req_id": reqs[-1]} if reqs else {})
        prev_requests_count = len(self.get_nanny_release_reqs(release_item.build_task_id))
        release_label = self._c_info.get_stable_release_label(release_item, release_num)
        release_input = self.get_release_input(
            self.Parameters.where_to_release, release_item, release_label, release_num, st_issue, add_msg
        )
        try:
            self._create_web_release(release_input)
            return Ok({"prev_req_count": prev_requests_count, "release_label": release_label})
        except cr.Client.HTTPError as exc:
            self._log_http_error(exc.response.text.encode("utf8"))
            eh.log_exception("Couldn't create release", exc)
        except Exception as exc:
            self._log_unknown_exception(exc)

        return Error("Couldn't create release for '{}'".format(release_item.name))

    def _log_http_error(self, json_string):
        try:
            error_dict = json.loads(json_string)
        except json.JSONDecodeError:
            self._log_unknown_exception(json_string)
            return
        if REASON_KEY in error_dict:
            exception_reason = "Can't create release. Reason: {}".format(error_dict[REASON_KEY])
            logging.exception(exception_reason)
            self.set_info(exception_reason)
        else:
            self.set_info('No reason in error message')
            self._log_unknown_exception(error_dict)

    def _log_unknown_exception(self, additional_info=''):
        if additional_info:
            exception_string = "Can't create release. Additional info is {}".format(additional_info)
            logging.exception(exception_string)
            self.set_info(exception_string)
        self.set_info(
            "Sorry, release was not created."
            " Check permission to release all resources, and if it doesn't help ask"
            " <a href={}>Release Machine chat</a>".format(rm_const.Urls.RELEASE_MACHINE_INVITE_LINK),
            do_escape=False,
        )

    def wait_after_release(self, item):
        if not self.wait_task_released(item.build_task_id):
            return Error("{}: timed out waiting released status for task {}.".format(item.name, item.build_task_id))
        if not self.wait_resource_released(item.resource.id):
            return Error("{}: timed out waiting released status for resource {}.".format(item.name, item.resource.id))
        return Ok()

    def _check_release(self, st_helper):
        """Sometimes we need to check if release fits some conditions"""
        if not self.Parameters.release_number:
            return u""

        check_result = self._check_prev_releases(self.Parameters.where_to_release)
        logging.info("Check prev release: %s", check_result)
        if not check_result.ok:
            if (
                self.Parameters.urgent_release or (self._c_info.releases_cfg__allow_old_releases and (
                    self._c_info.releases_cfg__allow_old_releases is True or
                    self._c_info.releases_cfg__allow_old_releases.get(self.Parameters.where_to_release)
                ))
            ):
                return u"!!ATTENTION!! {}\n".format(check_result.result)
            st_helper.comment(
                self.Parameters.release_number,
                "Component '{}' number {} has not been deployed to {} via this task: {}.\n{}".format(
                    self._c_info.name, self.Parameters.release_number,
                    self.Parameters.where_to_release, lb.task_wiki_link(self.id), check_result.result,
                ),
                self._c_info
            )
            self.Context.fail_msg = check_result.result
            eh.check_failed(self.Context.fail_msg)
        return u""

    def _check_prev_releases(self, release_stage):
        """No need to release obsolete component"""
        prev_releases_tasks = self.server.task.read(
            type=self.type,
            fields="input_parameters.release_number,input_parameters.minor_release_num",
            input_parameters=json.dumps({
                "component_name": self.Parameters.component_name,
                "where_to_release": release_stage,
            }),
            hidden=True,
            limit=20,
        )["items"]
        logging.info("Got previous tasks of this component releases: %s", prev_releases_tasks)
        for prev_releases_task in prev_releases_tasks:
            if self.is_newer(prev_releases_task):
                return rm_core.Error("Newer build {} had been already released to {} by task #{}.".format(
                    prev_releases_task["input_parameters.release_number"],
                    release_stage,
                    prev_releases_task["id"],
                ))
        return rm_core.Ok()

    def is_newer(self, prev_releases_task):
        if prev_releases_task.get("input_parameters.release_number", 0) > self.Parameters.release_number:
            return True
        if prev_releases_task.get("input_parameters.release_number", 0) < self.Parameters.release_number:
            return False
        if prev_releases_task.get("input_parameters.minor_release_num", 0) > self.Parameters.minor_release_num:
            return True
        return False

    def check_input_tasks(self):
        task_ids = rm_utils.get_tasks_to_check(self)
        if task_ids:
            utils2.check_tasks_to_finish_correctly(self, task_ids)
            for t_id in task_ids:
                if rm_utils.get_ctx_field(t_id, rm_const.ACCEPTANCE_FAIL_KEY, self.server, False):
                    self.Context.fail_msg = 'Task with id : {} has acceptance fail flag'.format(t_id)
                    eh.check_failed(self.Context.fail_msg)

    def _get_deploy_info(self, nanny_release_request_id):
        if nanny_release_request_id:
            ticket_url = "{}v1/requests/{}/".format(
                nanny_const.NANNY_API_URL,
                nanny_release_request_id
            )
            try:
                ticket_info = requests_wrapper.get(
                    ticket_url,
                    headers={"Authorization": "OAuth {}".format(self._token)},
                    verify=False,
                )
                logging.debug(
                    "Ticket info: %s\nText: %s\nHeaders: %s",
                    ticket_info, ticket_info.text, ticket_info.request.headers
                )
                ticket_info.raise_for_status()
                return ticket_info.json()
            except Exception:
                logging.info(
                    "Skip getting deploy status from url: %s\nError:\n%s",
                    ticket_url, eh.shifted_traceback()
                )

        logging.info("No nanny release ticket in Context")
        return {}

    def wait_time_for_deploy(self, st_helper, release_num):
        self.Context.wait_num += 1
        resources_to_wait = {}
        for res_type, release_req in self.Context.nanny_requests.iteritems():
            if release_req["state"] in self.END_RELEASE_STATUSES:
                logging.debug("Nothing to do: %s has %s state", res_type, release_req["state"])
                continue
            deploy_info = self._get_deploy_info(release_req["release_req_id"])
            state = deploy_info.get("status")
            self.Context.nanny_requests[res_type]["state"] = state
            if state in self.END_RELEASE_STATUSES:
                end_time = deploy_info.get("end_time", {}).get("$date")
                if end_time:
                    end_time = datetime.datetime.utcfromtimestamp(end_time / 1000).replace(
                        tzinfo=gettz('Europe/Moscow')
                    ).isoformat()
                    self.Context.nanny_requests[res_type]["end_time"] = end_time
                msg = "Deploy of {} is finished with status = {}. End time = {} MSK".format(res_type, state, end_time)
                self.set_info(msg)
                st_helper.comment(release_num, msg, self._c_info)
            elif state:
                resources_to_wait[res_type] = state
                logging.info("Deploy status for %s: %s", res_type, state)
            else:
                logging.info("Cannot get deploy status")

        if resources_to_wait:
            if self.Context.wait_num < self.MAX_WAIT_DEPLOY_NUM:
                logging.info("Wait for resources: %s", resources_to_wait)
                raise sdk2.WaitTime(self._c_info.releases_cfg__wait_for_deploy_time_sec)
            msg = "Timed out waiting for deploy of {res} resources. Task {task}".format(
                res=', '.join(resources_to_wait.keys()),
                task=self.id,
            )
            self.set_info(msg)
            st_helper.comment(release_num, msg, self._c_info)
            self.Context.tm_message = msg
        else:
            self.Context.tm_message = u"Release items [{}] of component '{}' were deployed to {} by task: {}".format(
                [lb.resource_link(i, plain=True) for i in self.Parameters.component_resources.values()],
                self.Parameters.component_name,
                self.Parameters.where_to_release,
                lb.task_link(self.id)
            )

    def _get_rm_proto_event_hash_items(self, event_time_utc_iso, status=None):
        return (
            self.Parameters.component_name,
            'ReleaseCreated',
            self.Parameters.release_number,
            self.Parameters.where_to_release,
            (self.Parameters.minor_release_num or self.Context.max_minor_release_num),
            "TASK_{}".format(status or ""),
            ",".join(str(value) for value in self.Parameters.component_resources.itervalues()),
        )

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):
        """Returns dict <spec_data_key> => <event specific data> for event specific data oneof field"""

        if status == ctt.Status.SUCCESS:

            result = {
                'release_created_data': rm_proto_events.ReleaseCreatedData(
                    job_name=self.get_job_name_from_gsid(),
                    scope_number=str(self.Parameters.release_number),
                    tag_number=str(self.Parameters.minor_release_num or self.Context.max_minor_release_num),
                    release_type=self.Parameters.where_to_release,
                    release_info=[
                        rm_proto_events.ReleasedResourceInfo(
                            resource_id=str(release_item.get('resource_id', '')),
                            resource_type=release_item.get('resource_type', ''),
                            resource_name=release_item.get('resource_type', ''),
                            nanny_ticket_key=release_item.get('nanny_ticket', ''),
                            release_label=release_item.get('release_label', ''),
                            build_task_id=str(release_item.get('build_task', '')),
                        ) for release_item in self.Context.release_info
                    ],
                    component_resources=[
                        rm_proto_events.ComponentResourceInfo(
                            key=resource_key,
                            resource_id=self.Parameters.component_resources[resource_key],
                        ) for resource_key in self.Parameters.component_resources
                    ],
                    author=self.author,
                )
            }

        elif status in (ctt.Status.FAILURE, ctt.Status.EXCEPTION, ctt.Status.TIMEOUT):

            result = {
                'release_failed_data': rm_proto_events.ReleaseFailedData(
                    job_name=self.get_job_name_from_gsid(),
                    scope_number=str(self.Parameters.release_number),
                    tag_number=str(self.Parameters.minor_release_num or self.Context.max_minor_release_num),
                    release_type=self.Parameters.where_to_release,
                    release_info=[
                        rm_proto_events.ReleasedResourceInfo(
                            resource_id=str(release_item.get('resource_id', '')),
                            resource_type=release_item.get('resource_type', ''),
                            resource_name=release_item.get('resource_type', ''),
                            nanny_ticket_key=release_item.get('nanny_ticket', ''),
                            release_label=release_item.get('release_label', ''),
                            build_task_id=str(release_item.get('build_task', '')),
                        ) for release_item in self.Context.release_info
                    ],
                    component_resources=[
                        rm_proto_events.ComponentResourceInfo(
                            key=resource_key,
                            resource_id=self.Parameters.component_resources[resource_key],
                        ) for resource_key in self.Parameters.component_resources
                    ],
                )
            }

        else:

            result = None

        return result


class MessageProcessor(object):
    messages = []

    def __init__(self, task, st_helper, st_issue=None):
        self.task = task
        self.c_info = task._c_info
        self.st_issue = st_issue
        self.st_helper = st_helper

    def add_message(self, released_item, nanny_req):
        self.messages.append({
            "build_task": released_item["item"].build_task_id,
            "resource_id": released_item["item"].resource.id,
            "resource_type": str(released_item["item"].resource.type),
            "release_label": released_item["label"],
            "nanny_ticket": nanny_req,
        })

    def process_messages(self):
        if not self.messages:
            logging.info("Nothing to send")
            return
        base_comment = []
        st_message = [self._get_comment_head(lb.task_wiki_link)]
        tm_message = [self._get_comment_head(lb.task_link)]
        for release_info in self.messages:
            base_comment.append(u"=====Релиз {} отправлен на выкатку в {} ({}, {})".format(
                release_info["release_label"],
                self.task.Parameters.where_to_release,
                tu.date_ymdhm(),
                lb.nanny_wiki_link(release_info["nanny_ticket"]) if release_info["nanny_ticket"] else "No tickets",
            ))
            st_message.append(self._get_comment_links(release_info, lb.sb_item_wiki_link, lb.nanny_wiki_link))
            tm_message.append(self._get_comment_links(release_info, lb.sb_item_link, lb.nanny_link))
        self.task.Context.tm_message = "\n".join(tm_message)
        try:
            self.task.send_tm_notify()
        except Exception as exc:
            eh.log_exception("PLS FIX TM notifications", exc)
        if not self.c_info.notify_cfg__use_startrek:
            logging.info("%s does not use Startrek notifications", self.c_info.name)
            return
        if self.st_issue is None:
            logging.info("%s does not have Startrek issue to comment", self.c_info.name)
            return
        if not self.c_info.notify_cfg__st__notify_on_release_to_release_st_ticket:
            logging.info("%s does not allow to post release notifications to release tickets", self.c_info.name)
            return
        self.st_issue.comments.create(text="\n".join(st_message))
        self._update_description(base_comment)

    def _get_comment_head(self, link_builder):
        return "Релиз отправлен на выкатку в {}\nBy request of {}\nRelease task: {}".format(
            self.task.Parameters.where_to_release,
            self.task.author,
            link_builder(self.task.id)
        )

    @staticmethod
    def _get_comment_links(release_info, sb_link_builder, nanny_link_builder):
        return "{}: Build task: {} / Resource [{}]: {} / Release request: {}".format(
            release_info["release_label"],
            sb_link_builder(release_info["build_task"], "task"),
            release_info["resource_type"],
            sb_link_builder(release_info["resource_id"], "resource"),
            nanny_link_builder(release_info["nanny_ticket"]) if release_info["nanny_ticket"] else "No tickets",
        )

    @decorators.retries(5, delay=2)
    def _update_description(self, base_comment):
        updated_issue = self.st_helper.get_ticket_by_key(self.st_issue.key)
        updated_issue.update(description="{}\n{}".format(
            string.all_to_str(updated_issue.description),
            string.all_to_str("\n".join(base_comment)),
        ))
