# -*- coding: utf-8 -*-
import contextlib
import json
import logging
import time

from sandbox import sdk2
from sandbox.common import rest
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt

from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import utils2
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine import rm_utils
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine import core as rm_core
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers import deploy as rm_deploy
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.projects.release_machine.rm_notify as rm_notify


@rm_notify.notify2()
class ReleaseRmComponent2(rm_bt.BaseReleaseMachineTask):
    """One task for all releases."""

    _c_info = None
    _deploy_system = None

    class Requirements(task_env.TinyRequirements):
        disk_space = 2 * 1024  # 2 Gb

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        deploy_results = []
        release_errors = []

    class Parameters(rm_params.DefaultReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        major_release_num = sdk2.parameters.Integer("Major release number")
        minor_release_num = sdk2.parameters.Integer("Minor release number")
        wait_params_block = sdk2.parameters.Info("Wait params")
        wait_for_deploy = sdk2.parameters.Bool("Wait for deploy", default_value=False)
        with wait_for_deploy.value[True]:
            wait_for_deploy_attempts = sdk2.parameters.Integer(
                "Wait for deploy attempts", default_value=6, required=True,
            )
            wait_for_deploy_time_sec = sdk2.parameters.Integer(
                "Wait for deploy time (seconds)", default_value=600, required=True,
            )

        deploy_params_block = sdk2.parameters.Info("Deploy specific params")
        with sdk2.parameters.String("Release via") as deploy_system:
            rm_params.set_choices(deploy_system, [i.name for i in rm_const.DeploySystem])

            with deploy_system.value["ya_tool"]:
                build_task_id = sdk2.parameters.Integer("Build task id to release")
                change_json_path = sdk2.parameters.String("Path in json to change (Ex: toolchain.horadric)")
            with deploy_system.value["ab"]:
                data_for_testid = sdk2.parameters.String("Data for testid", multiline=True)
            with deploy_system.value["qloud"]:
                registry_name = sdk2.parameters.String("Registry name")
                qloud_environment = sdk2.parameters.String(
                    "Path to qloud component (project.application.environment.component)"
                )
                qloud_vault_name = sdk2.parameters.String("Vault name of qloud oauth token")
                qloud_vault_owner = sdk2.parameters.String("Owner name of qloud oauth token")
                qloud_environment_dump_arcadia_path = sdk2.parameters.String(
                    "Arcadia path to custom environment json dump (e.g. 'service/deployments/qloud/env-dump.json')."
                    " If set, this json dump will be used instead of live dump of existing qloud environment"
                )
                qloud_environment_dump_svn_revision = sdk2.parameters.String("SVN revision of arcadia to checkout")
                qloud_custom_environment_name = sdk2.parameters.String(
                    "Custom environment name, to set in destination environment id before uploading"
                    " (e.g. if set will replace 'environment' part in 'project.application.environment.component'."
                    " Useful in PRs to create new env as a copy of existing env or from json dump in arcadia"
                )
            with contextlib.nested(
                deploy_system.value["nanny"],
                deploy_system.value["nanny_push"],
                deploy_system.value["ya_deploy"],
                deploy_system.value["sandbox"],
            ):
                component_resources = sdk2.parameters.Dict("Resources dict (ex: {'my_resource_name': 12345678})")
            with deploy_system.value["samogon"]:
                component_resource = sdk2.parameters.Resource("Resource to release")
                project = sdk2.parameters.String("Project to deploy", required=True)
                namespace = sdk2.parameters.Integer("Namespace to deploy", required=True, default_value=None)
                clusterapi_url = sdk2.parameters.String("Url", required=True)
                samogon_oauth_token_vault_name = sdk2.parameters.String("Samogon OAuth token Vault name", required=True)

        with sdk2.parameters.String("Release to", default=rm_const.ReleaseStatus.testing) as where_to_release:
            rm_params.set_choices(where_to_release, [x for x in dir(rm_const.ReleaseStatus) if not x.startswith("__")])

        additional_params_block = sdk2.parameters.Info("Additional params")
        check_task_ids = sdk2.parameters.List("Task ids to check before release", value_type=sdk2.parameters.Integer)
        additional_release_notes = sdk2.parameters.String("Additional release notes", default_value="")
        additional_release_followers = sdk2.parameters.List("Additional release followers")
        additional_release_parameters = sdk2.parameters.Dict("Additional release parameters")

        skip_on_empty_arcadia_patch = sdk2.parameters.Bool(
            "Skip execution if not a patched run."
            " Useful for skipping run in a patch-less run of pre-commit check in testenv",
            default_value=False
        )

        release_item_name = sdk2.parameters.String("Release Machine release item name")

    @property
    def c_info(self):
        if not self._c_info:
            self._c_info = rmc.COMPONENTS[self.Parameters.component_name]()
        return self._c_info

    @property
    def deploy_system(self):
        if not self._deploy_system:
            self._deploy_system = rm_deploy.DEPLOY_SYSTEM[self.Parameters.deploy_system](task=self, c_info=self.c_info)
        return self._deploy_system

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        if (
            self.Parameters.skip_on_empty_arcadia_patch and
            (
                self.Context.arcadia_patch is ctm.NotExists or
                not self.Context.arcadia_patch
            )
        ):
            self.set_info(
                "Task execution skipped: "
                "'skip_on_empty_arcadia_patch' parameter is True,"
                " but 'arcadia_patch' context field is missing or empty"
                " (task must be run in a patched run)"
            )
            return

        self._rm_token = rm_sec.get_rm_token(self)
        self._st_helper = STHelper(self._rm_token)

        deploy_system = self.deploy_system
        with self.memoize_stage.release(commit_on_entrance=False, commit_on_wait=False):
            check_result = self.check_release_is_allowed(deploy_system)
            if not check_result.ok:
                if check_result.result["fail_task"]:
                    eh.check_failed(check_result.result["msg"])
                else:
                    self.set_info("No release from this task: {}".format(check_result.result["msg"]))
                    return
            deploy_system.before_release()
            release_results = deploy_system.do_release()
            self.after_release(deploy_system, release_results)
        if self.Parameters.wait_for_deploy:
            self.wait_for_deploy_stage(deploy_system)
        self.cleanup_after_all(deploy_system)

    def wait_for_deploy_stage(self, deploy_system):
        with self.memoize_stage.save_data_for_deploy_results:
            self.Context.deploy_results = deploy_system.get_data_for_deploy_results()
        new_deploy_results = deploy_system.update_deploy_results(self.Context.deploy_results)
        not_ok_deploy_results = [i for i in new_deploy_results if not i.ok]
        ok_deploy_results = [i for i in new_deploy_results if i.ok]
        if ok_deploy_results:
            msg = "Got new successful deploy (to {}) results:\n{}".format(
                self.Parameters.where_to_release, "\n".join((str(i.result) for i in ok_deploy_results))
            )
            logging.info(msg)
            self.notify_st(msg)
        with self.memoize_stage.waiting(self.Parameters.wait_for_deploy_attempts):
            if not_ok_deploy_results:
                self.set_info(
                    "Wait ({} sec) for deploy results:\n{}".format(
                        self.Parameters.wait_for_deploy_time_sec, "\n".join(map(str, not_ok_deploy_results)),
                    ),
                )
                raise sdk2.WaitTime(self.Parameters.wait_for_deploy_time_sec)
        self.after_deploy(deploy_system, not_ok_deploy_results)

    def check_release_is_allowed(self, deploy_system):
        for check_result in self._lazy_checkers(deploy_system):
            if not check_result.ok:
                if self.Parameters.major_release_num:
                    self.notify_st(
                        "Component '{}' number {} has not been deployed to {} via this task: {}.\n{}".format(
                            self.c_info.name, self.Parameters.major_release_num,
                            self.Parameters.where_to_release, lb.task_wiki_link(self.id), check_result.result,
                        )
                    )
                return check_result
        return rm_core.Ok()

    def _lazy_checkers(self, deploy_system):
        """
        Iterate over all possible checkers.

        Each checker should return Ok() or Error({"msg": str, "fail_task": bool})
        """
        yield self._check_input_tasks()
        yield self._check_robot_allowance()
        yield self._check_old_releases()
        yield self.c_info.check_release_blocked(
            self.Parameters.where_to_release,
            self.Parameters.major_release_num,
            self.Parameters.minor_release_num,
        )

    def _check_input_tasks(self):
        if self.Parameters.check_task_ids:
            task_ids = list(self.Parameters.check_task_ids)
            utils2.check_tasks_to_finish_correctly(self, task_ids)
            for t_id in task_ids:
                if rm_utils.get_ctx_field(t_id, rm_const.ACCEPTANCE_FAIL_KEY, self.server, False):
                    return rm_core.Error({
                        "msg": "Task with id `{}` has acceptance fail flag".format(t_id),
                        "fail_task": True,
                    })
        return rm_core.Ok()

    def _check_robot_allowance(self):
        # Todo: generalize it to all deploy systems and all `where_to_release` options
        if (
            self.Parameters.deploy_system == "nanny" and
            self.Parameters.where_to_release == rm_const.ReleaseStatus.stable and
            not self.c_info.releases_cfg__allow_robots_to_release_stable and
            not utils2.is_human(self.author)
        ):
            return rm_core.Error({
                "msg": (
                    "Robots aren't allowed to make stable releases for the component: {}. "
                    "Add `YourComponentInfo.Releases.allow_robots_to_release_stable = True` "
                    "to your RM configuration "
                    "if you want to allow robots to perform automatic releases ( one can find examples "
                    "in https://nda.ya.ru/3UZoNM )".format(self.c_info.name),
                ),
                "fail_task": True,
            })
        return rm_core.Ok()

    def _check_old_releases(self):
        """Check if it is allowed to release obsolete component."""
        if (
            self._c_info.releases_cfg__allow_old_releases and (
                self._c_info.releases_cfg__allow_old_releases is True or
                self._c_info.releases_cfg__allow_old_releases.get(self.Parameters.where_to_release)
            )
        ):
            return rm_core.Ok(
                "Config allows old releases for {}, so skip check.".format(self.Parameters.where_to_release)
            )
        prev_releases_tasks = self.server.task.read(
            type=self.type,
            fields="input_parameters.major_release_num,input_parameters.minor_release_num",
            input_parameters=json.dumps({
                "component_name": self.Parameters.component_name,
                "where_to_release": self.Parameters.where_to_release,
            }),
            hidden=True,
            limit=20,
        )["items"]
        logging.info("Got previous tasks of this component releases: %s", prev_releases_tasks)
        for prev_releases_task in prev_releases_tasks:
            if self._is_newer(prev_releases_task):
                return rm_core.Error({
                    "msg": "Newer build had been already released to {} in task: {}.".format(
                        prev_releases_task["input_parameters.major_release_num"], prev_releases_task["id"]
                    ),
                    "fail_task": self.Parameters.where_to_release == rm_const.ReleaseStatus.stable,
                })
        return rm_core.Ok()

    def _is_newer(self, prev_releases_task):
        if prev_releases_task.get("input_parameters.major_release_num", 0) > self.Parameters.major_release_num:
            return True
        if prev_releases_task.get("input_parameters.major_release_num", 0) < self.Parameters.major_release_num:
            return False
        if prev_releases_task.get("input_parameters.minor_release_num", 0) > self.Parameters.minor_release_num:
            return True
        return False

    def after_release(self, deploy_system, release_results):
        """
        :type deploy_system: sandbox.projects.release_machine.helpers.deploy.basic_releaser.BasicReleaser
        :type release_results: List[sandbox.projects.release_machine.core.Result]
        """
        self.set_info("Release results:\n{}".format("\n".join(map(str, release_results))), do_escape=False)
        try:
            deploy_system.after_release(release_results)
        except Exception as e:
            self.set_info("Problem in deploy_system.after_release. Look at common.log for details")
            eh.log_exception("Problem in deploy_system.after_release", e, task=self)
        try:
            self.c_info.after_release(
                release_results, self.Parameters.where_to_release,
                self.Parameters.major_release_num, self.Parameters.minor_release_num,
                self._st_helper, self.get_st_issue(), self,
            )
        except Exception as e:
            self.set_info("Problem in c_info.after_release. Look at common.log for details")
            eh.log_exception("Problem in c_info.after_release", e, task=self)

        not_ok_release_results = [i for i in release_results if not i.ok]

        self.Context.release_errors = [str(item) for item in not_ok_release_results]
        self.Context.save()

        self.send_rm_proto_event(
            status=('FAILURE' if not_ok_release_results else 'SUCCESS'),
        )
        if not_ok_release_results:
            self.cleanup_after_all(deploy_system)
            eh.check_failed(
                "Release failed for some reasons:\n{}".format("\n".join(map(str, not_ok_release_results)))
            )
        self._start_infra_event()

    def after_deploy(self, deploy_system, not_ok_deploy_results):
        self.set_info("Deploy results:\n{}".format("\n".join(map(str, self.Context.deploy_results))), do_escape=False)
        try:
            # self.c_info.after_deploy(self.Context.deploy_results)
            deploy_system.after_deploy(self.Context.deploy_results)
        except Exception as e:
            self.set_info("Problem in after_deploy. Look at common.log for details")
            eh.log_exception("Problem in after_deploy", e, task=self)
        if not_ok_deploy_results:
            msg = "Failed to wait for deploy '{}' to '{}' in task {}\nNot ok results:\n{}".format(
                self.Parameters.component_name,
                self.Parameters.where_to_release,
                self.id,
                "\n".join(map(str, not_ok_deploy_results))
            )
            self.notify_st(msg)
            self.cleanup_after_all(deploy_system)
            eh.check_failed(msg)

    def cleanup_after_all(self, deploy_system):
        self.c_info.cleanup_after_release(
            self.Parameters.where_to_release,
            self.Parameters.major_release_num,
            self.Parameters.minor_release_num,
        )
        deploy_system.cleanup_after_all()

    def notify_st(self, msg):
        issue = self.get_st_issue()
        if issue:
            self._st_helper.create_comment(
                issue, msg,
                notify=self.c_info.notify_cfg__st__notify_on_robot_comments_to_tickets
            )

    def _start_infra_event(self):
        try:
            if not all([
                self._c_info.releases_cfg__infra_service_id,
                self._c_info.releases_cfg__infra_envs.get(self.Parameters.where_to_release),
                self._c_info.releases_cfg__infra_event_duration_sec,
            ]):
                logging.info("Skip starting infra event, some variables are not defined")
                return
            infra_client = rest.Client(rm_const.Urls.INFRA_API, self._rm_token)
            event_start = int(time.time())
            event_end = event_start + self._c_info.releases_cfg__infra_event_duration_sec
            st_issue_key = self.get_st_issue_key()
            infra_event_info = infra_client.events.create(
                title="New release {}. Author: {}".format(
                    self._c_info.release_id(self.Parameters.major_release_num, self.Parameters.minor_release_num),
                    self.author,
                ),
                description="Release event",
                environmentId=self._c_info.releases_cfg__infra_envs.get(self.Parameters.where_to_release),
                serviceId=self._c_info.releases_cfg__infra_service_id,
                startTime=event_start,
                finishTime=event_end,
                type="maintenance",
                severity="minor",
                tickets=st_issue_key if st_issue_key else None,
                sendEmailNotifications=False,
                setAllAvailableDc=True,
            )
            self.Context.infra_event_id = infra_event_info.get("id")
        except Exception as e:
            eh.log_exception("Unable to start infra event", e)

    def get_st_issue_key(self):
        """This method allows to reduce number of requests to ST if we need only issue key"""
        if self.Context.st_issue_key is ctm.NotExists:
            issue = self._st_helper.find_ticket_by_release_number(self.Parameters.major_release_num, self.c_info)
            self.Context.st_issue_key = issue.key if issue else None
        return self.Context.st_issue_key

    def get_st_issue(self):
        """Each time try to get updated issue to avoid modification conflicts."""

        if self.is_ci_launch and self.ci_context.config_info.id != self.c_info.jg__main_release_action_name:
            # RMDEV-3275
            return None

        if self.Context.st_issue_key is ctm.NotExists:
            issue = self._st_helper.find_ticket_by_release_number(self.Parameters.major_release_num, self.c_info)
            self.Context.st_issue_key = issue.key if issue else None
        elif self.Context.st_issue_key:
            logging.info("Got issue key: %s, try to get updated issue instance", self.Context.st_issue_key)
            issue = self._st_helper.get_ticket_by_key(self.Context.st_issue_key)
        else:  # issue was not found by release number before
            logging.info("Issue key was not found by major release number = %s", self.Parameters.major_release_num)
            issue = None
        return issue

    def _get_rm_proto_event_hash_items(self, event_time_utc_iso, status=None):
        return (
            self.Parameters.component_name,
            'ReleaseCreated',
            str(self.Parameters.major_release_num),
            self.Parameters.where_to_release,
            str(self.Parameters.minor_release_num or self.Parameters.major_release_num),
            status or self.status,
            ",".join([
                self.Parameters.component_resources[resource_key]
                for resource_key in self.Parameters.component_resources
            ]),
        )

    @decorators.memoized_property
    def _tag_number(self):
        if (
            self.c_info.release_cycle_type == rm_const.ReleaseCycleType.BRANCH or
            self.c_info.release_cycle_type == rm_const.ReleaseCycleType.CI
        ):
            return self.Parameters.minor_release_num
        else:
            return self.Parameters.major_release_num

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):
        if status == ctt.Status.SUCCESS:
            result = {
                'release_created_data': rm_proto_events.ReleaseCreatedData(
                    job_name=self.get_job_name_from_gsid(),
                    scope_number=str(self.Parameters.major_release_num),
                    tag_number=str(self._tag_number),
                    release_type=self.Parameters.where_to_release,
                    release_info=self.deploy_system.release_info,
                    component_resources=[
                        rm_proto_events.ComponentResourceInfo(
                            key=resource_key,
                            resource_id=self.Parameters.component_resources[resource_key],
                        ) for resource_key in self.Parameters.component_resources
                    ],
                    author=self.author,
                )
            }
        elif status in (ctt.Status.FAILURE, ctt.Status.EXCEPTION, ctt.Status.TIMEOUT):
            result = {
                'release_failed_data': rm_proto_events.ReleaseFailedData(
                    job_name=self.get_job_name_from_gsid(),
                    scope_number=str(self.Parameters.major_release_num),
                    tag_number=str(self._tag_number),
                    release_type=self.Parameters.where_to_release,
                    release_info=self.deploy_system.release_info,
                    component_resources=[
                        rm_proto_events.ComponentResourceInfo(
                            key=resource_key,
                            resource_id=self.Parameters.component_resources[resource_key],
                        ) for resource_key in self.Parameters.component_resources
                    ],
                    release_errors=self.Context.release_errors,
                )
            }
        else:
            result = None

        return result
