# -*- coding: utf-8 -*-
import json
import logging

import sandbox.common.errors as c_err
import sandbox.projects.common.binary_task as binary_task
import sandbox.projects.common.dynamic_models.const as models_const
import sandbox.projects.release_machine.helpers.ab_helper as rm_ab
import sandbox.projects.release_machine.helpers.startrek_helper as rm_st
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.security as rm_sec
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2


@rm_notify.notify2()
class ReleaseRmComponentAb(rm_bt.BaseReleaseMachineTask):
    """ For releases using AB (ex. flags.json)"""
    class Requirements(task_env.StartrekRequirements):
        disk_space = 1 * 1024  # 1 Gb

    class Parameters(rm_params.ComponentName2):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 15 * 60  # 15 min
        release_number = rm_params.release_number()
        data_for_testid = sdk2.parameters.String("Data for test id", multiline=True)
        wait_attempts = sdk2.parameters.Integer("Wait for exp formulas release (attempts)", default_value=2)
        wait_time_hours = sdk2.parameters.Integer("Wait hours per attempt", default_value=4)

    class Context(sdk2.Task.Context):
        waited_times = 0

    def on_execute(self):
        super(ReleaseRmComponentAb, self).on_execute()
        # self.check_exp_in_prod()
        ab_api = rm_ab.ABApi(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, "ab_formulas_release_token"), test=False)
        data = json.loads(self.Parameters.data_for_testid)
        ab_api.deploy_test_id(data)
        st_helper = rm_st.STHelper(rm_sec.get_rm_token(self))
        st_helper.comment(
            self.Parameters.release_number,
            "Test id '{}' was deployed to production".format(self.Parameters.data_for_testid),
            rmc.COMPONENTS[self.Parameters.component_name](),
        )

    def check_exp_in_prod(self):
        for formula_type in ["base", "mmeta"]:
            exp_folder = models_const.MODELS_EXPERIMENT_TAG_DIR_URL + formula_type
            last_exp_release_num = rm_svn.SvnHelper.get_highest_folder(exp_folder, only_num=True)
            c_info = rmc.COMPONENTS["exp_formulas_{}".format(formula_type)]()
            release_data = c_info.get_last_deploy()[0]
            logging.info("Release data: %s", release_data.to_json())
            if release_data.major_release < last_exp_release_num:
                logging.info(
                    "Last existing exp tag '%s' is not in prod yet. Last prod release is %s",
                    last_exp_release_num, release_data.major_release
                )
                if self.Context.waited_times < self.Parameters.wait_attempts:
                    raise c_err.WaitTime(self.Parameters.wait_time_hours * 60 * 60)
                else:
                    raise c_err.TaskError("Waiting for exp formulas attempts number exceeded")
