# -*- coding: utf-8 -*-
import json

import sandbox.common.types.task as ctt
import sandbox.projects.common.resource_selectors as rs
import sandbox.projects.common.link_builder as lb
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.tasks.ReleaseRmComponent2 as rm_release_task
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine.components.configs import release_machine_test as test_cfg

ARCHIVE_RESOURCE_KEY = u"tasks_archive_resource"
TAGS_KEY = u"tags"
CHECK_FAILED_TEMPLATE = u"Something really wrong with {} - there are no {}TE args"


@rm_notify.notify2()
class ReleaseRmFunctionalTest(rm_bt.BaseReleaseMachineTask):
    class Requirements(task_env.TinyRequirements):
        disk_space = 5 * 1024  # 5 Gb

    class Parameters(rm_params.BaseReleaseMachineParameters):
        debug_mode = sdk2.parameters.Bool("Debug mode", default=False)
        _lbrp = binary_task.binary_release_parameters(stable=True)

    @decorators.memoized_property
    def _c_info(self):
        return rmc.get_component(component_name=test_cfg.ReleaseMachineTestCfg.name)

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        with self.memoize_stage.create_children:
            self.Context.tasks_archive_resource = None
            if not self.Parameters.debug_mode:
                self.check_context()
            self.run_tasks_and_save_them_to_context()
        self.check_statuses_of_tasks()

    def run_tasks_and_save_them_to_context(self):  # todo: figure out how to test release block
        released_resource_info = self._c_info.releases_cfg__resources_info[0]
        last_released_resource_id, _ = rs.by_last_released_task(released_resource_info.resource_type)
        self.Context.resource_id = last_released_resource_id
        resource = sdk2.Resource[last_released_resource_id]
        release_item = rm_core.ReleasedItem(released_resource_info.name, resource, released_resource_info.build_ctx_key)
        major_num, minor_num = map(int, self._c_info.get_release_numbers(release_item))
        release_task_success = self.create_and_run_release_task(
            "Should finish successfully", major_num, minor_num
        )
        self.Context.release_task_success_id = release_task_success.id
        self.Context.save()
        self.set_task_info(release_task_success, u"Release")
        raise sdk2.WaitTask(release_task_success, (ctt.Status.Group.BREAK, ctt.Status.Group.FINISH))

    def set_task_info(self, task, task_name):
        self.set_info(
            u"{} task {} was created.".format(task_name, lb.task_link(task.id, task.type)),
            do_escape=False,
        )

    def create_and_run_release_task(self, description, major_release_num, minor_release_num):
        return rm_release_task.ReleaseRmComponent2(
            self,
            description=description,
            component_name=self._c_info.name,
            component_resources={self._c_info.releases_cfg__resources_info[0].resource_name: self.Context.resource_id},
            wait_for_deploy=False,
            where_to_release=rm_const.ReleaseStatus.stable,
            major_release_num=major_release_num,
            minor_release_num=minor_release_num,
            deploy_system=rm_const.DeploySystem.ya_deploy.name,
            tasks_archive_resource=self.Context.tasks_archive_resource,
            debug_mode=True,
        ).save().enqueue()

    def check_context(self):
        if not self.Context.__te_apiargs:
            eh.check_failed(CHECK_FAILED_TEMPLATE.format("task", ""))
        api_args = json.loads(self.Context.__te_apiargs)
        if TAGS_KEY not in api_args:
            eh.check_failed(CHECK_FAILED_TEMPLATE.format("task", "{} in ".format(TAGS_KEY)))
        if u"TESTENV-PRECOMMIT-CHECK" in api_args[TAGS_KEY] and not self.Context.arcadia_patch:
            self.set_info("No arcadia patch in precommit change")
            return
        if ARCHIVE_RESOURCE_KEY not in api_args:
            eh.check_failed(CHECK_FAILED_TEMPLATE.format("patch", "{} in ".format(ARCHIVE_RESOURCE_KEY)))
        self.Context.tasks_archive_resource = api_args[ARCHIVE_RESOURCE_KEY]

    def check_statuses_of_tasks(self):
        release_operation_status = self.find(id=self.Context.release_task_success_id).first().status
        failure = False
        if release_operation_status != ctt.Status.SUCCESS:
            self.set_info("Failed to release task {}".format(self.Context.release_task_success_id))
            failure = True
        if failure:
            eh.check_failed("ReleaseRmComponent tests are broken (see above)")
        self.set_info("All tests passed successfully")
