import graphviz
from jinja2 import Environment

from sandbox.common.types.task import Status

from .release_graph import build_release_status_graph


STATUS_COLOR_MAP = {
    Status.SUCCESS: "#18A651",
    Status.RELEASED: "#18A651",

    Status.ASSIGNED: "#1DAFF0",
    Status.PREPARING: "#1DAFF0",
    Status.SUSPENDING: "#1DAFF0",
    Status.EXECUTING: "#1DAFF0",
    Status.FINISHING: "#1DAFF0",

    Status.FAILURE: "#FD0D1B",
    Status.EXCEPTION: "#FD0D1B",
    Status.TIMEOUT: "#FD0D1B",
    Status.STOPPED: "#FD0D1B",

    # Following colors differ from original for better experience
    Status.WAIT_TIME: "#1DAFF0",
    Status.WAIT_TASK: "#1DAFF0",
}
LABEL_TEMPLATE_J2 = """<<table cellspacing="5" border="1" bgcolor="{% if test_info.sb_task_info -%}
{{ color_map.get(test_info.sb_task_info.status, "grey") }}{% else %}grey{%- endif %}">
<tr>
    <td border="0" colspan="2" valign="middle"><font face="arial" color="white"><b> {{ test_name }} </b></font></td>
</tr>
{% if test_info.sb_task_info %}
<tr>
    <td border="0" align="left" valign="middle"><font face="arial" color="white">status</font></td>
    <td border="0" valign="middle" href="{{ test_info.sb_task_info.link }}"><font color="white" face="arial"><b>{{ test_info.sb_task_info.status }}</b></font></td>
</tr>
<tr>
    <td border="0" align="left" valign="middle"><font face="arial" color="white">author</font></td>
    <td border="0" bgcolor="#383838" align="left" valign="middle" href="https://staff.yandex-team.ru/{{ test_info.sb_task_info.author }}">
        <table align="left" border="0" cellpadding="0" cellspacing="0">
            <tr>
                <td align="right" valign="middle"><font face="arial" color="red">{{ test_info.sb_task_info.author[0] }}</font></td>
                <td align="left" valign="middle"><font face="arial" color="white">{{ test_info.sb_task_info.author[1:] }}</font></td>
            </tr>
        </table>
    </td>
</tr>
{% if do_print_output %}
{% for k, v in test_info.sb_task_info.output_parameters.items() %}
<tr>
    <td border="0" align="left" valign="middle"><font face="arial" color="white">{{ k }}</font></td>
    <td border="0" align="left" valign="middle"><font face="arial" color="white">{{ v }}</font></td>
</tr>
{% endfor %}
{% endif %}
{% else %}
<tr>
    <td border="0" align="left" valign="middle"><font face="arial" color="white">No launch info</font></td>
</tr>
{% endif %}
</table>>"""


def get_label(test_name, test_info, label_template_j2, do_print_output):
    if label_template_j2 is None:
        return test_name
    env = Environment()
    label_template = env.from_string(label_template_j2)

    return label_template.render(test_name=test_name, test_info=test_info, color_map=STATUS_COLOR_MAP, do_print_output=do_print_output)


def create_release_status_diagram(testenv_helper, sandbox_client, component_name, major=None, minor=1, output_format='svg', label_template_j2=LABEL_TEMPLATE_J2, do_print_output=False):
    tests_graph = graphviz.Digraph(
        component_name,
        node_attr={
            'shape': 'plaintext',
        },
        edge_attr={
            'color': 'grey',
        },
        graph_attr={
            'bgcolor': 'transparent',
            'rankdir': 'LR',
        },
        format=output_format,
    )

    release_status_graph = build_release_status_graph(testenv_helper, sandbox_client, component_name, major, tag_number=minor)
    for test_name, test_info in release_status_graph.items():
        with tests_graph.subgraph(name='cluster_pre') as pre_tests_graph:
            pre_tests_graph.node(test_name, label=get_label(test_name, test_info, label_template_j2, do_print_output=do_print_output))
            for parent in test_info['parent_jobs']:
                pre_tests_graph.edge(parent, test_name)

    return tests_graph
