import json
import logging
from collections import defaultdict

from sandbox.projects.release_machine.components.all import get_component

from .testenv import get_aux_base_for_revision, get_launches_for_revision, get_revision_for_aux_base


logger = logging.getLogger(__name__)


OUTPUT_PARAMETERS_BLACKLIST = (
    "result_url",  # Contains HTML-tags and is too long
    "result_path",  # Too long
)


def get_task_link(task_id):
    return "https://sandbox.yandex-team.ru/task/{}".format(task_id)


def get_sb_tasks_info(sandbox_client, task_ids):
    try:
        tasks_info = sandbox_client.task.read(id=task_ids, limit=len(task_ids))
    except:
        return {task_id: None for task_id in task_ids}
    return {
        task_info['id']: {
            'link': get_task_link(task_info['id']),
            'status': task_info['status'],
            'author': task_info['author'],
            'output_parameters': {k: v for k, v in task_info['output_parameters'].items() if k not in OUTPUT_PARAMETERS_BLACKLIST},
        } for task_info in tasks_info['items']
    }


def get_release_graph(testenv_helper, component_name, test_info_keys=('parent_jobs', )):
    c_info = get_component(component_name)
    trunk_database_name = c_info.testenv_cfg__trunk_db

    tests = {
        t['name']: {
            key: t[key] for key in test_info_keys
        } for t in testenv_helper.get_jobs(trunk_database_name)
    }

    return tests


def get_branch_aux_database(sandbox_client, component_name, database_name, release_number):
    branch_tasks = sandbox_client.task.read(
        type="CREATE_BRANCH_OR_TAG",
        input_parameters=json.dumps({
            "component_name": component_name,
        }),
        output_parameters=json.dumps({
            "new_branch_number": release_number,
        }),
        hidden=True,
        limit=1,
    )["items"]
    if len(branch_tasks) == 0:
        return None
    branch_task = branch_tasks[0]
    task_context = sandbox_client.task[branch_task['id']].context.read()
    te_apiargs = json.loads(task_context['__te_apiargs'])
    return te_apiargs['context']['testenv_auxiliary_database']


def get_tag_revision(sandbox_client, component_name, database_name, release_number, tag_number):
    tag_tasks = sandbox_client.task.read(
        type="CREATE_BRANCH_OR_TAG",
        input_parameters=json.dumps({
            "component_name": component_name,
            "branch_number_for_tag": release_number,
        }),
        output_parameters=json.dumps({
            "new_tag_number": tag_number,
        }),
        hidden=True,
        limit=1,
    )["items"]
    if len(tag_tasks) == 0:
        return None
    return tag_tasks[0]['input_parameters']['revision_for_trunk']


def get_release_status_subgraph(
    testenv_helper,
    sandbox_client,
    release_graph,
    database_name,
    root_node_name,
    desired_revision=None,
    desired_aux_db_name=None,
    only_manual_runs=True,
):
    release_status = defaultdict(dict)
    if desired_aux_db_name is None and desired_revision is not None:
        desired_aux_db_name = get_aux_base_for_revision(testenv_helper, database_name, root_node_name, desired_revision)

    if desired_aux_db_name is not None and desired_revision is None:
        desired_revision = get_revision_for_aux_base(testenv_helper, database_name, desired_aux_db_name)

    if desired_revision is not None:
        sb_tasks_dict = {}
        if not only_manual_runs:
            sb_tasks_dict.update(get_launches_for_revision(testenv_helper, database_name, desired_revision, release_graph.keys()))

        if desired_aux_db_name is not None:
            sb_tasks_dict.update(get_launches_for_revision(testenv_helper, desired_aux_db_name, desired_revision, release_graph.keys()))

        sb_tasks_info = get_sb_tasks_info(sandbox_client, sb_tasks_dict.values())
        for test_name, task_id in sb_tasks_dict.items():
            release_status[test_name].update(sb_task_info=sb_tasks_info.get(task_id))

    return release_status


def build_release_status_graph(testenv_helper, sandbox_client, component_name, release_number=None, tag_number=1):
    release_graph = get_release_graph(testenv_helper, component_name)
    pre_release_status = {}
    testing_release_status = {}
    if release_number:
        c_info = get_component(component_name)
        pre_release_aux_db_name = get_branch_aux_database(sandbox_client, component_name, c_info.testenv_cfg__trunk_db, release_number)
        logger.debug("Found pre_release aux database: %s", pre_release_aux_db_name)
        pre_release_status = get_release_status_subgraph(
            testenv_helper,
            sandbox_client,
            release_graph,
            c_info.testenv_cfg__trunk_db,
            "_ACTION_PRE_RELEASE__{}".format(component_name.upper()),
            desired_revision=None,
            desired_aux_db_name=pre_release_aux_db_name,
            only_manual_runs=True,
        )

        testing_release_revision = get_tag_revision(sandbox_client, component_name, c_info.testenv_cfg__db_template.format(testenv_db_num=release_number), release_number, tag_number)
        logger.debug("Found testing_release_revision: %s", testing_release_revision)
        testing_release_status = get_release_status_subgraph(
            testenv_helper,
            sandbox_client,
            release_graph,
            c_info.testenv_cfg__db_template.format(testenv_db_num=release_number),
            "_TESTING_RELEASE_{}".format(component_name.upper()),
            desired_revision=testing_release_revision,
            desired_aux_db_name=None,
            only_manual_runs=False,
        )

    status_graph = {}
    for test_name, test_info in release_graph.items():
        test_info.update(pre_release_status.get(test_name, {}))
        test_info.update(testing_release_status.get(test_name, {}))
        status_graph[test_name] = test_info

    return status_graph
