import hashlib
import logging

import sandbox.projects.release_machine.core as rm_core
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.helpers.events_helper as events_helper
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.tasks.MergeRollbackBaseTask as rm_merge_rollback_task
import sandbox.sdk2 as sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine import arc_helper
from sandbox.projects.release_machine.helpers import commit
from sandbox.projects.release_machine.helpers import merge_helper
from sandbox.projects.release_machine.helpers import review_helper
from sandbox.projects.release_machine.helpers import svn_helper
from sandbox.projects.release_machine.input_params2 import set_choices


RollbackMode = rm_const.RollbackMode


@rm_notify.notify2()
class RollbackCommit(rm_merge_rollback_task.MergeRollbackBaseTask):
    """
        Rolls back specific commit(s) from one or more specified branches.
        It has various modes:

        - Trunk mode is to rollback commits from trunk.
        - Release Machine component mode is to rollback commit from branch, created by Release Machine.
        - Trunk+merge mode is to rollback commit from trunk and merge that rollback into branch simultaneously.
        - Custom mode is to rollback from other paths, for example not RM-branches.

        Console mode of this task is also available as arcadia/devtools/search-env/rollback_commit.py

        Docs: https://wiki.yandex-team.ru/ReleaseMachine/merge-scripts/
    """

    class Parameters(rm_merge_rollback_task.MergeRollbackBaseTask.Parameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.String("Rollback mode", default=RollbackMode.TRUNK_MODE) as rollback_mode:
            rollback_mode.values[RollbackMode.TRUNK_MODE] = rollback_mode.Value(value="Trunk")
            rollback_mode.values[RollbackMode.TRUNK_AND_MERGE_MODE] = rollback_mode.Value(value="Trunk + merge")
            rollback_mode.values[RollbackMode.RELEASE_MACHINE_MODE] = rollback_mode.Value(
                value="Release Machine component",
            )
            rollback_mode.values[RollbackMode.CUSTOM_MODE] = rollback_mode.Value(value="Custom branch")

        with rollback_mode.value[RollbackMode.TRUNK_MODE]:
            # RMDEV-391
            with sdk2.parameters.String(
                "Component name (if applicable)",
                default="",
                required=False,
            ) as component_name_in_trunk:
                set_choices(component_name_in_trunk, rmc.get_component_names())

        with rollback_mode.value[RollbackMode.TRUNK_AND_MERGE_MODE]:
            branch_nums_to_merge = sdk2.parameters.List(
                "Branch numbers",
                sdk2.parameters.Integer,
            )
            with sdk2.parameters.String("Component name", default="", required=True) as component_name_to_merge:
                set_choices(component_name_to_merge, rmc.get_component_names())

            merge_to_old_branches = sdk2.parameters.Bool(
                "Merge to old branches (SEARCH-2257). "
                "By default, merge to branches older than released stable is NOT performed",
                default_value=False,
            )

        with rollback_mode.value[RollbackMode.RELEASE_MACHINE_MODE]:
            # Rollback commits from RM component branches, but not from trunk
            # We provide only single-branch rollbacks because commit number in different branches
            # can be different in each branch (e.g. when commit was merged from trunk).
            # The ony case when it will be the same is the case of very old commit that
            # exists in several branches. This case should be very rare, and even in this case
            # commit should be reverted from trunk and merged to branches via 'trunk + merge' mode.
            branch_num = sdk2.parameters.Integer("Branch number", required=True)
            with sdk2.parameters.String("Component name", default="", required=True) as component_name:
                set_choices(component_name, rmc.get_component_names())

        with rollback_mode.value[RollbackMode.CUSTOM_MODE]:
            branch_path = sdk2.parameters.String("Branch path", required=True)
            revision_to_checkout = sdk2.parameters.Integer(
                "Revision to checkout. Please, leave it blank",
                default_value=None,
            )

        revs = sdk2.parameters.String("Revisions to rollback")
        set_te_marker = sdk2.parameters.Bool(
            "Set '[fix:component:revision]' marker for TestEnv",
            description="RMDEV-391",
            default=False,
        )

    class Context(rm_merge_rollback_task.MergeRollbackBaseTask.Context):
        type = "ROLLBACK_COMMIT"
        rolled_back = False
        rolled_back_revs = []

    action_type = rm_const.ActionType.ROLLBACK

    @property
    def action_mode(self):
        return self.Parameters.rollback_mode

    @sdk2.header()
    def header(self):
        header_parts = super(RollbackCommit, self).header()
        header_parts.append("Rolled back revisions:")
        for revision in self.Context.rolled_back_revs:
            if revision:
                header_parts.append(lb.revision_link(revision))
        if self.action_mode == RollbackMode.TRUNK_AND_MERGE_MODE:
            header_parts.append("Merged revisions:")
            for revision in self.Context.merged_revs:
                if revision:
                    header_parts.append(lb.revision_link(revision))

        return "</br\n>".join(header_parts)

    def _check_input_params(self):
        results = []
        if self.action_mode != RollbackMode.RELEASE_MACHINE_MODE:
            results.append(arc_helper.extract_revisions(self.Parameters.revs, self._arc))
        if self.action_mode == RollbackMode.TRUNK_MODE:
            if self.Parameters.component_name_in_trunk:
                results.append(self._check_component_name(self.Parameters.component_name_in_trunk))
        elif self.action_mode == RollbackMode.TRUNK_AND_MERGE_MODE:
            if self.Parameters.branch_nums_to_merge:
                results.append(merge_helper.check_branch_numbers_correctness(self.Parameters.branch_nums_to_merge))
            results.append(self._check_component_name(self.Parameters.component_name_to_merge))
        elif self.action_mode == RollbackMode.RELEASE_MACHINE_MODE:
            branch_num_check = merge_helper.check_branch_numbers_correctness([self.Parameters.branch_num])
            component_check = self._check_component_name(self.Parameters.component_name)
            results.append(branch_num_check)
            results.append(component_check)
            component_name = self.Parameters.component_name
            if branch_num_check.ok and component_check.ok:
                results.append(arc_helper.extract_revisions(
                    self.Parameters.revs,
                    self._arc,
                    reverse=True,
                    c_info=rmc.COMPONENTS[self.Parameters.component_name](),
                    branch_number=self.Parameters.branch_num,
                ))
                results.append(
                    merge_helper.check_branch_existence(
                        comp_name=component_name,
                        branch=rmc.COMPONENTS[component_name]().full_branch_path(self.Parameters.branch_num),
                        rev_to_checkout=self.Parameters.revision_to_checkout,
                    )
                )
        elif self.action_mode == RollbackMode.CUSTOM_MODE:
            if self.Parameters.revision_to_checkout:
                results.append(merge_helper.check_revs_correctness([str(self.Parameters.revision_to_checkout)]))
        else:
            results.append(rm_core.Error("You should choose at least one mode!"))
        return self._check_results_list(results)

    def _prepare_components_and_branch_num(self, revs):
        if self.action_mode in [RollbackMode.TRUNK_MODE, RollbackMode.TRUNK_AND_MERGE_MODE]:
            rollback_path = svn_helper.RollbackTrunkPath(getattr(self.ramdrive, "path", None))
        elif self.action_mode == RollbackMode.RELEASE_MACHINE_MODE:
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            rollback_path = svn_helper.RollbackReleaseMachinePath(
                self.Parameters.branch_num, c_info, getattr(self.ramdrive, "path", None)
            )
        elif self.action_mode == RollbackMode.CUSTOM_MODE:
            custom_path = commit.check_or_complete_custom_path(self.Parameters.branch_path)
            rollback_path = svn_helper.RollbackCustomPath(custom_path, getattr(self.ramdrive, "path", None))

        if self.action_mode == RollbackMode.RELEASE_MACHINE_MODE:
            rollback_action_items = [rm_const.ActionItem(self.action_mode, c_info, [rollback_path])]
        else:
            rollback_action_items = [rm_const.ActionItem(self.action_mode, None, [rollback_path])]
        commit_info = self._get_commit_info_for_path(
            revs, rollback_path.full, rm_const.ActionType.ROLLBACK
        )
        return rollback_action_items, {}, commit_info

    def commit_without_checks(self, rollback_action_items, revs, commit_info, rollback_path):
        # We don't need to run precommit checks before merge
        common_path = svn_helper.get_common_path(revs, rollback_path.full)
        logging.info("Trying to rollback to branch: %s", rollback_path.full)
        merge_result = self._do_action(rollback_action_items, revs, commit_info, common_path)
        self.Context.rolled_back_revs = merge_result.result_revs
        if self.action_mode == RollbackMode.TRUNK_AND_MERGE_MODE:
            revs = map(int, self.Context.rolled_back_revs)
            merge_action_items, _ = self.get_components_and_branch_nums(
                None,  # We don't need commit_info because we don't use `merge_to` in RollbackCommit
                self.Parameters.component_name_to_merge,
            )
            eh.ensure(
                len(merge_action_items) == 1,
                "There are %s items in merge_action_items list, but only 1 is acceptable" % len(merge_action_items),
            )

            try:
                common_path = svn_helper.get_common_path(revs, rollback_path.full)
                merge_result = self._do_action(merge_action_items, revs, commit_info, common_path)
                self.Context.merged_revs = merge_result.result_revs
            except Exception as exc:
                logging.error("Got exception while merging %s", exc)
                if not self.Parameters.do_commit and not revs:
                    logging.debug("Dry mode, don't fail while merging")
                else:
                    raise
        return merge_result

    def _on_execute(self, revs, commit_info, rollback_action_items, components_and_last_branches):

        rollback_path = rollback_action_items[0].merge_types[0]  # We have just one merge_type in rollback
        if not self.Parameters.need_check:
            merge_result = self.commit_without_checks(rollback_action_items, revs, commit_info, rollback_path)
        else:
            self.check_review_creation_possibility(action_items=rollback_action_items)
            merge_result = self._create_and_wait_reviews(rollback_action_items, revs, commit_info, rollback_path)
        self.add_info_for_conflicts()
        eh.ensure(merge_result.is_ok(), "Rollback failed, see logs for details")
        if self.Parameters.do_commit:
            if self.action_mode in [RollbackMode.TRUNK_AND_MERGE_MODE, RollbackMode.TRUNK_MODE]:
                rollback_msg_part = "trunk"
            else:
                rollback_msg_part = "branch: {}".format(
                    ", ".join([path.short for path in merge_result.successes]),
                )
            msg_part = u"successfully rolled back from {}\n".format(rollback_msg_part)

            if self.action_mode == RollbackMode.TRUNK_AND_MERGE_MODE:
                msg_part += u"Also merged to branches: {}\n".format(
                    ", ".join([path.short for path in merge_result.successes]),
                )

            if self.action_mode != RollbackMode.TRUNK_MODE or self.Parameters.component_name_in_trunk:
                # Now these notifications are sent only to `rm_maintainers` (aka Release Machine Error Monitor) chat
                # We don't want to see them (in case of success), at least now.
                # See also RMDEV-1301
                self._send_notifications_for_succeed_action(rollback_action_items, revs, msg_part, commit_info)

    def _create_and_wait_reviews(self, rollback_action_items, revs, commit_info, rollback_path):
        do_commit = self.Parameters.do_commit
        with self.memoize_stage.review_creation(commit_on_entrance=False):
            # We should create reviews first
            rollback_reviews_results = list(review_helper.create_reviews(
                task=self,
                action_items=rollback_action_items,
                revs=revs,
                commit_info=commit_info,
                base_url=rollback_path.full,
                do_commit=do_commit,
            ))
            self._present_reviews_results(reviews_results=rollback_reviews_results)
            if self._check_reviews_results(reviews_results=rollback_reviews_results):
                self.Context.save()
                # Sleep for 30 minutes to wait tests statuses
                raise sdk2.WaitTime(30 * 60)
            else:
                eh.check_failed("Reviews for some merges haven't been created")
        # We need to check review_results twice in case TRUNK_AND_MERGE_MODE
        max_runs = 2 if self.action_mode == RollbackMode.TRUNK_AND_MERGE_MODE else 1
        with self.memoize_stage.review_checking(max_runs=max_runs, commit_on_entrance=False):
            if not self.Context.rolled_back:
                # We have waited for all tests to finish, now we need to check their statuses
                merge_result = self._check_all_reviews(
                    action_items=rollback_action_items,
                    do_commit=do_commit,
                    revs=revs,
                    commit_info=commit_info,
                )
                self.Context.rolled_back_revs = merge_result.result_revs
                if self.action_mode == RollbackMode.TRUNK_AND_MERGE_MODE:
                    merge_reviews_results = self._create_reviews_for_merges(commit_info, rollback_path)
                    self._present_reviews_results(reviews_results=merge_reviews_results)
                    if self._check_reviews_results(reviews_results=merge_reviews_results):
                        self.Context.rolled_back = True
                        self.Context.save()
                        # Sleep for 30 minutes to wait tests statuses
                        raise sdk2.WaitTime(30 * 60)
                    else:
                        eh.check_failed("Reviews for some merges haven't been created")
            else:
                revs = self.Context.rolled_back_revs
                merge_action_items = self._prepare_c_info_and_merge_types()
                merge_result = self._check_all_reviews(
                    action_items=merge_action_items,
                    do_commit=do_commit,
                    revs=revs,
                    commit_info=commit_info,
                )
                self.Context.merged_revs = merge_result.result_revs
        return merge_result

    def _prepare_c_info_and_merge_types(self):
        c_info = rmc.COMPONENTS[self.Parameters.component_name_to_merge]()
        branch_nums, need_check = self._get_branch_nums(c_info)
        logging.debug("Got branches to merge to %s for component %s", branch_nums, c_info.name)
        merge_types = []
        for branch_num in branch_nums:
            merge_types.append(
                svn_helper.MergeReleaseMachinePath(branch_num, c_info, getattr(self.ramdrive, "path", None))
            )
        merge_action_items = [rm_const.ActionItem(self.action_mode, c_info, merge_types)]
        return merge_action_items

    def _create_reviews_for_merges(self, commit_info, rollback_path):
        merge_action_items = self._prepare_c_info_and_merge_types()
        revs = self.Context.rolled_back_revs
        if revs[0]:
            # Need to create new reviews to merge revision of rollback to branches
            merge_reviews_results = list(review_helper.create_reviews(
                task=self,
                action_items=merge_action_items,
                revs=revs,
                commit_info=commit_info,
                base_url=rollback_path.full,
                do_commit=self.Parameters.do_commit,
            ))
            return merge_reviews_results

    def _on_success_commit(self, merge_type, commited_revision, revs):
        # Fix for RMDEV-912
        # Use original commit ids as removed for rollback events
        proto_events = []

        for rev in revs:
            if self.action_mode in [RollbackMode.TRUNK_MODE, RollbackMode.TRUNK_AND_MERGE_MODE]:
                # Rollbacks from trunk
                proto_event = self._get_rollback_proto_event(
                    merge_type.c_info.name,
                    rev,
                    commited_revision,
                    True,
                    'SUCCESS',
                )
                if proto_event:
                    proto_events.append(proto_event)

            if self.action_mode in [RollbackMode.TRUNK_AND_MERGE_MODE, RollbackMode.RELEASE_MACHINE_MODE]:
                # Rollbacks from branches
                proto_event = self._get_rollback_proto_event(
                    merge_type.c_info.name,
                    rev,
                    commited_revision,
                    False,
                    'SUCCESS',
                )
                if proto_event:
                    proto_events.append(proto_event)

        events_helper.post_proto_events(proto_events)

    def _get_rollback_proto_event(self, component_name, removed, commit_id, trunk_rollback, status=None):

        if not self.is_binary_run:
            logging.warning("This is not a binary run so not building any new-style protobuf events")
            return

        try:

            from release_machine.common_proto import events_pb2
            from release_machine.public import events

            status = status or self.status

            event = events_pb2.EventData(
                general_data=events_pb2.EventGeneralData(
                    hash=events.get_event_hash(
                        component_name,
                        'RollbackCommit',
                        removed,
                        commit_id,
                        status,
                    ),
                    component_name=component_name,
                    referrer="sandbox_task:{}".format(self.id),
                ),
                task_data=events_pb2.EventSandboxTaskData(
                    task_id=self.id,
                    status=status,
                    created_at=self.created.isoformat(),
                    updated_at=self.updated.isoformat(),
                ),
                rollback_commit_data=events_pb2.RollbackCommitData(
                    rolled_back_commit_id=str(removed),
                    new_commit_id=str(commit_id),
                    scope_number=str(self.Parameters.branch_num),
                    is_trunk_rollback=bool(trunk_rollback),
                    rollback_notes=self.Parameters.commit_message,
                ),
            )

            return event

        except Exception:
            # not using log_exception here since there's no need to bother our users
            logging.exception(
                "Unable to build proto event for %s: removed = %s, commit = %s, trunk_rollback = %s, status = %s",
                component_name,
                removed,
                commit_id,
                trunk_rollback,
                status,
            )

    def _get_rollback_started_proto_event(self, component_name, removed, trunk_rollback, status=None):

        if not self.is_binary_run:
            logging.warning("This is not a binary run so not building any new-style protobuf events")
            return

        try:

            from release_machine.common_proto import events_pb2

            event = events_pb2.EventData(
                general_data=events_pb2.EventGeneralData(
                    hash=hashlib.md5(u"{}".format(rm_const.EVENT_HASH_SEPARATOR.join([
                        component_name,
                        'RollbackStarted',
                        removed,
                        status,
                    ]))).hexdigest(),
                    component_name=component_name,
                    referrer="sandbox_task:{}".format(self.id),
                ),
                task_data=events_pb2.EventSandboxTaskData(
                    task_id=self.id,
                    status=(status or self.status),
                    created_at=self.created.isoformat(),
                    updated_at=self.updated.isoformat(),
                ),
                rollback_started_data=events_pb2.RollbackStartedData(
                    rolled_back_commit_id=str(removed),
                    scope_number=str(self.Parameters.branch_num),
                    is_trunk_rollback=bool(trunk_rollback),
                    rollback_notes=self.Parameters.commit_message,
                ),
            )

            return event

        except Exception:
            # not using log_exception here since there's no need to bother our users
            logging.exception(
                "Unable to build proto event for %s: removed = %s, commit = %s, trunk_rollback = %s, status = %s",
                component_name,
                removed,
                trunk_rollback,
                status,
            )

    def _get_started_events(self):

        if self.Parameters.rollback_mode == RollbackMode.TRUNK_MODE:
            component_name = self.Parameters.component_name_in_trunk
            trunk_flag_values = (True, )
        elif self.Parameters.rollback_mode == RollbackMode.TRUNK_AND_MERGE_MODE:
            component_name = self.Parameters.component_name_to_merge
            trunk_flag_values = (True, False)
        else:
            return []

        extracted_commits_info = self._get_extracted_commits_info()

        events = []

        for rev in extracted_commits_info.revs:

            for trunk_flag in trunk_flag_values:

                event = self._get_rollback_started_proto_event(
                    component_name,
                    rev,
                    trunk_flag,
                )

                if not event:
                    continue

                events.append(event)

        return events
