import os
import logging

from sandbox import sdk2
from sandbox.common import errors as sb_errors
from sandbox.common import utils as sb_utils
from sandbox.common.types import resource as sb_resource_types
from sandbox.projects.common import binary_task
from sandbox.projects.common import constants as sb_common_constants
from sandbox.projects.release_machine.tasks import base_task as rm_bt
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine import resources as rm_res
import sandbox.projects.release_machine.input_params2 as rm_params2
from sandbox.sdk2.helpers import subprocess as sp

ALEMBIC_COMMAND_DEFAULT = "upgrade head"
DB_URL_YAV_SECRET_NAME_DEFAULT = "db_url"


class LastReleasedAlembicResource(sdk2.parameters.Resource):
    description = "https://a.yandex-team.ru/arc/trunk/arcadia/release_machine/release_machine/alembic"
    resource_type = rm_res.RM_ALEMBIC_BINARY
    attrs = {"released": "stable"}

    @sb_utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            attrs=cls.attrs,
            state=sb_resource_types.State.READY,
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class RunAlembicMigrations(rm_bt.BaseReleaseMachineTask):
    """
    Runs alembic migrations
    """
    class Requirements(task_env.TinyRequirements):
        disk_space = 4 * 1024  # 4 Gb

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        rm_proto_event = ""
        event_ready = None

    class Parameters(rm_params2.DefaultReleaseMachineParameters):

        _lbrp = binary_task.binary_release_parameters(stable=True)

        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl(
            name=sb_common_constants.ARCADIA_URL_KEY,
            required=True,
            description='Svn url for arcadia',
            default_value=sdk2.svn.Arcadia.trunk_url(),
        )

        db_url_yav_secret_uuid = sdk2.parameters.YavSecret(
            "YaV UUID for DB URL (should contain login data as well)",
            description="You'll need to delegate this secret to Sandbox first. "
                        "This should be done once per secret. "
                        "https://wiki.yandex-team.ru/sandbox/yav/",
        )
        db_url_yav_secret_name = sdk2.parameters.String(
            "DB URL Yav secret name",
            description="The name of the DB URL entry in the given secret",
            default=DB_URL_YAV_SECRET_NAME_DEFAULT,
        )

        alembic_binary_resource = LastReleasedAlembicResource("Alembic binary resource")

        project_root = sdk2.parameters.String(
            "Where should we run alembic (relative to arcadia root)",
            description="The task will cd to this directory and run alembic binary",
        )

        alembic_command = sdk2.parameters.String(
            "Alembic command to run",
            description="This will be sent to alembic binary (`./alembic <alembic_command>`). "
                        "Do not edit unless you known what you're doing",
            default=ALEMBIC_COMMAND_DEFAULT,
        )

    @property
    def db_url(self):
        return self.Parameters.db_url_yav_secret_uuid.data()[self.Parameters.db_url_yav_secret_name]

    @property
    def alembic_env(self):
        return {
            "DB_URL": self.db_url,
        }

    @property
    def alembic_binary_path(self):
        return str(sdk2.ResourceData(self.Parameters.alembic_binary_resource).path)

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        project_local_path = self.checkout_and_cd_to_project_root()
        self.run_migrations(project_local_path)
        self.store_migration_history(project_local_path)

    def checkout_and_cd_to_project_root(self):
        checkout_url = sdk2.svn.Arcadia.append(self.Parameters.checkout_arcadia_from_url, self.Parameters.project_root)
        local_path = sdk2.svn.Arcadia.checkout(checkout_url, self.Parameters.project_root)
        return local_path

    def run_migrations(self, project_local_path):

        cmd = [self.alembic_binary_path]
        cmd.extend(self.Parameters.alembic_command.split(" "))

        with sdk2.helpers.ProcessLog(self, logger="alembic_run") as pl:
            alembic_process = sp.Popen(
                cmd,
                cwd=project_local_path,
                env=self.alembic_env,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            alembic_process.wait()

        if alembic_process.returncode:
            raise sb_errors.TaskFailure('Migration failed')

    def store_migration_history(self, project_local_path):

        cmd = [self.alembic_binary_path, "history"]

        with sdk2.helpers.ProcessLog(self, logger="alembic_history_run") as pl:

            process = sp.Popen(
                cmd,
                cwd=project_local_path,
                env=self.alembic_env,
                stdout=sp.PIPE,
                stderr=pl.stderr,
            )

            process.wait()

        if process.returncode:
            self.set_info("Unable to load migration history")
            return

        out = process.communicate()

        logging.debug(out)

        try:
            history_list = out[0].split(os.linesep)[1:]
        except IndexError:
            self.set_info("Unable to parse migration history")
            return

        self.set_info("\n".join(history_list))
