import os
import sys
import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import error_handlers
from sandbox.projects.common import parameters as common_parameters
from sandbox.projects.release_machine.core import task_env, Ok, Error, const
from sandbox.projects.release_machine.tasks import base_task
from sandbox.projects.release_machine import input_params2
from sandbox.projects.release_machine.tasks.RunIceFlame import exceptions
from sandbox.projects import resource_types


if sys.version_info[0:2] >= (3, 8):  # python 3.8 or higher
    from functools import cached_property
else:
    from sandbox.projects.common.decorators import memoized_property as cached_property


def dry_run_rescue(default_value="", catch=(AttributeError, KeyError)):
    """
    Return default value instead of raising exception when `dry_run` task parameter is on

    :param default_value: default value to be returned in case of exception of one of the given types
    :param catch: a tuple of exceptions that should be caught in suppressed if dry_run is on
    """

    def decorator(f):

        def wrapper(instance, *args, **kwargs):

            try:

                return f(instance, *args, **kwargs)

            except catch:

                logging.exception("Exception caught in %s", f.__name__)

                if instance.Parameters.dry_run:
                    logging.info("Dry run rescue!")
                    return default_value

                raise

        return wrapper

    return decorator


ICEFLAME_ARCADIA_PATH = "https://a.yandex-team.ru/arc/trunk/arcadia/release_machine/iceflame"


class RunIceFlame(base_task.BaseReleaseMachineTask):
    """
    Run IceFlame

    https://a.yandex-team.ru/arc/trunk/arcadia/release_machine/iceflame/README.md

    https://wiki.yandex-team.ru/iceflame/

    Hint: In case you already got a protobuf text config for IceFlame
    just use `iceflame -c <your_config_file> print-config --json` to JSONify it
    """

    YA_ARCADIA_PATH = "ya"
    STACKCOLLAPSE_ARCADIA_PATH = "contrib/tools/flame-graph/stackcollapse-perf.pl"
    FLAMEGRAPH_ARCADIA_PATH = "contrib/tools/flame-graph/flamegraph.pl"

    class Requirements(task_env.TinyRequirements):
        disk_space = 3 * 1024  # 3 Gb

    class Context(base_task.BaseReleaseMachineTask.Context):
        scope_number = 0    # Production version major release number (Release Machine mode only)
        tag_number = 0      # Production version minor release number (Release Machine mode only)
        skip_event = False  # When set to True prevents events for being sent or detected by polling tools

    # class Parameters(input_params2.BaseReleaseMachineParameters):
    # ^ A temporary solution for DEVTOOLSSUPPORT-7356
    class Parameters(sdk2.Parameters):

        ICEFLAME_CONFIG_SOURCE_INPUT_JSON = "input_json"
        ICEFLAME_CONFIG_SOURCE_FILE = "file"

        _lbrp = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group("IceFlame Config and Run Options") as iceflame_config_parameters:

            with sdk2.parameters.String(
                "Config Source",
                description="Choose IceFlame config source. You can either load one of the IceFlame public configs or "
                            "provide your own configuration as a JSON using the appropriate input parameter",
            ) as iceflame_config_source:

                iceflame_config_source.values[ICEFLAME_CONFIG_SOURCE_INPUT_JSON] = iceflame_config_source.Value(
                    "Input JSON",
                )

                iceflame_config_source.values[ICEFLAME_CONFIG_SOURCE_FILE] = iceflame_config_source.Value(
                    "Choose File",
                    default=True,
                )

                with iceflame_config_source.value.input_json:

                    iceflame_config_json = sdk2.parameters.JSON(
                        "IceFlame config JSON",
                        description="IceFlame config ({}) as JSON".format(
                            os.path.join(ICEFLAME_ARCADIA_PATH, "proto/config.proto")
                        ),
                    )

                    iceflame_config_ignore_unknown = sdk2.parameters.Bool(
                        "IceFlame Config: ignore unknown",
                        description="Ignore unknown fields when parsing iceflame_config_json",
                        default_value=False,
                    )

                with iceflame_config_source.value.file:

                    iceflame_config_file = sdk2.parameters.String(
                        "IceFlame config file name",
                        description="File name from {}".format(
                            os.path.join(ICEFLAME_ARCADIA_PATH, "configs/public")
                        ),
                    )

            iceflame_command_run_options_json = sdk2.parameters.JSON(
                "IceFlame command Run options (remote collector/analyzer) as JSON",
                description=os.path.join(ICEFLAME_ARCADIA_PATH, "proto/run_options.proto"),
            )

        with sdk2.parameters.Group("SSH Parameters") as ssh_params_block:

            ssh_key_secret = sdk2.parameters.YavSecret("YaV secret with SSH private key")

            ssh_login = sdk2.parameters.String(
                "SSH login",
                description="Custom SSH login (leave blank to use task author)",
                default_value="",
            )

        with sdk2.parameters.Group("Tokens and envs") as token_and_env_block:
            sandbox_token_secret = sdk2.parameters.YavSecret("YaV secret with Sandbox token")
            nanny_token_secret = sdk2.parameters.YavSecret("YaV secret with Nanny token")
            env_vars = common_parameters.EnvironmentVariables()

        with sdk2.parameters.Group("Release Machine") as release_machine_block:

            release_machine_mode = sdk2.parameters.Bool(
                "release_machine_mode",
                description="Run in ReleaseMachine mode",
                default_value=False,
            )

            with release_machine_mode.value[True]:

                component_name = input_params2.ComponentName2.component_name()
                release_item_name = sdk2.parameters.String(
                    "RM releasable item name (check Releases.releasable_items section of your config)",
                )
                service_name = sdk2.parameters.String("Service name (for Nanny) or stage name (for Ya.Deploy)")
                stage_label = sdk2.parameters.String(
                    "Stage label: stable/testing/prestable/unstable",
                    default_value="stable",
                )

                fail_if_service_version_cannot_be_determined = sdk2.parameters.Bool(
                    "Fail if service version cannot be determined",
                    default_value=False,
                )

                check_production_version = sdk2.parameters.Bool(
                    "Ensure that the task is not running for the same production version twice",
                    description="When True allows to prevent rerunning on the same production version. Finishes "
                                "successfully (without actually launching IceFlame tool) in case an earlier launch for "
                                "the current production version is found. Launches regularly otherwise.",
                    default_value=False,
                )

        with sdk2.parameters.Group("Debug Options") as debug_options:

            debug = sdk2.parameters.Bool("debug", default_value=False)

            with debug.value[True]:

                custom_rm_host = sdk2.parameters.String("RM host", default_value=const.Urls.RM_HOST)

                dry_run = sdk2.parameters.Bool("Dry run", default_value=False)

        # -= OUTPUT =-
        #
        with sdk2.parameters.Output():
            flamegraph = sdk2.parameters.Resource("FlameGraph", resource_type=resource_types.FLAME_GRAPH_SVG)
            perf_report = sdk2.parameters.Resource("Perf Report", resource_type=resource_types.PERF_REPORT)

    @cached_property
    def rm_url(self):
        return self.Parameters.custom_rm_host or const.Urls.RM_HOST

    @cached_property
    def rm_model_client(self):
        from release_machine.release_machine.services.release_engine.services.Model import ModelClient
        return ModelClient.from_address(self.rm_url)

    @cached_property
    def rm_state_client(self):
        from release_machine.release_machine.services.release_engine.services.State import StateClient
        return StateClient.from_address(self.rm_url)

    @cached_property
    def service_version(self):

        if not self.Parameters.release_machine_mode:
            return None

        from release_machine.release_machine.proto.structures import message_pb2
        from search.martylib.core.exceptions import MaxRetriesReached
        import grpc

        service_version = None

        try:

            service_version = self.rm_model_client.get_service_version(
                message_pb2.GetServiceVersionRequest(
                    component_name=self.Parameters.component_name,
                    service_name=self.Parameters.service_name,
                    stage_label=self.Parameters.stage_label,
                    release_item_name=self.Parameters.release_item_name,
                ),
            )

            if service_version:

                self.Context.scope_number = service_version.major_release_number
                self.Context.tag_number = service_version.minor_release_number

        except (grpc.RpcError, MaxRetriesReached) as error:
            error_handlers.log_exception("Unable to determine service version. RM responded with error", error)

        logging.info(
            "Service version detected for component %s, service %s (%s, release item: %s): %s",
            self.Parameters.component_name,
            self.Parameters.service_name,
            self.Parameters.stage_label,
            self.Parameters.release_item_name,
            service_version,
        )

        if not service_version and self.Parameters.fail_if_service_version_cannot_be_determined:
            error_handlers.check_failed("Service version cannot be determined")

        return service_version

    @cached_property
    def analyze_inplace(self):
        return bool(
            self.Parameters.iceflame_command_run_options_json.get(
                "remote_collector_options",
                {},
            ).get(
                "analyze_inplace",
                False,
            )
        )

    @cached_property
    def iceflame_config(self):

        logging.debug("Going to load config from task parameters")

        logging.info("Config source set to %s", self.Parameters.iceflame_config_source)

        if self.Parameters.iceflame_config_source == self.Parameters.ICEFLAME_CONFIG_SOURCE_INPUT_JSON:
            config = self._load_iceflame_config_from_json_input_parameter()
        else:
            config = self._load_iceflame_config_from_iceflame_public_configs()

        logging.debug("The following config loaded: %s", config)

        logging.debug(
            "Now going to patch it a bit in order to correct local paths for the required binaries and scripts",
        )

        arcadia_svn = "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia"

        ya_path = sdk2.svn.Arcadia.export(
            os.path.join(arcadia_svn, self.YA_ARCADIA_PATH),
            "ya",
        )

        if not self.analyze_inplace:

            stackcollapse_perf_script = sdk2.svn.Arcadia.export(
                os.path.join(arcadia_svn, self.STACKCOLLAPSE_ARCADIA_PATH),
                "stackcollapse_perf_script",
            )
            flamegraph_script = sdk2.svn.Arcadia.export(
                os.path.join(arcadia_svn, self.FLAMEGRAPH_ARCADIA_PATH),
                "flamegraph_script",
            )

            config.flamegraph.stackcollapse_perf_script = stackcollapse_perf_script
            config.flamegraph.flamegraph_script = flamegraph_script

        config.ya.ya_bin = ya_path
        config.custom_sandbox_resource_owner = self.owner

        logging.debug("Patched config: %s", config)

        return config

    @cached_property
    def iceflame_command_run_options(self):

        from release_machine.iceflame.proto import run_options_pb2
        from google.protobuf import json_format

        run_options = run_options_pb2.RunOptions()

        json_format.ParseDict(
            self.Parameters.iceflame_command_run_options_json,
            message=run_options,
            ignore_unknown_fields=self.Parameters.iceflame_config_ignore_unknown,
        )

        return run_options

    @property
    @dry_run_rescue()
    def ssh_key(self):
        return self.Parameters.ssh_key_secret.data()[self.Parameters.ssh_key_secret.default_key]

    @property
    @dry_run_rescue()
    def sandbox_token(self):
        return self.Parameters.sandbox_token_secret.data()[self.Parameters.sandbox_token_secret.default_key]

    @property
    @dry_run_rescue()
    def nanny_token(self):
        return (
            self.Parameters.nanny_token_secret.data()[self.Parameters.nanny_token_secret.default_key]
            if self.Parameters.nanny_token_secret
            else ""
        )

    @cached_property
    def iceflame_tasklib_object(self):

        from release_machine.iceflame.src.lib import task as iceflame_task_lib

        return iceflame_task_lib.IceFlameTask(
            config=self.iceflame_config,
            run_options=self.iceflame_command_run_options,
            ssh_login=(self.Parameters.ssh_login or self.author),
            ssh_key=self.ssh_key,
            sandbox_token=self.sandbox_token,
            nanny_token=(
                self.Parameters.nanny_token_secret.data()[self.Parameters.nanny_token_secret.default_key]
                if self.Parameters.nanny_token_secret
                else ""
            ),
        )

    def on_execute(self):
        base_task.BaseReleaseMachineTask.on_execute(self)

        check_result = self._check_production_version()

        if not check_result.ok:
            self._notify_finish_with_no_launch(check_result.result)
            return

        self._set_env()

        if self.Parameters.debug and self.Parameters.dry_run:
            self._notify_finish_with_no_launch("Dry run. IceFlame tool not going to be launched")
            logging.info("Tasklib object: %s", self.iceflame_tasklib_object)
            return

        self.iceflame_tasklib_object.run(exit_after_run=False, reraise_exception=True)

        self._set_output()

    def on_break(self, prev_status, status):
        self.iceflame_tasklib_object.run_garbage_collection()
        super(RunIceFlame, self).on_break(prev_status, status)

    def _set_env(self):

        logging.info("Going to set environment")

        env_vars = common_parameters.EnvironmentVariablesContainer(self.Parameters.env_vars).as_dict_deref()

        if not env_vars:
            logging.info("No custom environment variables provided")
            return

        os.environ.update(env_vars)

    def _set_output(self):

        from release_machine.iceflame.src.lib.runner import Result

        status, message = self.iceflame_tasklib_object.get_result()

        self.set_info("Result {}: {}".format(status, message))

        self._set_output_resources()

        error_handlers.ensure(
            status != Result.FAILURE,
            message,
        )

    def _set_output_resources(self):

        if not self.iceflame_tasklib_object.analyzer:
            logging.info("No analyzer initialized so no output resources can be set")
            return

        flamegraph_resource_id = self.iceflame_tasklib_object.analyzer.flamegraph_resource_id
        perf_report_resource_id = self.iceflame_tasklib_object.analyzer.perf_report_resource_id

        logging.info("FlameGraph resource: %s", flamegraph_resource_id)
        logging.info("Perf report resource: %s", perf_report_resource_id)

        if flamegraph_resource_id:
            self.Parameters.flamegraph = flamegraph_resource_id

        if perf_report_resource_id:
            self.Parameters.perf_report = perf_report_resource_id

    def _validate_secrets(self):

        self._validate_yav_secret_parameter(self.Parameters.ssh_key_secret.name)
        self._validate_yav_secret_parameter(self.Parameters.sandbox_token_secret.name)

        if self.Parameters.nanny_token_secret:
            self._validate_yav_secret_parameter(self.Parameters.nanny_token_secret.name)

    def _validate_yav_secret_parameter(self, param_name):

        param = getattr(self.Parameters, param_name)

        param_default_key = param.default_key

        if not param_default_key:
            raise exceptions.RunIceFlameSecretParameterDefaultKeyMissing(param_name)

    def _load_iceflame_config_from_json_input_parameter(self):

        logging.info("Loading config from JSON input parameter")

        from release_machine.iceflame.proto import config_pb2
        from google.protobuf import json_format

        config = config_pb2.IceFlameConfig()

        json_format.ParseDict(
            self.Parameters.iceflame_config_json,
            message=config,
            ignore_unknown_fields=self.Parameters.iceflame_config_ignore_unknown,
        )

        return config

    def _load_iceflame_config_from_iceflame_public_configs(self):

        logging.info("Loading config from file")

        from release_machine.iceflame.configs.public import loader

        path = str(self.Parameters.iceflame_config_file)

        try:

            if not path.startswith(loader.PREFIX):
                logging.info("Patching path %s", path)
                path = os.path.join(loader.PREFIX, path)
                logging.info("Result: %s", path)

            return loader.get_config(path)

        except loader.ConfigResourceNotFound as crnf:

            error_handlers.log_exception(
                "Config {path} cannot be found. Here's a list of available public configs:\n - {config_list}".format(
                    path=path,
                    config_list="\n - ".join(loader.list_configs()),
                ),
                crnf,
            )

            error_handlers.fail(
                "Stopping execution: incorrect parameter value iceflame_config_file = {}".format(
                    path,
                ),
            )

    def _check_production_version(self):

        if not self.Parameters.release_machine_mode:
            logging.info("No need to check production version: the task is NOT run is ReleaseMachine mode")
            return Ok()

        self.set_info("The task is running in ReleaseMachine mode")

        if not self.Parameters.component_name:
            return Error("No component")

        logging.info("Current production version (estimate): %s", self.service_version)

        if not self.Parameters.check_production_version:
            logging.info("No need to check production version: the check is omitted according to the launch parameters")
            return Ok()

        logging.info("Going to check current production version")

        from release_machine.release_machine.proto.structures import message_pb2

        previous_run_results = self.rm_state_client.get_ice_flame_result(
            message_pb2.IceFlameResultRequest(
                component_name=self.Parameters.component_name,
                scope_number=self.service_version.major_release_number,
                tag_number=self.service_version.minor_release_number,
            ),
        )

        if previous_run_results.launch_list:
            logging.info("Previous launch(es) found: %s", previous_run_results)
            self.set_info(
                "Previous launch detected (see task logs for details). According to the specified input parameters"
                "(check_production_version = True) no further actions will take place, the task is going to stop.",
            )
            return Error("Previous launch detected")

        return Ok()

    def _get_rm_proto_event_general_data(self, event_time_utc_iso, status=None):
        return self.iceflame_tasklib_object.get_rm_event_general_data(
            component_name=self.Parameters.component_name,
            referrer="sandbox_task:{}".format(self.id),
        )

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):

        if self.Context.skip_event:
            return {}

        return {
            "run_ice_flame_data": self.iceflame_tasklib_object.get_rm_event_run_ice_flame_data(
                flamegraph_resource_id=self.Parameters.flamegraph.id if self.Parameters.flamegraph else "",
                perf_report_resource_id=self.Parameters.perf_report.id if self.Parameters.perf_report else "",
                scope_number=self.Context.scope_number,
                tag_number=self.Context.tag_number,
            ),
        }

    def _notify_finish_with_no_launch(self, msg):
        self.set_info("Finishing (no launch): {}".format(msg))
        self.Context.skip_event = True
        self.Context.rm_proto_event = ""
        self.Context.event_ready = False
        self.Context.save()
