# -*- coding: utf-8 -*-
import logging
import time
import itertools as it
from collections import defaultdict

from sandbox import sdk2
from sandbox.common import enum
from sandbox.common.rest import Client
from sandbox.projects.common import binary_task
from sandbox.projects.common import decorators
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.nanny import client as nanny_client
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine import morty_client
from sandbox.projects.release_machine import resources as rm_res
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.helpers.deploy import basic_releaser
from sandbox.projects.release_machine.tasks import base_task as rm_bt
from sandbox.projects.websearch.begemot import resources as begemot_resources


class ScheduleMode(enum.Enum):
    create_and_schedule = "create_and_schedule"
    just_schedule = "just_schedule"


class ScheduleRelease(rm_bt.BaseReleaseMachineTask):
    """
    Schedule releases by saving them as sandbox resource and pushing it to Morty.

    Now only nanny is supported. Y.Deploy is on its way
    """

    class Requirements(task_env.TinyRequirements):
        disk_space = 512  # 512 Mb

    class Parameters(rm_params.DefaultReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 10 * 60  # 10 min
        major_release_num = sdk2.parameters.Integer("Major release number")
        minor_release_num = sdk2.parameters.Integer("Minor release number")
        flows = sdk2.parameters.List("Flows", default=[])

        warden_component_name = sdk2.parameters.String("Warden component name")
        warden_parent_component_name = sdk2.parameters.String("Warden parent component name")
        with sdk2.parameters.String("Release via") as deploy_system:
            rm_params.set_choices(deploy_system, [i.name for i in rm_const.DeploySystem])

        with sdk2.parameters.String("Release to", default=rm_const.ReleaseStatus.testing) as where_to_release:
            rm_params.set_choices(where_to_release, [x for x in dir(rm_const.ReleaseStatus) if not x.startswith("__")])

        with sdk2.parameters.String("Schedule mode") as schedule_mode:
            schedule_mode.values.create_and_schedule = schedule_mode.Value(
                ScheduleMode.create_and_schedule, default=True
            )
            schedule_mode.values.just_schedule = schedule_mode.Value(ScheduleMode.just_schedule)

        with schedule_mode.value[ScheduleMode.create_and_schedule]:
            component_resources = sdk2.parameters.Dict("Resources dict (ex: {'my_resource_name': 12345678})")
        with schedule_mode.value[ScheduleMode.just_schedule]:
            resource_with_data_to_schedule = sdk2.parameters.Resource(
                "Release data to schedule",
                resource_type=[
                    rm_res.ScheduledRmReleaseData,
                    begemot_resources.BEGEMOT_STABLE_RELEASE_CONFIG,
                    begemot_resources.BEGEMOT_TESTING_RELEASE_CONFIG,
                ],
            )

        with sdk2.parameters.Output:
            release_data = sdk2.parameters.Resource(
                "Release data scheduled",
                resource_type=[
                    rm_res.ScheduledRmReleaseData,
                    begemot_resources.BEGEMOT_STABLE_RELEASE_CONFIG,
                    begemot_resources.BEGEMOT_TESTING_RELEASE_CONFIG,
                ],
            )

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        if self.Parameters.deploy_system not in [
            rm_const.DeploySystem.nanny.name,
            rm_const.DeploySystem.nanny_push.name,
        ]:
            eh.check_failed("Deploy system '{}' is not supported yet!".format(self.Parameters.deploy_system))

        with self.memoize_stage.schedule_release_data:
            self.prepare_release_data()
            self.schedule_release_data()
        self.sb_release_release_data()

    def prepare_release_data(self):
        if self.Parameters.schedule_mode == ScheduleMode.create_and_schedule:
            if not self.Parameters.component_resources:
                eh.check_failed("Please, specify at least one item in component_resources!")
            resources_by_service = defaultdict(list)
            for sb_resource, services in self.iter_over_resources():
                for service in services:
                    resources_by_service[service].append(sb_resource)

            release_data = self.build_release_data(resources_by_service)
            self.save_release_data(release_data)
        elif self.Parameters.schedule_mode == ScheduleMode.just_schedule:
            self.Parameters.release_data = self.Parameters.resource_with_data_to_schedule

    def schedule_release_data(self):
        mc = morty_client.MortyClient(token=self._rm_token)
        for flow in self.Parameters.flows:
            logging.info("Adding event with flow: %s", flow)
            result = mc.add_event({
                "type": "RELEASE",
                "config": {
                    "release_machine": {
                        "rm_component": self.Parameters.component_name,
                        "sandbox_resource_id": str(self.Parameters.release_data.id),
                        "major_release_num": str(self.Parameters.major_release_num),
                        "minor_release_num": str(self.Parameters.minor_release_num),
                    },
                    "flow": flow,
                    "component_name": self.Parameters.warden_component_name,
                    "parent_component_name": self.Parameters.warden_parent_component_name,
                },
                "description": {
                    "title": "RM release {} {}".format(
                        self.Parameters.component_name,
                        self._c_info.release_id(self.Parameters.major_release_num, self.Parameters.minor_release_num)
                    )
                }
            })
            logging.info("Adding event result: %s", result)
            time.sleep(1)

    def sb_release_release_data(self):
        release_data_json = fu.json_load(sdk2.ResourceData(self.Parameters.release_data).path)
        resources_to_release = {i["resource_id"] for i in it.chain.from_iterable(release_data_json.values())}
        task_ids_to_release = {sdk2.Resource[i].task_id for i in resources_to_release}
        logging.info("Going to press release button in tasks: %s", list(task_ids_to_release))
        sb_rest_client = Client()
        for i in task_ids_to_release:
            basic_releaser.press_release_button_on_build_task(
                i, self.Parameters.where_to_release, sb_rest_client,
                major_release_num=self.Parameters.major_release_num,
            )

    def iter_over_resources(self):
        stage_info = list(self._c_info.releases_cfg__iter_over_deploy_info(self.Parameters.where_to_release))
        for sb_resource, res_name in self.get_resource_candidates():
            for res_info, deploy_info in stage_info:
                if res_info.resource_name != res_name:
                    logging.debug("Skip %s, because %s != %s", res_info, res_info.resource_name, res_name)
                    continue
                elif res_info.resource_type != sb_resource["type"]:
                    logging.debug("Skip %s, because %s != %s", res_info, res_info.resource_type, sb_resource["type"])
                    continue
                elif (
                    res_info.attributes and
                    any(sb_resource["attributes"].get(k) != v for k, v in res_info.attributes)
                ):
                    logging.debug("Skip %s, because attributes did not match: %s", res_info, sb_resource["attributes"])
                    continue
                logging.info("Found %s for %s", deploy_info, sb_resource["id"])
                yield sb_resource, self._c_info.get_deploy_services(deploy_info, self._nn_client)
                break
            else:  # means, that no matched resource found in rm config
                self.set_info("Resource '{} {}' was skipped. Check your config!".format(res_name, sb_resource["id"]))

    def get_resource_candidates(self):
        component_resource_names = {int(j): i for i, j in self.Parameters.component_resources.items()}
        resources = self.server.resource.read(
            id=self.Parameters.component_resources.values(),
            limit=len(self.Parameters.component_resources)
        )["items"]
        for resource in resources:
            yield resource, component_resource_names[int(resource["id"])]

    def build_release_data(self, resources_by_service):
        release_data = {}
        for service_id, sb_resources in resources_by_service.items():
            nanny_resources = self._nn_client.get_service_resources(service_id)
            nanny_sandbox_files = nanny_resources["content"]["sandbox_files"]
            changes = []
            for sb_resource in sb_resources:
                modified = False
                for nanny_sandbox_file in nanny_sandbox_files:
                    if nanny_sandbox_file["resource_type"] == sb_resource["type"]:
                        if modified:
                            self.set_info(
                                "{}: more then one {} resource detected".format(service_id, sb_resource["type"])
                            )
                        if nanny_sandbox_file["resource_id"] == sb_resource["id"]:
                            self.set_info("{}: {} resource is already deployed".format(service_id, sb_resource["id"]))
                            continue
                        self.set_info("{} (local_path={}):\nid: {}\n>>\nid: {}".format(
                            service_id, nanny_sandbox_file["local_path"],
                            nanny_sandbox_file["resource_id"], sb_resource["id"]
                        ))
                        changes.append({
                            "local_path": nanny_sandbox_file["local_path"],
                            "resource_id": sb_resource["id"],
                        })
            if changes:
                release_data[service_id] = changes
        return release_data

    def save_release_data(self, result_diff):
        diff_path = "release_data__{}_{}_{}.json".format(
            self._c_info.name, self.Parameters.major_release_num, self.Parameters.minor_release_num,
        )
        fu.json_dump(diff_path, result_diff, indent=2, sort_keys=True)
        self.Parameters.release_data = rm_res.ScheduledRmReleaseData(
            self,
            "Release diff for {} (major={}, minor={})".format(
                self._c_info.name, self.Parameters.major_release_num, self.Parameters.minor_release_num,
            ),
            diff_path,
            deploy_system=self.Parameters.deploy_system,
            rm_component=self._c_info.name,
            major_release_num=self.Parameters.major_release_num,
            minor_release_num=self.Parameters.minor_release_num,
        )

    @decorators.memoized_property
    def _c_info(self):
        return rmc.get_component(self.Parameters.component_name)

    @decorators.memoized_property
    def _rm_token(self):
        return rm_sec.get_rm_token(self)

    @decorators.memoized_property
    def _nn_client(self):
        return nanny_client.NannyClient(api_url=rm_const.Urls.NANNY_BASE_URL, oauth_token=self._rm_token)
