#!/usr/bin/python
# -*- coding: utf-8 -*-

import os
import logging
import sys
import urllib2
import json

from sandbox.sandboxsdk import process
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine.helpers.soy_helper import SoyApi

from sandbox.projects.release_machine.tasks.ScrapeRequests2 import static as st
from sandbox.projects.release_machine.tasks.ScrapeRequests2 import scraper_over_yt_script


class BatchStatus(object):
    QUEUEING = 1
    FAILED = 2
    COMPLETED = 3
    PROCESSING = 4
    FETCHING = 5

    STATUS_NAME = {
        QUEUEING: "Queueing",
        FAILED: "Failed",
        COMPLETED: "Completed",
        PROCESSING: "Processing",
        FETCHING: "Fetching",
    }


BATCH_STATUS_TYPES = {
    "running": BatchStatus.PROCESSING,
    "pending": BatchStatus.QUEUEING,
    "failed": BatchStatus.FAILED,
    "canceled": BatchStatus.FAILED,
    "completed": BatchStatus.COMPLETED,
}


class ScraperOverYtWrapper:
    """
        This wrapper simplifies using of scraper over yt
    """
    @staticmethod
    def create_client(server, token):
        from yt.wrapper import YtClient

        client = YtClient(server, token, config={
            'pickling': {
                'python_binary': '/skynet/python/bin/python'
            }
        })
        logging.info("Created YT client on server %s", server)
        return client

    @staticmethod
    def get_queries(queries_list, host, platform, cgi, id_prefix, region_prefix, uids, user_agent=None):
        """
        Load queries into table
        :return: list of json requests
        """
        coll = st.COLLECTIONS[platform]["name"]
        cgi = cgi or ''
        uri = "https://{}/{}".format(host, coll) + "?text={text}&lr={lr}" + cgi
        query_id = 0
        requests = []
        for idx, query in enumerate(queries_list):
            if isinstance(query, basestring):
                (text, region) = query, 31  # TODO: set default region
            else:
                (text, region) = query
            try:
                final_url = uri.format(text=urllib2.quote(text.encode("utf-8")), lr=region)
            except Exception:
                logging.info("FIND ME")
                logging.info("%s %s", repr(uri), repr(text))
                eh.fail("Found error with encoding")

            requests.append({
                "uri": final_url,
                "id": "-".join([region_prefix, id_prefix, str(query_id)]),
                "method": "GET",
                "headers": ["User-Agent: " + user_agent] if user_agent else [],
                "cookies": ["yandexuid={}".format(uids[idx % len(uids)])] if uids else [],
            })
            query_id += 1
        return requests

    @staticmethod
    def _create_job(input_table, guid, pool="default"):
        spec = {
            "input_table": input_table,
            "movable": False,
            "ignore_sampling": True,
            "prs_ops_mode": False,
            "fetch_timeout": "15s",
            "map_operation_type": "scraper",
        }
        job = {
            "id": guid,
            "pool": pool,
            "added_to_queue": 1,
            "spec": spec,
        }
        return job

    @staticmethod
    def _create_job_for_api(input_table, guid, pool="default", description=""):
        return {
            "input_table": input_table,
            "id": guid,
            "pool": pool,
            "description": description,
        }

    @staticmethod
    def launch_scraper(requests, server, input_table, guid, pool="default", use_api=False, launch_description=""):
        """
        Load requests into YT and launch scraper process
        """
        from yt.wrapper import JsonFormat

        client = ScraperOverYtWrapper.create_client(server, os.environ["YT_TOKEN"])
        logging.info("Create YtClient on server %s", server)

        client.create("table", input_table, recursive=True, ignore_existing=True)
        client.write_table(input_table, requests, format=JsonFormat(attributes={"encode_utf8": False}))
        logging.info("Input data loaded to YT table %s", input_table)

        if use_api:
            logging.info("Making batch via api")
            job = ScraperOverYtWrapper._create_job_for_api(input_table, guid, pool, launch_description)
            logging.info("Job configuration is %s", json.dumps(job))
            api = SoyApi(token=os.environ['SOY_TOKEN'], server=server)
            create_result = api.create(job)
            if create_result.get('status') == 'error':
                error_message = create_result.get('error_msg') or ''
                logging.error(
                    'Failed to start batch with error `%s`, SoY API response is:\n%s',
                    error_message, json.dumps(create_result),
                )
                raise Exception('Failed to start batch. Error: {}'.format(error_message))

            logging.info("Created SoY batch with API response %s", json.dumps(create_result))
        else:
            logging.warning("Not using SOY API (use_api = False)")
            job = ScraperOverYtWrapper._create_job(input_table, guid, pool)
            logging.info("Job configuration is %s", json.dumps(job))
            logging.info("Enqueuing the job")
            client.insert_rows(
                "//home/search-runtime/robot-scraperoveryt/scraper_over_yt/scheduler/operations_queue/queue",
                [job],
                format="json",
            )

    @staticmethod
    def collect_responses(server, result_table, scraper_table):
        commands = [
            (
                (
                    "{command} {soy_table} {output_table} --server {server}"
                ).format(
                    command="{} {}".format(sys.executable, scraper_over_yt_script.__file__),
                    soy_table=scraper_table,
                    output_table=result_table,
                    server=server,
                ),
                "scraper_over_yt_script.py"
            ),
        ]
        for cmd, log_prefix in commands:
            logging.info("Start %s part of task with command %s", log_prefix, cmd)
            process.run_process(
                [cmd], work_dir="./", timeout=3600 * 3, shell=True, check=True,
                log_prefix=log_prefix, outputs_to_one_file=True, environment=os.environ.copy(),
            )

    @staticmethod
    def _get_queued_jobs(client, id):
        return list(client.lookup_rows(
            "//home/search-runtime/robot-scraperoveryt/scraper_over_yt/scheduler/operations_queue/queue",
            [{"id": id}],
            format="json",
        ))

    @staticmethod
    def _get_processed_jobs(client, id):
        return list(client.lookup_rows(
            "//home/search-runtime/robot-scraperoveryt/scraper_over_yt/scheduler/operations_queue/processed",
            [{"id": id}],
            format="json",
        ))

    @staticmethod
    def get_batch_status_via_api(server, guid):
        """
        Get status of soy batch and operation_id
        :return: (status, output_table)
        """
        api = SoyApi(server=server)
        status = api.status(guid)
        operation_status = status["simplified_operation_status"]

        return BATCH_STATUS_TYPES[operation_status], status

    @staticmethod
    def get_batch_status(server, guid):
        """
        Get status of soy batch and operation_id
        :return: (status, id)
        """
        client = ScraperOverYtWrapper.create_client(server, os.environ["YT_TOKEN"])

        queued_jobs = ScraperOverYtWrapper._get_queued_jobs(client, guid)
        if queued_jobs:
            return BatchStatus.QUEUEING, None

        processed_jobs = ScraperOverYtWrapper._get_processed_jobs(client, guid)
        if not processed_jobs:
            return BatchStatus.PROCESSING, None

        result_job = processed_jobs[0]
        operation_id = result_job["operation_id"]
        message = result_job.get("message") or result_job["error_message"]
        if not operation_id or message:
            logging.info("Where is error with scraper over YT: %s", message)
            return BatchStatus.FAILED, None

        scraper_table = (
            "//home/search-runtime/robot-scraperoveryt/"
            "scraper_over_yt/scheduler/operations_archive/{}/output_table".format(operation_id)
        )

        is_table_exist = client.exists(scraper_table)
        if not is_table_exist:
            return BatchStatus.FETCHING, operation_id

        return BatchStatus.COMPLETED, operation_id
