# -*- coding: utf-8 -*-

import json
import logging
import struct
from collections import defaultdict

from sandbox.sandboxsdk import process

COLLECTIONS = {
    "search": {
        "name": "search/",
        "parser": "search",
        "platform": "desktop",
    },
    "touchsearch": {
        "name": "search/touch/",
        "parser": "search",
        "platform": "touch",
        "user_agent": (
            "Mozilla/5.0 (Linux; U; Android 4.0.4; en-gb; GT-I9300 Build/IMM76D) "
            "AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.30"
        ),
    },
    "padsearch": {
        "name": "search/pad/",
        "parser": "search",
        "platform": "pad",
        "user_agent": (
            "Mozilla/5.0 (Linux; Android 4.1.1; Nexus 7 Build/JRO03D) "
            "AppleWebKit/535.19 (KHTML, like Gecko) Chrome/18.0.1025.166 Safari/535.19"
        ),
    },
    "images": {
        "name": "images/search",
        "parser": "search",
    },
    "video": {
        "name": "video/search",
        "parser": "video",
        "platform": "desktop",
    },
    "people": {
        "name": "people/search",
        "parser": "search",
        "platform": "desktop",
        "user_agent": ("Mozilla/5.0 (X11; Linux x86_64; rv:10.0) Gecko/20100101 Firefox/10.0")
    },
    "touchpeople": {
        "name": "people/search/touch",
        "parser": "search",
        "platform": "touch",
        "user_agent": (
            "Mozilla/5.0 (Linux; U; Android 4.0.4; en-gb; GT-I9300 Build/IMM76D) "
            "AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.30"
        ),
    }
}


def run_cmd(cmd, log_prefix):
    process.run_process(
        [cmd],
        work_dir="./",
        timeout=3600,
        shell=True,
        check=True,
        log_prefix=log_prefix,
        outputs_to_one_file=True,
    )


def create_table_and_sort(server, table, user, mr_yt_binary, file_with_data):
    logging.info("Save results into MR, server: %s, path: %s", server, table)

    cmd = "{mr_yt_binary} -write {table} -lenval -server {server} -subkey -opt user={user} < {data}".format(
        mr_yt_binary=mr_yt_binary,
        table=table,
        server=server,
        user=user,
        data=file_with_data,
    )
    run_cmd(cmd, "MR")

    cmd = "{mr_yt_binary} -server {server} -opt user={user} -sort {table}".format(
        mr_yt_binary=mr_yt_binary,
        table=table,
        server=server,
        user=user,
    )
    run_cmd(cmd, "MR")


def save_record(fout, key, subkey, value):
    int_struct = struct.Struct("i")
    for item in key, subkey, value:
        fout.write(int_struct.pack(len(item)))
        fout.write(item)


def convert_cgi_to_json(cgi):
    js = defaultdict(list)
    for (key, value) in [s.split('=', 1) for s in cgi.split('&') if s]:
        js[key].append(value)
    return js


class JsonFormatter:

    FirstKey = "tmpl_data"

    @staticmethod
    def DeepDict():
        return defaultdict(JsonFormatter.DeepDict)

    @staticmethod
    def ResizeArray(jsonRef, keyToArray, targetSize):
        if isinstance(jsonRef[keyToArray], list):
            jsonRef[keyToArray] += [JsonFormatter.DeepDict() for i in xrange(targetSize - len(jsonRef[keyToArray]))]
        else:
            jsonRef[keyToArray] = [JsonFormatter.DeepDict() for i in xrange(targetSize)]

    @staticmethod
    def JsonToStandardFormatRecursively(jsonRef, currentKey, joinedMultikey, value):
        if not joinedMultikey:
            jsonRef[currentKey] = value
            return

        if '.' in joinedMultikey:
            newKey, joinedMultikey = joinedMultikey.split('.', 1)
        else:
            newKey, joinedMultikey = joinedMultikey, ""

        if newKey.isdigit():
            newKey = int(newKey)
            JsonFormatter.ResizeArray(jsonRef, currentKey, newKey + 1)
            JsonFormatter.JsonToStandardFormatRecursively(jsonRef[currentKey], newKey, joinedMultikey, value)
        elif newKey == "*":
            if isinstance(value, list):
                JsonFormatter.ResizeArray(jsonRef, currentKey, len(value))
                realKeys = range(len(value))
            else:
                realKeys = value.keys()
            for realKey in realKeys:
                JsonFormatter.JsonToStandardFormatRecursively(
                    jsonRef[currentKey],
                    realKey,
                    joinedMultikey,
                    value[realKey],
                )
        else:
            JsonFormatter.JsonToStandardFormatRecursively(
                jsonRef[currentKey],
                newKey,
                joinedMultikey,
                value,
            )

    @staticmethod
    def JsonToStandardFormat(responseJsonDump):
        currentFormat = json.loads(responseJsonDump)
        if JsonFormatter.FirstKey in currentFormat.keys():
            return json.dumps({JsonFormatter.FirstKey: currentFormat[JsonFormatter.FirstKey]})

        standardFormat = {JsonFormatter.FirstKey: JsonFormatter.DeepDict()}
        for joinedMultikey in currentFormat.keys():
            JsonFormatter.JsonToStandardFormatRecursively(
                standardFormat,
                JsonFormatter.FirstKey,
                joinedMultikey,
                currentFormat[joinedMultikey],
            )
        return json.dumps(standardFormat)

    @staticmethod
    def CheckJson(response):
        try:
            json.loads(response)
        except Exception:
            # Next lines for cutting bad suffix that was generated by java report
            badJsonCommentSuffix = "<!--UNANSWERS-->"
            badSuffixPosition = response.find(badJsonCommentSuffix)
            if badSuffixPosition != -1 and response.count(badJsonCommentSuffix) == 2:
                response = response[:badSuffixPosition]

        # Try to load from json once again
        json.loads(response)
        return response
