import json
import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine import notify_helper as nh
from sandbox.projects.release_machine.tasks import base_task as rm_bt
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as ctn


RECIPIENT_KEY = 'recipient'
BODY_KEY = 'body'
SUBJECT_KEY = 'subject'


class SendEmail(rm_bt.BaseReleaseMachineTask):
    """
    A simple task to send multiple emails
    """
    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(rm_params.BaseReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)

        notifications = [
            sdk2.Notification(
                statuses=[ctt.Status.FAILURE, ctt.Status.Group.BREAK, ctt.Status.EXCEPTION],
                recipients=['ilyaturuntaev'],
                transport=ctn.Transport.TELEGRAM,
            )
        ]

        messages = sdk2.parameters.JSON(
            "A JSON of your messages",
            description=(
                'This should be in a form `{{"messages": [<m1>, <m2>, ...]}}`, where '
                '`<m1>`, `<m2>`, ... are objects with "recipient", "subject", "body" keys: '
                '`{{"{recipient}": "test@example.com", "{message_key}": "Hi there!"}}`'.format(
                    recipient=RECIPIENT_KEY,
                    message_key=BODY_KEY,
                )
            )
        )

        add_self_reference = sdk2.parameters.Bool("Add a link to this task at the bottom of each message", default=True)

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        pass

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        messages = self.Parameters.messages
        common_message_footer = self._get_message_footer()
        overall_success = True
        if not isinstance(messages, dict):
            try:
                messages = json.loads(messages)
            except ValueError:
                logging.error("Unable to parse messages parameter. You should check it")
                eh.fail("'messages' parameter is not a valid JSON")
        logging.debug('START message loop')
        for msg in messages.get('messages', []):
            if RECIPIENT_KEY not in msg:
                logging.info("Skipping item %s since no '%s' provided", msg, RECIPIENT_KEY)
                continue
            if BODY_KEY not in msg or not msg[BODY_KEY]:
                logging.info("Skipping item %s since no %s provided or it's empty", msg, BODY_KEY)
                continue
            logging.debug("Sending message to %s", msg[RECIPIENT_KEY])

            recipient = msg[RECIPIENT_KEY]
            subject = msg.get(SUBJECT_KEY, '')

            success, error_msg = nh.email2(
                self,
                recipients=recipient,
                subject=subject,
                body='\n\n'.join((msg[BODY_KEY], common_message_footer)),
            )

            if not success:
                self.set_info(
                    u"Unable to send message to {recipient} ({subj}) due to the following error: {error_msg}".format(
                        recipient=recipient,
                        subj=subject[:10],
                        error_msg=error_msg,
                    )
                )

            overall_success &= success

        logging.debug('END message loop')

        if not overall_success:
            eh.fail("Failing due to previous errors")

    def _get_message_footer(self):
        if not self.Parameters.add_self_reference:
            return ''
        return 'Sent by Release Machine Notification Service'
