import json
import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common.decorators import memoized_property as cached_property
from sandbox.projects.release_machine import input_params2 as rm_params
from sandbox.projects.release_machine import resources as rm_res
from sandbox.projects.release_machine.core import task_env
from sandbox.projects.release_machine.tasks import base_task as rm_bt
from sandbox.common import telegram
from sandbox.common import utils as common_utils
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as ctn


PARSE_MODE_HTML = "HTML"
PARSE_MODE_MD = "Markdown"

CHAT_ID_KEY = 'chat_id'
MESSAGE_KEY = 'message'
PIN_KEY = 'pin'
MENTION_KEY = 'mention'
ERROR_CODE_KEY = 'error_code'
ERROR_DESCRIPTION_KEY = 'description'

MAX_MESSAGE_LEN = 4096
MAX_MESSAGE_LEN_EXCEEDED_MESSAGE_TEMPLATE = (
    'The original message is too long. See the full message <a href="{url}">here</a>\n'
)
MESSAGE_HEAD_LEN = 1500
EXTREMELY_LONG_MESSAGE_FILENAME_TEMPLATE = 'message_{}.html'
CUT_HERE_MARKER = "=== CUT_HERE ==="


class SendTelegramMessages(rm_bt.BaseReleaseMachineTask):
    """
    A simple task to send Telegram messages

    The task takes messages configuration from the `messages` input parameter (JSON)
    and sends each of them to the appropriate recipients. Telegram bot token is
    received from Yandex Vault so you'll need to provide correct secret UUID and
    token name. Before the first run you'll also need to delegate your secret
    to sandbox by executing
    ```
    curl -d '{"secrets": [{"id": "sec-***"}]}' -H 'Authorization: OAuth ***' -H "Content-Type: application/json" \
    -X POST "https://sandbox.yandex-team.ru/api/v1.0/yav/tokens"
    ```
    See https://wiki.yandex-team.ru/sandbox/yav/ for more information
    """

    # See https://core.telegram.org/bots/api#html-style
    ALLOWED_TAGS = {'b', 'strong', 'i', 'em', 'u', 'ins', 's', 'strike', 'del', 'a', 'code', 'pre'}

    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(rm_params.BaseReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 600  # 10 min

        notifications = [
            sdk2.Notification(
                statuses=[ctt.Status.FAILURE, ctt.Status.Group.BREAK, ctt.Status.EXCEPTION],
                recipients=['ilyaturuntaev'],
                transport=ctn.Transport.TELEGRAM,
            )
        ]

        with sdk2.parameters.String(
                "Message parsing mode",
                default=PARSE_MODE_HTML,
                description=(
                    "How Telegram should treat your message: "
                    "HTML/Markdown (https://tlgrm.ru/docs/bots/api#sendmessage)"
                ),
        ) as parse_mode:
            rm_params.set_choices(parse_mode, (PARSE_MODE_HTML, PARSE_MODE_MD))

        messages = sdk2.parameters.JSON(
            "A JSON of your messages",
            description=(
                "This should be in a form `{{\"messages\": [<m1>, <m2>, ...]}}`, where "
                "`<m1>`, `<m2>`, ... are objects with \"chat_id\" and \"message\" keys: "
                "`{{\"{chat_id_key}\": 1235689061, \"{message_key}\": \"Hi there!\"}}`. "
                "Each <mi> object can optionally contain \"pin\" key. If pin is true "
                "then the message will be pinned".format(
                    chat_id_key=CHAT_ID_KEY,
                    message_key=MESSAGE_KEY,
                )
            )
        )

        token = sdk2.parameters.YavSecret(
            "YaV UUID for Telegram token",
            description="You'll need to delegate this secret to Sandbox first. "
                        "This should be done once per secret. "
                        "https://wiki.yandex-team.ru/sandbox/yav/",
        )
        yav_token_name = sdk2.parameters.String(
            "Telegram token name in YaV",
            description="The name of the token entry in the given secret",
        )

        add_self_reference = sdk2.parameters.Bool("Add a link to this task at the bottom of each message", default=True)

    class Context(rm_bt.BaseReleaseMachineTask.Context):
        succeeded_messages = []

    @cached_property
    def common_message_footer(self):
        return self._get_message_footer()

    def _get_token(self):
        """Obtain Telegram token from YaV"""
        logging.debug('Obtaining token from YaV...')
        return self.Parameters.token.data()[self.Parameters.yav_token_name]

    def _get_telegram_bot(self, token):
        logging.debug('Initializing Telegram bot...')
        return telegram.TelegramBot(token)

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        try:
            token = self._get_token()
        except Exception as e:
            eh.log_exception("Attempt to receive token from Yandex.Vault (yav) FAILED", e)
            eh.fail("Unable to obtain Telegram token")

        bot = self._get_telegram_bot(token)
        parse_mode = self.Parameters.parse_mode
        logging.debug('START message loop')
        messages = self.Parameters.messages
        succeeded_messages = set(self.Context.succeeded_messages)

        if not isinstance(messages, dict):
            try:
                messages = json.loads(messages)
            except ValueError:
                logging.error("Unable to parse messages parameter. You should check it")
                eh.fail("'messages' parameter is not a valid JSON")

        messages_list = messages.get('messages', [])

        for index, msg in enumerate(messages_list):

            if index in succeeded_messages:
                continue

            message_text = self._prepare_message(msg, index)
            pin = msg.get(PIN_KEY, False)

            if parse_mode == PARSE_MODE_HTML:
                message_text = self.sanitize_html_message(message_text)

            response = bot.send_message(
                chat_id=msg[CHAT_ID_KEY],
                text=message_text,
                parse_mode=parse_mode,
            )

            logging.debug("Response: %s", response)

            if ERROR_CODE_KEY in response:
                self.set_info('Message to {} with body {} failed. Description: {}'.format(
                    msg[CHAT_ID_KEY],
                    msg[MESSAGE_KEY],
                    response[ERROR_DESCRIPTION_KEY],
                ))
                continue

            succeeded_messages.add(index)

            if pin:

                try:
                    bot.pin_message(
                        chat_id=msg[CHAT_ID_KEY],
                        message_id=response["result"]["message_id"],
                    )
                except Exception:
                    logging.exception("Unable to pin message")

        if len(succeeded_messages) != len(messages_list):
            eh.check_failed("Not all messages was sent: {} success, {} failures".format(
                len(succeeded_messages),
                len(messages_list) - len(succeeded_messages),
            ))

        self.Context.succeeded_messages = list(succeeded_messages)
        logging.debug('END message loop')

    def _prepare_message(self, message_dict, message_index):

        if CHAT_ID_KEY not in message_dict:
            logging.info("Skipping item %s since no '%s' provided", message_dict, CHAT_ID_KEY)
            return

        if MESSAGE_KEY not in message_dict or not message_dict[MESSAGE_KEY]:
            logging.info("Skipping item %s since no %s provided or it's empty", message_dict, MESSAGE_KEY)
            return

        logging.debug("Preparing message for %s", message_dict[CHAT_ID_KEY])

        message_text = message_dict[MESSAGE_KEY] + self.common_message_footer

        mentions = message_dict.get(MENTION_KEY, [])

        if mentions:

            message_text = "{mentions}\n{text}".format(
                mentions=" ".join(["@{}".format(m) for m in mentions]),
                text=message_text,
            )

        if len(message_text) > MAX_MESSAGE_LEN or CUT_HERE_MARKER in message_text:
            url = self._create_resource_for_extremely_long_message(message_text, message_index)
            message_text = "{head}\n{note}{footer}".format(
                head=self.get_too_long_message_head(message=message_text),
                note=MAX_MESSAGE_LEN_EXCEEDED_MESSAGE_TEMPLATE.format(url=url),
                footer=self.common_message_footer,
            )

        return message_text

    @classmethod
    def sanitize_html_message(cls, message):

        from bs4 import BeautifulSoup

        logging.debug("Sanitizing %s", message)
        logging.debug("Allowed tags are: %s", cls.ALLOWED_TAGS)

        soup = BeautifulSoup(message, "html.parser")

        logging.debug("Original soup: %s", soup)

        for tag in soup.find_all(True):

            if tag.name.lower() not in cls.ALLOWED_TAGS:

                tag.extract()

        logging.debug("Result soup: %s", soup)

        return soup.renderContents()

    @classmethod
    def get_too_long_message_head(cls, message):

        from bs4 import BeautifulSoup

        logging.debug("Going to get head of %s", message)

        split_position = message.find(CUT_HERE_MARKER)

        if split_position == -1 or split_position > MAX_MESSAGE_LEN:
            split_position = MESSAGE_HEAD_LEN

        soup = BeautifulSoup("{}...".format(message[:split_position]), "html.parser")

        logging.debug("Soup: %s", soup)

        result = soup.renderContents()

        logging.debug("Result: %s", result)

        return result

    def _get_message_footer(self):
        if not self.Parameters.add_self_reference:
            return ''
        return '\nSent by SB:<a href="{url}">{name}</a>'.format(
            url=common_utils.get_task_link(self.id),
            name=self.id,
        )

    def _create_resource_for_extremely_long_message(self, message, message_index):
        message = message.replace("\n", "<br/>")
        message = '<p style="white-space: pre;">{}</p>'.format(message.replace(CUT_HERE_MARKER, "", 1))
        filename = EXTREMELY_LONG_MESSAGE_FILENAME_TEMPLATE.format(message_index)
        fu.write_file(filename, message)
        resource = rm_res.TELEGRAM_MESSAGE(self, "Extremely long message", filename)
        sdk2.ResourceData(resource).ready()
        return resource.http_proxy
