# -*- coding: utf-8 -*-
import json
import logging
import re

import sandbox.projects.common.file_utils as fu
import sandbox.projects.common.time_utils as tu
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.changelogs as ch
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.resources as rm_res
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sdk2 as sdk2
from sandbox.projects.common import binary_task


class SimpleChangelog(rm_bt.BaseReleaseMachineTask):
    class Requirements(task_env.TinyRequirements):
        disk_space = 5 * 1024  # 5 Gb

    class Parameters(rm_params.BaseReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        kill_timeout = 30 * 60  # 30 min
        first_revision = sdk2.parameters.Integer("First revision of changelog")
        baseline_path = sdk2.parameters.ArcadiaUrl("Path of baseline item", required=True)
        baseline_revision = sdk2.parameters.Integer("Revision of baseline item (should be in baseline_path)")
        candidate_path = sdk2.parameters.ArcadiaUrl("Path of release candidate item", required=True)
        candidate_revision = sdk2.parameters.Integer("Revision of release candidate item (should be in candidate_path)")
        use_filters = sdk2.parameters.Bool("Use filters", default=False)
        with use_filters.value[True]:
            paths_filter = sdk2.parameters.List(
                "Paths (starting from trunk) to use for filtering revisions (Ex: arcadia/search)"
            )
            marker_filter = sdk2.parameters.List("Commit message markers to use for filtering revisions")
            review_filter = sdk2.parameters.List("Review groups to use for filtering revisions")
            testenv_filter = sdk2.parameters.List("TE databases to use for filtering revisions")

        with sdk2.parameters.Output():
            changelog = sdk2.parameters.Resource("Changelog", resource_type=rm_res.SimpleChangelog)

    def on_enqueue(self):
        rm_bt.BaseReleaseMachineTask.on_enqueue(self)
        with self.memoize_stage.create_changelog_on_enqueue:
            self.Parameters.changelog = rm_res.SimpleChangelog(
                task=self,
                description="Changelog",
                path="changelog.json",
            )

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)
        changelog_master = ch.ChangeLogMaster(
            self.Parameters.first_revision,
            self.Parameters.baseline_path,
            self.Parameters.baseline_revision,
            self.Parameters.candidate_path,
            self.Parameters.candidate_revision,
            self.get_filters(),
        )
        changelog = [i.to_dict() for i in changelog_master.get_changelog()]
        logging.info("Got changelog:\n%s", json.dumps(changelog, indent=2, separators=(',', ': ')))
        full_changelog = {
            "first_revision": self.Parameters.first_revision,
            "baseline_path": self.Parameters.baseline_path,
            "baseline_revision": self.Parameters.baseline_revision,
            "candidate_path": self.Parameters.candidate_path,
            "candidate_revision": self.Parameters.candidate_revision,
            "changes": changelog,
            "creation_time": tu.date_ymdhm(),
        }
        fu.write_file(self.Parameters.changelog.path, json.dumps(full_changelog, indent=2, separators=(',', ': ')))

    def get_filters(self):
        if not self.Parameters.use_filters:
            return
        filters = []
        if self.Parameters.paths_filter:
            filters.append(ch.PathsFilter([(1, path) for path in self.Parameters.paths_filter]))
        if self.Parameters.marker_filter:
            filters.append(ch.MarkerFilter([(1, re.compile(marker)) for marker in self.Parameters.marker_filter]))
        if self.Parameters.review_filter:
            filters.append(ch.ReviewFilter(
                self.Parameters.review_filter,
                self.Parameters.first_revision,
                self.Parameters.candidate_revision
            ))
        if self.Parameters.testenv_filter:
            filters.append(ch.TestenvFilter(self.Parameters.testenv_filter))
        logging.info("Got %s filters: %s", len(filters), filters)
        return filters
