# -*- coding: utf-8 -*-
import datetime
import logging
import os

import sandbox.projects.common.file_utils as file_utils
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.metrics_helper as metrics_helper
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.tasks.base_task as rm_bt
import sandbox.sandboxsdk.environments as env
import sandbox.sandboxsdk.process as process
import sandbox.sdk2 as sdk2
from sandbox.projects.common import binary_task

import get_basket


@rm_notify.notify2(people=["rm_maintainers"])
class UpdateMetricsBasket(rm_bt.BaseReleaseMachineTask):
    """
    Collect user search queries for specified days count and specified wizards.
    Update appropriate wizard's query basket.
    """

    _result_path = "basket.json"

    class Parameters(rm_params.BaseReleaseMachineParameters):
        _lbrp = binary_task.binary_release_parameters(stable=True)
        unified_basket_id = sdk2.parameters.Integer("Unified basket for all wizards", default=0)
        wizard_baskets = sdk2.parameters.Dict("Wizard name: basket id")
        days_count = sdk2.parameters.Integer("Collect user queries for specified days", default=14)

    class Requirements(task_env.TinyRequirements):
        pass

    class Context(sdk2.Task.Context):
        fail_on_any_error = True

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        with env.VirtualEnvironment() as venv:
            logging.info("Installing yql")
            # Collect queries for new basket
            env.PipEnvironment("yql", version="1.2.37", venv=venv, use_wheel=True).prepare()
            token = sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.YQL_TOKEN_NAME)
            os.environ["YQL_TOKEN"] = token

            metrics_api = metrics_helper.MetricsApi(token=token)

            self.failed_wizards = []

            if self.Parameters.unified_basket_id:
                all_json_queries = []
                for wizard_name in self._get_all_wizards():
                    wizard_name = 'slices-{}'.format(wizard_name)
                    json_queries = self._process_wizard(venv, token, wizard_name)
                    if not self._check(wizard_name, json_queries):
                        continue

                    all_json_queries.extend(json_queries)

                # Update unified basket
                metrics_api.update_basket(self.Parameters.unified_basket_id, all_json_queries)
            else:
                for wizard_name in self.Parameters.wizard_baskets:
                    basket_id = self.Parameters.wizard_baskets.get(wizard_name)
                    json_queries = self._process_wizard(venv, token, wizard_name)
                    if not self._check(wizard_name, json_queries):
                        continue

                    # Update specified metrics basket
                    metrics_api.update_basket(basket_id, json_queries)

            if self.failed_wizards:
                message = 'Failed wizards: {}'.format(', '.join(self.failed_wizards))
                self.set_info(message)
                eh.check_failed(message)

    def _check(self, wizard_name, json_queries):
        if not json_queries:
            self.failed_wizards.append(wizard_name)
            logging.error('Failed wizard: %s', wizard_name)
            return False

        return True

    def _process_wizard(self, venv, token, wizard_name):
        """
        :return json [list] with queries
        """
        logging.info('Processing wizard %s', wizard_name)
        basket_path = "basket_{}.json".format(wizard_name)

        # Collect some queries for basket

        # user sessions should be ready, see explanation
        # in https://st.yandex-team.ru/RMDEV-1548#5eb29d482fdfe65e619a7e51
        now = datetime.datetime.now()
        day_shift = 1 if now.hour < 12 else 2
        end_date = (now - datetime.timedelta(days=day_shift)).strftime("%Y%m%d")

        if self.is_binary_run:
            get_basket.update_basket(
                wizard_type=wizard_name,
                end_date=end_date,
                days=str(self.Parameters.days_count),
                out=basket_path,
            )
        else:
            cmd = [
                venv.executable, os.path.join(os.path.dirname(__file__), "get_basket.py"),
                "-w", wizard_name,  # Test it on news basket
                "--end_date", end_date,
                "--days", str(self.Parameters.days_count),
                "--out", basket_path,
            ]

            logging.info("Process %s wizard with params: %s", wizard_name, cmd)
            process.run_process(cmd, log_prefix='result_releases', outputs_to_one_file=False)

        return file_utils.json_load(basket_path)

    def _get_all_wizards(self):
        # FIXME(mvel): spike
        return [
            'BIATHLON',
            # 'COLLECTIONS_BOARD', failed (deleted)
            'DRUGS',
            'ENTITY_SEARCH',
            'GAMES',
            'GAMES_SINGLE',
            'GEOV',
            'NEWS_WIZARD',
            # 'PEOPLEWIZ', failed (deleted)
            'RABOTA',
            'SHINY_DISCOVERY',
            'STATIC_FACT',
            # 'TURBO_SNIPPET',
            'TUTOR',
            'TV_TRANSLATION',
            'VERTIS_GENERAL',
            'VHS_WIZARD',
            'VIDEOWIZ',
            'WEB_Q',
            'WEB_QUESTION',
            'WEB_WEATHER',
            'WEB_ZNATOKI',
            'WIZAUTO',
            'WIZDISTRICT_GEO',
            'WIZHEALTHENCYCLOPEDIA',
            'WIZIMAGES',
            'WIZMAPS',
            'WIZMARKET',
            'WIZMUSIC',
            'WIZREALTY',
            'WIZTRANSLATE',
            'WIZUSLUGI',
            'WIZWEATHER',
            'WIZZNATOKIQUESTION',
            'YABS_PROXY_WIZ_EXP',
            'YA_TALENTS',
            'Y_CONTENTGEN',
            'Y_DISTRICT',
            'Y_EDADEAL',
            'Y_GAMES',
            'Y_MAG_AUTO',
            'Y_QUESTION',
            'Y_TALENTS',
            'Y_TUTOR_GDZ',
            'Y_TUTOR',
            'Y_TUTOR_PDF',
            'Y_ZEN',
            'Y_ZNATOKI',
            'ZEN',
            'ZNATOKI_MERGED',
        ]
