# -*- coding: utf-8 -*-

import argparse
import json
import os
import logging
from datetime import datetime
from datetime import timedelta


def standard_log(msg, *args, **kwargs):
    logging.debug(msg, *args, **kwargs)


def print_log(msg, *args, **kwargs):
    print(msg % args)


def update_basket(wizard_type, end_date, days, out, log=standard_log):
    from yql.api.v1.client import YqlClient

    days = int(days)

    log("update_basket called with params: %s, %s, %s, %s", wizard_type, end_date, days, out)

    token = os.environ.get("YQL_TOKEN")
    if not token:
        log("Token does not set")
    client = YqlClient(db="hahn", token=token)
    start_date = (datetime.strptime(end_date, "%Y%m%d") - timedelta(days=days)).strftime("%Y%m%d")

    if 'slices-' in wizard_type:
        # intent
        intent = wizard_type.replace('slices-', 'intent-')
        query = ("""
            $m = ($s, $intent) -> {{
                RETURN $s['v7'][$intent].win - $s['v7'][$intent].loss;
            }};

            SELECT query, 1
            FROM RANGE(`home/blender/surplus_squeeze/production`, `{start_date}`, `{end_date}`)
            WHERE service == 'www.yandex' AND $m(vsurplus, '{intent}') > 0
            LIMIT {limit};
        """.format(
            start_date=start_date,
            end_date=end_date,
            intent=intent,
            limit=10,
        ))
    elif wizard_type == "misprint":
        end_date = datetime.strptime(end_date, "%Y%m%d").strftime("%Y-%m-%d")
        query = (
            "select q, c as total "
            "from `home/antispam/export/misprint_queries/{end_date}` "
            "order by total desc "
            "limit 50"
        ).format(end_date=end_date)
    else:
        query = (
            "select q, count(*) as total "
            "from range (`home/blender/surplus_squeeze/production`, `{start_date}`, `{end_date}`) where "
            "wizard['{wizard_type}'].clicked and (ui == 'desktop' or ui == 'touch') and locale == 'ru' "
            "group by query as q "
            "order by total desc "
            "limit 50"
        ).format(
            start_date=start_date,
            end_date=end_date,
            wizard_type=wizard_type,
        )

    log("Query: %s", query)

    request = client.query(query, syntax_version=1)
    request.run()

    json_basket = []
    for table in request.get_results():
        for user_query, _ in table.rows:
            try:
                json_basket += [
                    {
                        "text": user_query.encode("utf-8"),
                        "regionId": 213,
                        "device": "DESKTOP",
                    },
                    {
                        "text": user_query.encode("utf-8"),
                        "regionId": 213,
                        "device": "TOUCH",
                    },
                ]

            except UnicodeDecodeError as ude:
                log("Unable to decode query '%s'", query)
                raise ude

    with open(out, "wb") as fd:
        fd.write(json.dumps(json_basket, indent=4))


def main():

    args = argparse.ArgumentParser(description="Get user queries for specified wizard")
    args.add_argument("--wizard_type", "-w", help="Wizard type", required=True)
    args.add_argument("--end_date", "-e", help="Until that date queries will be collected (format YYYYMMDD)")
    args.add_argument("--days", "-d", type=int, help="Collect queries for specified days number", default=1)
    args.add_argument("--out", "-o", help="Output path")

    opts = args.parse_args()

    update_basket(opts.wizard_type, opts.end_date, opts.days, opts.out, log=print_log)


if __name__ == '__main__':
    main()
