import base64
import logging
import time
import sys
import os

import sandbox.common.types.misc as misc
import sandbox.sdk2 as sdk2
import sandbox.projects.common.binary_task as binary_task
import sandbox.projects.common.search.bugbanner2 as bb2
import sandbox.projects.common.time_utils as tu
import sandbox.projects.common.decorators as decorators
import sandbox.projects.common.error_handlers as eh
import sandbox.projects.common.gsid_parser as gsid_parser
import sandbox.projects.common.link_builder as lb
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.events_helper as events_helper
import sandbox.projects.release_machine.helpers.solomon_helper as solomon_helper
import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.security as rm_sec
import sandbox.projects.release_machine.events as release_machine_events


class BaseReleaseMachineTask(bb2.BugBannerTask, binary_task.LastBinaryTaskRelease):

    class Requirements(task_env.TinyRequirements):
        # RMDEV-1971
        pass

    class Context(sdk2.Task.Context):
        rm_proto_event = ""
        event_ready = False
        rm_ui_link = ""

    @decorators.memoized_property
    def component_name(self):
        return getattr(self.Parameters, "component_name", None)

    @decorators.memoized_property
    def rm_component_link(self):
        return os.path.join(rm_const.Urls.RM_URL, "component", self.component_name)

    @decorators.memoized_property
    def testenv_database(self):
        return gsid_parser.get_testenv_database_from_gsid(self.gsid)

    @decorators.memoized_property
    def testenv_launch_revision(self):
        return gsid_parser.get_svn_revision_from_gsid(self.gsid)

    @decorators.memoized_property
    def rm_ui_link(self):
        logging.info("Building RM UI link")

        if not self.component_name:
            logging.info("No component name found - cannot build RM UI link")
            return ""

        if not self.is_binary_run:
            logging.info("Not a binary run - fallback to component link (no scope)")
            return self.rm_component_link

        if self.is_ci_launch:
            return os.path.join(
                rm_const.Urls.RM_URL,
                "component",
                self.component_name,
                "manage2?action_id={action_id}&scope={scope}&tag={tag}".format(
                    action_id=self.ci_context.config_info.id,
                    scope=self.ci_context.version_info.major or "0",
                    tag=self.ci_context.version_info.minor or "0",
                ),
            )

        if not self.gsid and not self.is_ci_launch:
            logging.info("Not a TE or CI launch - cannot build RM UI link")
            return self.rm_component_link

        from release_machine.release_machine.proto.structures import message_pb2
        from release_machine.release_machine.services.release_engine.services.Model import ModelClient
        import grpc

        try:

            mc = ModelClient.from_address(rm_const.Urls.RM_HOST)

            request = message_pb2.GetScopeInfoByTestenvDatabaseRequest(
                testenv_database=self.testenv_database,
                revision=self.testenv_launch_revision,
            )

            logging.info("Requesting get_scope_info_by_testenv_database with %s", request)

            response = mc.get_scope_info_by_testenv_database(
                request=request,
            )

            logging.info("get_scope_info_by_testenv_database response: %s", response)

            return response.release_machine_url

        except grpc.RpcError:
            logging.exception("Unable to build the full link (RM responded with error)")

        return self.rm_component_link

    @decorators.memoized_property
    def is_ci_launch(self):
        return bool(release_machine_events.get_ci_context_task_context_value(self))

    @decorators.memoized_property
    def ci_context(self):
        return self._get_rm_proto_event_ci_job_context()

    def add_info_to_startrek_full(self, st_helper_instance, release_number, message, c_info):
        issue = st_helper_instance.comment(
            release_number,
            message,
            c_info,
        )
        self.set_info(
            "Comment: {comment} added in {issue}".format(comment=message, issue=lb.st_link(issue.key)),
            do_escape=False,
        )

    def log_exception(self, message, exc):
        self.set_info("{}. See log for more info".format(message))
        eh.log_exception(message, exc, task=self)

    def add_info_to_startrek(self, st_helper_instance, message):
        c_info = None
        release_number = None
        try:
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            release_number = self.Parameters.release_number
        except Exception as exc:
            self.log_exception("Unable to add info to startrek", exc)
        if c_info and release_number:
            self.add_info_to_startrek_full(st_helper_instance, release_number, message, c_info)

    def set_proto_event(self, event_message):
        if event_message:
            self.Context.rm_proto_event = self._serialize_event(event_message)
            self.Context.event_ready = True
            self.Context.save()

    def _serialize_event(self, event_message):
        serialized = base64.b64encode(event_message.SerializeToString())
        if sys.version_info.major == 3:
            return serialized.decode("utf-8")
        return serialized

    def _get_rm_proto_event_hash_items(self, event_time_utc_iso, status=None):
        """Return a tuple of items that should be used to hash event."""
        return str(self.id), status or self.status, event_time_utc_iso

    def _get_rm_proto_event_general_data(self, event_time_utc_iso, status=None):
        """Get general_data for RM proto event."""
        return release_machine_events.get_task_event_general_data(
            task_obj=self,
            hash_items=self._get_rm_proto_event_hash_items(
                event_time_utc_iso=event_time_utc_iso,
                status=status,
            ),
        )

    def _get_rm_proto_event_ci_job_context(self):
        return release_machine_events.get_task_event_ci_job_context(task_obj=self)

    def _get_rm_proto_event_task_data(self, event_time_utc_iso, status=None):
        """Get RM proto event task data."""
        return release_machine_events.get_task_event_task_data(
            task_obj=self,
            event_time_utc_iso=event_time_utc_iso,
            status=status,
        )

    def _get_rm_proto_event_specific_data(self, rm_proto_events, event_time_utc_iso, status=None):
        """
        Get RM proto event specific data.

        Should be defined in subclasses!
        :return: dict <spec_data_key> => <event specific data> for event specific data oneof field
        """
        return {}

    def _get_rm_proto_event(self, status=None):
        """Build and return RM proto event."""
        logging.debug("Building new event proto data")

        from release_machine.common_proto import events_pb2 as rm_proto_events

        event_time_utc_iso = tu.datetime_utc_iso()

        event_specific_data = self._get_rm_proto_event_specific_data(
            rm_proto_events,
            event_time_utc_iso=event_time_utc_iso,
            status=status,
        )
        if not event_specific_data:
            logging.warning("Event specific data not found! Skip this proto event sending.")
            return

        return rm_proto_events.EventData(
            general_data=self._get_rm_proto_event_general_data(
                event_time_utc_iso=event_time_utc_iso,
                status=status,
            ),
            task_data=self._get_rm_proto_event_task_data(
                event_time_utc_iso=event_time_utc_iso,
                status=status,
            ),
            ci_job_context=self.ci_context,
            **event_specific_data  # noqa C815
        )

    def _load_rm_proto_event_from_context(self, status=None):
        logging.debug("Loading event data from context")
        if not self.Context.rm_proto_event:
            return

        from google.protobuf.message import DecodeError

        try:
            return events_helper.parse_rm_proto_event_from_b64_encoded_str(self.Context.rm_proto_event)
        except (TypeError, base64.binascii.Error) as e:
            self.log_exception(
                "Unable to parse '{}'. "
                "It is possible that it has been improperly encoded".format(self.Context.rm_proto_event),
                e,
            )
        except DecodeError as e:
            self.log_exception(
                "Unable to parse event data from string '{}'. "
                "It seems that the message has been encoded using another scheme".format(self.Context.rm_proto_event),
                e,
            )

    def store_rm_proto_event(self, status=None):
        event_message = self._get_rm_proto_event(status=status)
        self.set_proto_event(event_message)
        return event_message

    def send_rm_proto_event(self, status=None, load_from_context=False):
        """Send events to release engine."""
        logging.info("Going to send proto event (load_from_context = %s)", load_from_context)

        if not self.is_binary_run:
            logging.warning("Cannot send proto event since task is not run in binary mode")
            return

        try:
            if not self.Parameters.component_name:
                logging.warning("No component_name provided so no event is going to be sent. Aborting")
                return
        except AttributeError as ae:
            self.log_exception(
                "This task inherits BaseReleaseMachineTask but does not define a 'component_name' (input parameter). "
                "No events are going to be sent and some other stuff may work improperly",
                ae,
            )

        try:

            if load_from_context:
                event_message = self._load_rm_proto_event_from_context(status=status)
                logging.debug("Event data loaded")
            else:
                event_message = self.store_rm_proto_event(status=status)
                logging.debug("Event data stored")

            if event_message is not None:
                events_helper.EventHelper().post_grpc_events((event_message,))
            else:
                logging.warning("rm_proto_event is empty. Nothing to send!")

        except Exception as e:
            self.log_exception("Unable to send proto event", e)

    def get_job_name_from_gsid(self):
        return gsid_parser.get_job_name_from_gsid(self.gsid)

    def get_job_name_from_ci_context(self):
        ci_context = self.ci_context

        if not ci_context:
            return ""

        return ci_context.job_instance_id.job_id

    @property
    def gsid(self):
        return self.Context.__values__.get(rm_const.GSID_CONTEXT_KEY, "")

    @property
    def job_name(self):
        ci_job_name = self.get_job_name_from_ci_context()

        if ci_job_name:
            return ci_job_name

        return self.get_job_name_from_gsid()

    @property
    def is_binary_run(self):
        return self.Parameters.binary_executor_release_type != 'none'

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        bb2.BugBannerTask.on_save(self)

    def on_wait(self, prev_status, status):
        logging.debug("Update and send event before changing status %s -> %s", prev_status, status)
        self.send_rm_proto_event(status=status)
        bb2.BugBannerTask.on_wait(self, prev_status, status)

    def on_enqueue(self):
        bb2.BugBannerTask.on_enqueue(self)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        bb2.BugBannerTask.on_execute(self)

        self.add_bugbanner(bb2.Banners.ReleaseMachine)
        if self.Context.task_start_time == misc.NotExists:
            self.Context.task_start_time = int(time.time())
            self.Context.save()

        self.add_rm_ui_link()

    def on_success(self, prev_status):
        bb2.BugBannerTask.on_success(self, prev_status)

        task_finish_time = int(time.time())
        executing_time = task_finish_time - self.Context.task_start_time
        solomon_api = solomon_helper.SolomonApi(rm_sec.get_rm_token(self))
        response = solomon_api.post_task_exec_time_to_solomon(
            executing_time,
            str(self.type),
            getattr(self.Parameters, "component_name", None),
        )
        logging.debug("Got response while posting to Solomon %s", response)

    def on_finish(self, prev_status, status):
        try:
            self.send_rm_proto_event(status=status)
        except Exception as exc:
            self.log_exception("Event sending FAILED!", exc)
        bb2.BugBannerTask.on_finish(self, prev_status, status)

    def add_rm_ui_link(self):

        try:

            link = self.Context.rm_ui_link

            if link:
                logging.info("RM UI link already set")
                return

            link = self.rm_ui_link

            if not link:
                logging.info("RM UI link is empty => no RM UI banner is going to be added")
                return

            self.Context.rm_ui_link = link
            self.Context.save()

            self.set_info(
                '<a href="{link}">{component_name} (RM UI)</a>'.format(
                    link=link,
                    component_name=self.component_name,
                ),
                do_escape=False,
            )
        except Exception as exc:
            self.log_exception("Cannot set RM UI link", exc)


class ComponentErrorReporter(object):
    """Mixin for error reporting into task info by component."""

    def _report_error(self, component_name, message=None, exception=None):
        if exception is not None:
            eh.log_exception('{} `{}'.format(message, component_name), exception)
        else:
            logging.error('ERROR processing component `%s`: %s', component_name, message)
        self.set_info(
            '<b color="red">ERROR</b> [{}]: {}: {}'.format(component_name, message, exception),
            do_escape=False,
        )

    def _report_success(self, component_name, message):
        self.set_info(
            '<b color="#007f0000">OK</b> [{}]: {}'.format(component_name, message),
            do_escape=False,
        )
