# -*- coding: utf-8 -*-
from __future__ import unicode_literals


import logging
import io
import os
import time
import json
import datetime
import six
from collections import Iterable
from difflib import HtmlDiff

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common.sdk_compat import task_helper
from sandbox.sandboxsdk.channel import channel


ARC_PATCHES_DIR = "arcadia:/arc/trunk/arcadia/search/priemka/yappy/data/patches"
MAX_WAIT_NUM = 5
WAIT_MULTIPLIER = 100
DATETIME_KEY = "datetime"


def cmp_beta_infos(info_before, info_after):
    # todo: take into account only relevant changes
    if info_before != info_after:
        diff_filename = "diff_configuration_beta.html"
        fu.write_file(
            diff_filename,
            HtmlDiff(tabsize=4).make_file(
                json.dumps(info_before, indent=2).split("\n"),
                json.dumps(info_after, indent=2).split("\n")
            )
        )
        res = channel.task.create_resource(
            diff_filename, diff_filename, resource_types.OTHER_RESOURCE,
        )
        return res.id


def get_betas_to_stop(existing_prev_betas, work_betas_num):
    stop_betas = None
    if not work_betas_num:
        # stop all prev betas
        stop_betas = existing_prev_betas
    elif work_betas_num == 1:
        # left only baseline beta
        stop_betas = existing_prev_betas[1:]
    elif work_betas_num > 1:
        # left baseline beta and some last betas
        stop_betas = existing_prev_betas[1:1 - work_betas_num]
    else:
        eh.check_failed("Working beta nums should be > 0. Current num = {}".format(work_betas_num))
    logging.info("Betas expected to be stopped: %s", stop_betas)
    return stop_betas


def write_single_patch(patch_name, data, user, ssh_key):
    """
    Checkout yappy patches and change one of them.

    :param patch_name: path to patch, e. g. src-setup/16-1.yaml
    :param data: patch content as dict, it would be converted to yaml
    :param user: ssh_key owner
    :param ssh_key: sandboxsdk.ssh.Key to commit data
    """
    local_yappy_path = os.path.abspath("patches")
    sdk2.svn.Arcadia.checkout(ARC_PATCHES_DIR, local_yappy_path)
    patch_path = os.path.join(local_yappy_path, patch_name)
    new_patch = not os.path.exists(patch_path)
    with io.open(patch_path, "w") as yaml_file:
        import yaml
        yaml.dump(data, yaml_file, default_flow_style=False, allow_unicode=True)
    if new_patch:
        sdk2.svn.Svn.add(patch_path)

    with ssh_key:
        sdk2.svn.Arcadia.commit(patch_path, "Add new patch for component: {}".format(patch_name), user=user)


def wait_consistency(task, api, beta_names, beta_statuses_field=None):
    """
    Waits for the given betas consistency
    :param task: Sandbox task object
    :param api: Beta api object
    :param beta_names: list of beta names
    :param beta_statuses_field: name of the task context field in which beta statuses should be stored
    """
    if isinstance(beta_names, (six.text_type, six.binary_type)):
        beta_names = {beta_names}
    elif isinstance(beta_names, Iterable) and not isinstance(beta_names, set):
        beta_names = set(beta_names)
    curr_wait_num = task_helper.input_or_ctx_field(task, "wait_num", 0)
    beta_conf_revision = task_helper.input_or_ctx_field(task, "beta_conf_revision")
    max_wait_num = task_helper.input_or_ctx_field(task, "max_wait_num", MAX_WAIT_NUM)
    logging.info("Waiting for beta consistency for the %s time", curr_wait_num)
    if not curr_wait_num:
        time.sleep(300)  # Wait for applying patch
    beta_consistency = {}
    beta_none = []
    current_datetime = datetime.datetime.now().strftime("%b %d %Y %H:%M:%S")
    beta_consistency[DATETIME_KEY] = current_datetime
    for beta_name in beta_names:
        beta_consistency[beta_name] = get_beta_consistency(api, beta_conf_revision, beta_name)
        logging.debug("Beta %s is %s", beta_name, 'CONSISTENT' if beta_consistency[beta_name] else 'INCONSISTENT')
        if beta_consistency[beta_name] is None:
            beta_none.append(beta_name)
    if beta_statuses_field:
        status = task_helper.ctx_field(task, beta_statuses_field)
        status.append(beta_consistency)
        logging.debug("Status is %s", status)
        task_helper.ctx_field_set(task, beta_statuses_field, status)
    if beta_none:
        return False, "Betas {} don't exist in Yappy!".format(', '.join(beta_none))
    if all(beta_consistency.values()):
        task_helper.ctx_field_set(task, "wait_num", 0)
        return True, "Betas '{}' are consistent now".format(', '.join(beta_names))
    if curr_wait_num < max_wait_num:
        task_helper.ctx_field_set(task, "wait_num", curr_wait_num + 1)
        wait_num = task_helper.input_or_ctx_field(task, "wait_num")
        logging.debug("current wait_num is %s", wait_num)
        wait_sec = wait_num * getattr(task, "_WAIT_MULTIPLIER", WAIT_MULTIPLIER)
        logging.info("Wait %s seconds for consistency", wait_sec)
        if task_helper.is_sdk2(task):
            raise sdk2.WaitTime(wait_sec)
        task.wait_time(wait_sec)
    task_helper.ctx_field_set(task, "wait_num", 0)
    time_now = time.time()
    hour_seconds = datetime.timedelta(hours=1).total_seconds()
    time_hour_ago = time_now - hour_seconds
    time_hour_after = time_now + hour_seconds
    error_message = []
    for beta_name in beta_consistency:
        if not beta_consistency[beta_name]:
            error_message.append("Consistency check for '{beta_names}' timed out! Check out {beta_yasm_link}".format(
                beta_names=lb.yappy_beta_link(beta_name),
                beta_yasm_link=lb.yappy_beta_status_chart_link(
                    beta_name=beta_name,
                    link_name='{} consistency chart'.format(beta_name),
                    time_from=int(time_hour_ago * 1000),
                    time_to=int(time_hour_after * 1000),
                ),
            ))
    error_message.append(
        "Please make yappy beta consistent manually and rerun this task: {}".format(lb.task_wiki_link(task.id))
    )
    return False, "\n".join(error_message)


def get_sample_beta(c_info, task, api):
    """
    Return yappy beta, branch_num and tag_num from which beta was created
    :param c_info: component info
    :param task: task instance
    :param api: BetaApi instance
    :return: (beta, branch_num, tag_num)
    """
    logging.info("Try to get default sample beta with last released binary for: %s", c_info.name)
    last_released_res = next(c_info.get_last_release())
    branch_n, tag_n = last_released_res.major_release, last_released_res.minor_release
    logging.info("Found branch_num %s and tag_num %s from context", branch_n, tag_n)
    sample_beta = ''
    if branch_n and tag_n:
        for t_n in reversed(range(1, int(tag_n) + 1)):
            # Beta could not exist for released beta, so try to find the closest one
            sample_beta = c_info.yappy_cfg.get_beta_name(
                beta_conf_type=task.Parameters.beta_conf_type,
                patch_name="{}-{}".format(branch_n, t_n),
            )

            if api.yappy.beta_exists_no_defaults(sample_beta):
                logging.info("Default sample beta found: '%s'", sample_beta)
                return sample_beta, branch_n, t_n
    eh.fail("Can't find default sample beta: {}".format(sample_beta if sample_beta else "Unknown"))


def get_beta_consistency(api, beta_conf_revision, beta_name):
    try:
        return api.is_beta_consistent(beta_name, beta_conf_revision)
    except Exception as exc:
        raise errors.TemporaryError("Unable to get beta consistency: %s", exc)
