# -*- coding: utf-8 -*-

import logging
import os.path

import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.merge_helper as merge_helper
from sandbox.common.types import client
from sandbox import sdk2
from sandbox.sandboxsdk import svn as sandbox_svn

from sandbox.projects.common import error_handlers as eh
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.helpers import startrek_helper
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.components.configs.userfeat import UserfeatCfg


def collect_merges(branch_path, trunk_path):
    """
    :param branch_path: full path to the branch
    :param trunk_path: full path to the trunk
    :return: tuple of
             - trunk revision preceding the fork of this branch
             - list of trunk commits merged since the fork
    """
    cur_branch = branch_path
    cur_revision = "HEAD"
    merges = []
    prefix = "arcadia:/arc"
    while cur_branch != trunk_path:
        eh.ensure(cur_branch.startswith(prefix), "Branch path has to be full")
        commits = sandbox_svn.Arcadia.log(
            cur_branch,
            revision_from=cur_revision,
            revision_to="1",
            limit=10000,
            stop_on_copy=True,
            track_copy=True
        )
        eh.ensure(len(commits) > 0, "Branch needs to have commit(s)")
        eh.ensure("copies" in commits[-1], "First commit in branch contains copying")
        logging.info("Retrieved '%s' history of %d commit(s)", cur_branch, len(commits))
        for commit in commits:
            merges += merge_helper.get_merges_from_commit_message(commit["msg"])
        copies = commits[-1]["copies"]
        cur_branch_relpath = cur_branch[len(prefix):]
        parent_branch = None
        parent_revision = None
        for from_path, from_rev, path in copies:
            logging.info("copy: %s:%s -> %s",
                         from_path, from_rev, path)
            if path == cur_branch_relpath:
                parent_branch = prefix + from_path
                parent_revision = from_rev
                break
        eh.ensure(parent_branch and parent_revision,
                     "Branch {} must have parent".format(cur_branch))
        cur_branch = parent_branch
        cur_revision = parent_revision
    return int(cur_revision), merges


class NeedToBackport(sdk2.Task):
    """
    Given old and new stable branches, the task finds trunk commits,
    merged into the old branch via usual methods (Subversion hook or MERGE_TO_STABLE task),
    and not present in the new branch.

    The findings are optionally posted to a Startrek issue.

    For instance, here the task will find that commit C probably needs to be backported to new stable:

     +----A--------B-----C---> old stable branch
     |    ^        ^     ^
    -+----A---+----B-----C---> trunk
              |    v
              +----B---------> new stable branch
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 2048
        cores = 1
        client_tags = (~client.Tag.LINUX_LUCID &
                       (client.Tag.LINUX_PRECISE | client.Tag.LINUX_TRUSTY | client.Tag.LINUX_XENIAL))
        environments = [task_env.TaskRequirements.startrek_client]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup("RM component name") as component_name:
            component_name.values[UserfeatCfg.name] = component_name.Value(value=UserfeatCfg.name)
            # Add more supported components here

        target_branch_path = sdk2.parameters.String(
            label="Target branch path",
            required=True,
            description="New stable branch path, e.g. 'arcadia:/arc/branches/userfeat/stable-12'"
        )
        enable_post_to_st = sdk2.parameters.Bool(
            label="Enable to post to Startrek issue",
            default_value=True
        )

    class Context(sdk2.Task.Context):
        release_number = -1

    def on_execute(self):
        c_info = rmc.COMPONENTS[self.Parameters.component_name]()
        new_branch = self.Parameters.target_branch_path
        release_number = c_info.get_branch_id_from_path(new_branch)
        self.Context.release_number = release_number

        if release_number <= 1:
            logging.info("release_number is %d, there are no preceding branches to port commits from",
                         release_number)
            return

        trunk = os.path.join(c_info.svn_cfg__repo_base_url, "trunk")
        old_branch = c_info.full_branch_path(release_number - 1)
        logging.info("trunk %s, old_branch %s, new_branch %s",
                     trunk, old_branch, new_branch)

        old_start, merged_to_old = collect_merges(old_branch, trunk)
        new_start, merged_to_new = collect_merges(new_branch, trunk)

        merged_to_old = set(merged_to_old)
        merged_to_new = set(merged_to_new)

        logging.info("old: start %d, merged %s", old_start, str(merged_to_old))
        logging.info("new: start %d, merged %s", new_start, str(merged_to_new))

        not_merged_to_new = []
        for commit in merged_to_old.difference(merged_to_new):
            if commit > new_start:
                not_merged_to_new.append(str(commit))

        if len(not_merged_to_new) == 0:
            logging.info("not merged commits are not found")
            return

        summary = (
            "IMPORTANT!\n"
            "The following trunk commits are merged to previous stable branch, "
            "but not present in new stable branch and might need to be merged:\n\n{}"
        ).format(",".join(sorted(not_merged_to_new)))

        logging.info("Summary: %s", summary)

        if self.Parameters.enable_post_to_st:
            self.post_to_st_issue(c_info, summary)

    def post_to_st_issue(self, c_info, text):
        st_auth_token = sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME)
        st_helper = startrek_helper.STHelper(st_auth_token)
        issue = st_helper.find_ticket_by_release_number(self.Context.release_number, c_info, fail=True)
        assignee = issue.assignee.login
        logging.info("Commenting and summoning the issue assignee: %s", assignee)
        st_helper.comment(self.Context.release_number, text, c_info, summonees=[assignee])
