# -*- coding: utf-8 -*-

import collections
import logging
import time

import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2
from sandbox import common
from sandbox.common.types import client
from sandbox.common.types import task as sandbox_task

from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils2
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.common.sdk_compat.task_helper as rm_task
from sandbox.projects.release_machine.helpers import startrek_helper
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.components.configs import abt
from sandbox.projects.release_machine.components.configs import user_sessions
from sandbox.projects.common.z2 import z2_client
from sandbox.projects.common import link_builder as lb


PackageInfo = collections.namedtuple("PackageInfo", "task_id name version")


@rm_notify.notify2()
class ReleaseSearchComponentZ2(sdk2.Task):
    """
    The task deploys Debian packages with Z2 as a step in RM chain.

    In particular, it does the following:
    - takes as an input a list of YA_PACKAGE tasks
    - reads package name and version from every input task
    - makes sure it was published
    - updates specified Z2 configuration with newer versions
    - clicks button "Release" for every input task
    - posts status updates to ST issue
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 2048
        cores = 1
        client_tags = (~client.Tag.LINUX_LUCID &
                       (client.Tag.LINUX_PRECISE | client.Tag.LINUX_TRUSTY | client.Tag.LINUX_XENIAL))
        environments = [task_env.TaskRequirements.startrek_client]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup("RM component name") as component_name:
            component_name.values[abt.ABTCfg.name] = component_name.Value(value=abt.ABTCfg.name)
            component_name.values[user_sessions.UserSessionsCfg.name] = component_name.Value(
                value=user_sessions.UserSessionsCfg.name
            )
            # Add more supported components here

        package_tasks = sdk2.parameters.List(
            label="Package tasks",
            value_type=sdk2.parameters.Integer,
            required=True,
            description="IDs of YA_PACKAGE tasks to release",
        )

        release_number = sdk2.parameters.Integer(
            label="Release number",
            required=True,
            description="Major release number",
        )

        marks_resources_as_released = sdk2.parameters.Bool(
            label="Mark resources as released",
            required=False,
            description="Set True if you need mark resources as released in sandbox",
            default=True,
        )

        minutes_to_wait = sdk2.parameters.Integer(
            label="Wait time",
            required=False,
            description="Max time while task will try to update packets in Z2",
            default=60,
        )

        need_send_st = sdk2.parameters.Bool(
            label="Need send st message",
            required=False,
            description="Is need send message to st ticket",
            default=True,
        )

        with sdk2.parameters.RadioGroup("Roll packages to") as roll_to:
            roll_to.values[rm_const.ReleaseStatus.stable] = roll_to.Value(
                value=rm_const.ReleaseStatus.stable,
                default=True
            )
            roll_to.values[rm_const.ReleaseStatus.testing] = roll_to.Value(value=rm_const.ReleaseStatus.testing)
            roll_to.values[rm_const.ReleaseStatus.prestable] = roll_to.Value(value=rm_const.ReleaseStatus.prestable)
            roll_to.values[rm_const.ReleaseStatus.unstable] = roll_to.Value(value=rm_const.ReleaseStatus.unstable)

        with sdk2.parameters.RadioGroup("Z2 configuration") as z2_config_id:
            z2_config_id.values["MR_VELES02"] = z2_config_id.Value(value="MR_VELES02")
            z2_config_id.values["MR_VELES"] = z2_config_id.Value(value="MR_VELES")
            z2_config_id.values["MR_VELES01"] = z2_config_id.Value(value="MR_VELES01")
            # Add more supported configurations here

    class Context(sdk2.Task.Context):
        fail_msg = ""

    def on_execute(self):
        c_info = rmc.COMPONENTS[self.Parameters.component_name]()
        target_is_stable = self.Parameters.roll_to == rm_const.ReleaseStatus.stable
        if target_is_stable and not c_info.releases_cfg__allow_robots_to_release_stable and not utils2.is_human(self.author):
            msg = "Robots aren't allowed to make stable releases for component: {}".format(c_info.name)
            self.Context.fail_msg = msg
            eh.check_failed(self.Context.fail_msg)

        # update Z2 config
        z2client = z2_client.Z2Client(self.get_z2_key(), self.Parameters.z2_config_id, dry_run=False)
        packages = self.read_package_tasks()
        if not self.update_z2_config(z2client, packages, self.Parameters.minutes_to_wait):
            self.Context.fail_msg = ("Failed to update Z2 config. This could've put it into inconsistent state." +
                                     "Please, check Z2 via web interface!")
            eh.check_failed(self.Context.fail_msg)

        # mark tasks as "released"
        if self.Parameters.marks_resources_as_released:
            self.release_tasks()

        # post status to ST
        task_list = []
        for task_id in self.Parameters.package_tasks:
            task_list.append("- {}\n".format(common.utils.get_task_link(task_id)))
        start_comment = "The following tasks are being released in {}:\n{}".format(
            common.utils.get_task_link(self.id),
            "".join(task_list)
        )
        self.post_to_st_issue(c_info, start_comment)

        # wait for finish of the releasing process
        if self.Parameters.marks_resources_as_released:
            if not self.wait_for_released():
                self.Context.fail_msg = ("Failed to mark tasks as released to {}".format(self.Parameters.roll_to) +
                                         " after successful Z2 configuration update. \n" +
                                         "Please, check the Sandbox tasks via web interface!")
                eh.check_failed(self.Context.fail_msg)

        # post update to ST
        if self.Parameters.need_send_st:
            packages_list = []
            for info in packages:
                packages_list.append("- {}={}\n".format(info.name, info.version))
            finish_comment = u"Tasks are released to {}.\nPackages in {} are updated to:\n{}".format(
                self.Parameters.roll_to,
                self.Parameters.z2_config_id,
                "".join(packages_list)
            )
            self.post_to_st_issue(c_info, finish_comment)

    def on_failure(self, prev_status):
        super(ReleaseSearchComponentZ2, self).on_failure(prev_status)
        c_info = rmc.COMPONENTS[self.Parameters.component_name]()
        self.post_to_st_issue(c_info, u"{task} FAILED".format(
            task=lb.task_wiki_link(self.id, self.type),
        ))

    def get_z2_key(self):
        vault_key = "z2_api_key_{}".format(self.Parameters.z2_config_id)
        return sdk2.Vault.data(vault_key)

    def read_package_tasks(self):
        """
        :return: list of PackageInfo items
        """
        packages = []
        for task_id in self.Parameters.package_tasks:
            task = sdk2.Task.find(
                id=task_id,
                status=sandbox_task.Status.SUCCESS,
            ).first()
            eh.ensure(task, "not released yet task with id {} was not found".format(task_id))
            # verify
            package_type = rm_task.input_or_ctx_field(task, "package_type")
            eh.ensure(
                package_type == "debian",
                "task {} has unsupported package type: {}".format(task_id, package_type)
            )
            publish_package = rm_task.input_or_ctx_field(task, "publish_package")
            eh.ensure(publish_package, "task {} doesn't publish packages".format(task_id))
            # The right way to get package is to check output resources, but there is no good way
            # to tell that resource is a Debian package.
            # Therefore we parse the context field '_package_resources' instead.
            # In YaPackage2 or KosherYaPackage we have `package_resources` in Context,
            # and in YaPackage `_package_resources`.
            package_info = rm_task.ctx_field(task, "package_resources") or rm_task.ctx_field(task, "_package_resources")
            logging.info("package info: {}".format(package_info))
            resources = package_info["resources"]
            resource_ids = [item[0] for item in resources]
            for resource_id in resource_ids:
                resource = sdk2.Resource.find(id=resource_id).first()
                eh.ensure(resource, "Not found resource #{} produced by task #{}".format(resource_id, task_id))
                package_info = PackageInfo(task_id, resource.resource_name, resource.resource_version)
                logging.info("read package: task %d, %s - %s", task_id, package_info.name, package_info.version)
                packages.append(package_info)
        eh.ensure(len(packages) > 0, "No packages to update")
        return packages

    @staticmethod
    def update_z2_config(client, packages, max_wait_time=60, z2_update_wait_time=180, retry_wait_time=60):
        retry_count = 0
        start_time = int(time.time())
        # wait for update finish up to 60 minutes
        while int(time.time()) - start_time < 60 * max_wait_time:
            logging.info("update_z2_config try count = " + str(retry_count))
            retry_count += 1
            try:
                assert client.edit_items(packages), "failed to edit items in Z2 config"
                assert client.update(), "failed to start Z2 config update"
                time.sleep(z2_update_wait_time)
                assert client.update_status(), "z2 update not finished in {} seconds".format(z2_update_wait_time)
                return True
            except Exception as e:
                logging.warning("update_z2_config failed, retrying. error msg: " + str(e))
                time.sleep(retry_wait_time)
        logging.error("Z2 update didn't finish for too long. The task won't proceed further.")
        return False

    def release_tasks(self):
        api = common.rest.Client()
        roll_to = self.Parameters.roll_to
        for task_id in self.Parameters.package_tasks:
            logging.info("Releasing {} to {}...".format(task_id, roll_to))
            payload = {
                "task_id": task_id,
                "cc": [],
                "to": [],
                "params": {},
                "message": "",
                "type": roll_to,
                "subject": "",
            }
            logging.debug("API request 'release' with body: {}".format(payload))
            self.release_single_task(api, payload)

    @staticmethod
    @decorators.retries(max_tries=3, delay=5, raise_class=common.errors.TaskError)
    def release_single_task(api, payload):
        api.release(payload)

    def wait_for_released(self):
        start_time = int(time.time())
        # wait for build task released status up to 30 minutes
        while int(time.time()) - start_time < 30 * 60:
            if self.check_released_tasks():
                return True
            time.sleep(15)
        return False

    def check_released_tasks(self):
        """
        :return: True if all tasks got "RELEASED", False otherwise
        """
        not_released = 0
        for task_id in self.Parameters.package_tasks:
            task_status = rm_task.task_status(task_id)
            logging.info("task #{} has status {}".format(task_id, task_status))
            if task_status != sandbox_task.Status.RELEASED:
                not_released += 1
        return not_released == 0

    def post_to_st_issue(self, c_info, text):
        st_auth_token = sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME)
        st_helper = startrek_helper.STHelper(st_auth_token)
        st_helper.comment(self.Parameters.release_number, text, c_info)
