# -*- coding: utf-8 -*-

import collections
import logging

import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2
from sandbox import common
from sandbox.common.types import task as sandbox_task
from sandbox.common.types import client

from sandbox.projects.common import error_handlers as eh
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.helpers import startrek_helper
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.components.configs.userfeat import UserfeatCfg


TaskInfo = collections.namedtuple("TaskInfo", "task_id task_type status diff")


class SummarizeDiffs(sdk2.Task):
    """
    The task checks diffs produced by parents and reports a summary

    In particular, it does the following:
    - check 'has_diff' field in context of its parents
    - post summary to ST issue
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 2048
        cores = 1
        client_tags = (~client.Tag.LINUX_LUCID &
                       (client.Tag.LINUX_PRECISE | client.Tag.LINUX_TRUSTY | client.Tag.LINUX_XENIAL))
        environments = [task_env.TaskRequirements.startrek_client]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup("RM component name") as component_name:
            component_name.values[UserfeatCfg.name] = component_name.Value(value=UserfeatCfg.name)
            # Add more supported components here

        comparing_tasks = sdk2.parameters.List(
            label="Comparing tasks",
            value_type=sdk2.parameters.Integer,
            required=True,
            description="IDs of comparing tasks to check",
        )

        release_number = sdk2.parameters.Integer(
            label="Release number",
            required=True,
            description="Release number to detect ST issue",
        )

    class Context(sdk2.Task.Context):
        pass

    def on_execute(self):
        c_info = rmc.COMPONENTS[self.Parameters.component_name]()

        task_info = []
        for task_id in self.Parameters.comparing_tasks:
            task = self.check_single_task(task_id)
            logging.info("Checked task: %s", str(task))
            task_info.append(task)

        eh.ensure(
            len(task_info) > 0,
            "No tasks found to report on"
        )

        summary = self.prep_summary(task_info)

        self.post_to_st_issue(c_info, summary)

    @staticmethod
    def check_single_task(task_id):
        """
        :param task_id:
        :return: TaskInfo
        """
        task = sdk2.Task[task_id]
        task_type = task.type.type
        diff_label = "n/a"
        eh.ensure(
            task.status not in {sandbox_task.Status.ENQUEUED, sandbox_task.Status.EXECUTING},
            "Task #{} didn't finish yet".format(task_id)
        )
        if task.status == sandbox_task.Status.SUCCESS:
            diff_label = "Dirty" if task.Context.has_diff else "Clear"
        return TaskInfo(task_id, task_type, task.status, diff_label)

    @staticmethod
    def prep_summary(task_info):
        result = "===Regression testing===\n"
        result += "\n"
        result += "#|\n"
        result += "||**Status**|**Diff**|**Task**|**Type**||\n"
        for task in task_info:
            task_id = task.task_id
            cells = [
                task.status,
                task.diff,
                "(({} {}))".format(common.utils.get_task_link(task_id), task_id),
                task.task_type,
            ]
            result += "||{}||\n".format("|".join(cells))
        result += "|#"
        return result

    def post_to_st_issue(self, c_info, text):
        st_auth_token = sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME)
        st_helper = startrek_helper.STHelper(st_auth_token)
        st_helper.comment(self.Parameters.release_number, text, c_info)
