# -*- coding: utf-8 -*-
import logging
import re

from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask

import sandbox.projects.release_machine.input_params as rm_params
import sandbox.projects.release_machine.components.all as rmc
from sandbox.projects.common.testenv_client import TEClient
from sandbox.projects.common import error_handlers as eh

_TE_REQUEST_RETRIES_NUMBER = 10
_LAST_GOOD_REVISION_INFO = "last_good_revision_info"
_CHECK_INTERVAL = 30 * 60  # 30 minutes
_MAXIMUM_NUMBER_OF_CHECKS = 5 * 24 * 2  # 5 days
_CHECKS_COUNTER = "checks_counter"
_TRUNK_REVISION = "trunk_revision"
_TE_TRUNK_DB = "testenv_cfg__trunk_db"
_OBSTACLE_PROBLEMS = "obstacle_problems"
_MISSION_COMPLETE = "mission_complete"
_IGNORE_RELEASE_JOBS = "ignore_release_jobs"
_TE_PROBLEM_RE = re.compile(r'Problem (\d+) at revision (\d+)')
_TE_PROBLEM_URL_TMPL = "https://testenv.yandex-team.ru/?screen=problem&database={database}&id={problem_id}"
_ARCANUM_REVISION_PAGE_URL_TMPL = "https://a.yandex-team.ru/arc/commit/{revision}"
_SANDBOX_TASK_PAGE_URL_TMPL = "https://sandbox.yandex-team.ru/task/{task_id}/view"
_STAFF_URL_TMPL = "https://staff.yandex-team.ru/{person}"


class WaitTrunkDB(SandboxTask):
    """
        **Release-machine**

        Executes until TE database corresponding to trunk of specified component
        finishes checking revision that originated specified branch
    """
    type = "WAIT_TRUNK_DB"

    input_parameters = [
        rm_params.ComponentName,
        rm_params.ReleaseNum,
    ]
    cores = 1

    def get_trunk_revision(self, branch_path):
        """
            Get trunk revision that originated the branch
            :param branch_path: Branch path
            :return: Trunk revision
        """

        # NOTE Arcadia.log returns entries ordered similarly to svn log --xml

        # Find out first revision in a branch
        info_list = Arcadia.log(branch_path, "HEAD", revision_to=1, stop_on_copy=True)
        eh.verify(info_list, "Cannot find first branch revision for branch {}".format(branch_path))

        branch_first_rev = info_list[-1]["revision"]
        logging.info("First revision in branch %s is %s", branch_path, branch_first_rev)

        # Find previous revision
        info_list = Arcadia.log(branch_path, branch_first_rev, revision_to=1, limit=2)
        eh.verify(len(info_list) == 2, "Cannot find trunk revision for branch {}".format(branch_path))

        trunk_revision = info_list[1]["revision"]
        logging.info("Trunk origin revision for branch %s is %s", branch_path, trunk_revision)

        return trunk_revision

    def on_execute(self):
        """
        Wait until last good revision is greater than branch origin revision in trunk
        """

        component_name = self.ctx[rm_params.ComponentName.name]

        # Find out trunk revision that was copied to create the branch
        if _TRUNK_REVISION not in self.ctx:
            c_info = rmc.COMPONENTS[component_name]()
            c_info.check_testenv_db(self)
            branch_num = self.ctx[rm_params.ReleaseNum.name]
            branch_path = c_info.full_branch_path(branch_num)
            self.ctx[_TRUNK_REVISION] = self.get_trunk_revision(branch_path)

        # Save relevant TE database running on trunk
        if _TE_TRUNK_DB not in self.ctx:
            self.ctx[_TE_TRUNK_DB] = c_info.testenv_cfg__trunk_db

        # Look if we need to ignore release jobs when looking for last good revision
        if _IGNORE_RELEASE_JOBS not in self.ctx:
            c_info = rmc.COMPONENTS[component_name]()
            self.ctx[_IGNORE_RELEASE_JOBS] = c_info.testenv_last_good_revision_ignore_release_jobs

        # Get actual TE progress and compare with origin revision
        self.ctx[_LAST_GOOD_REVISION_INFO] = TEClient.get_last_good_revision_info(
            self.ctx[_TE_TRUNK_DB],
            self.ctx[_IGNORE_RELEASE_JOBS],
        )
        self.ctx[_CHECKS_COUNTER] = self.ctx.get(_CHECKS_COUNTER, 0) + 1
        last_good_revision = int(self.ctx[_LAST_GOOD_REVISION_INFO]["revision"])
        trunk_revision = int(self.ctx[_TRUNK_REVISION])

        if last_good_revision < trunk_revision:
            logging.info(
                "Last good revision %s is less than trunk revision %s",
                last_good_revision, trunk_revision,
            )

            unresolved_problems = TEClient.get_te_problems(self.ctx[_TE_TRUNK_DB], unresolved_only=True)
            logging.info("Unresolved problems count: %s", len(unresolved_problems))
            self.ctx[_OBSTACLE_PROBLEMS] = [
                x for x in unresolved_problems["rows"] if x["revision"] <= self.ctx[_TRUNK_REVISION]
            ]
            logging.info("Obstacle problems count: %s", len(self.ctx[_OBSTACLE_PROBLEMS]))

            if self.ctx[_CHECKS_COUNTER] >= _MAXIMUM_NUMBER_OF_CHECKS:
                eh.check_failed("Maximum number of checks reached")

            self.wait_time(_CHECK_INTERVAL)
        else:
            self.ctx[_OBSTACLE_PROBLEMS] = []
            self.ctx[_MISSION_COMPLETE] = True

    @property
    def footer(self):
        """
        Feedback method

        Displays last check results if available.
        Tries to beautify description with hyperlinks.
        """
        report = ""

        trunk_revision = self.ctx.get(_TRUNK_REVISION)
        if trunk_revision:
            report += "Branch origin revision is <b>{}</b>".format(trunk_revision)

        testenv_cfg__trunk_db = self.ctx.get(_TE_TRUNK_DB)
        if testenv_cfg__trunk_db:
            if report:
                report += "<br>"
            report += "TE database: {}".format(testenv_cfg__trunk_db)

        good_revision_info = self.ctx.get(_LAST_GOOD_REVISION_INFO)
        if good_revision_info:
            description = good_revision_info["description"]
            match = _TE_PROBLEM_RE.match(description)
            if match:
                problem_id = match.group(1)
                problem_page_url = _TE_PROBLEM_URL_TMPL.format(
                    database=testenv_cfg__trunk_db, problem_id=problem_id
                )
                problem_revision = match.group(2)
                problem_revision_page_url = _ARCANUM_REVISION_PAGE_URL_TMPL.format(
                    revision=problem_revision
                )
                description = "Problem <a href=\"{}\">{}</a> at revision <a href=\"{}\">{}</a>".format(
                    problem_page_url, problem_id, problem_revision_page_url, problem_revision
                )
            if report:
                report += "<br>"
            report += "Last good revision is <b>{}</b><br><i>{}</i>".format(good_revision_info["revision"], description)

            if self.ctx[_OBSTACLE_PROBLEMS]:
                report += "\n<br><br>\nObstacle problems<br>\n<table>\n"
                for problem in self.ctx[_OBSTACLE_PROBLEMS]:
                    problem_description = {
                        "test_name": problem["test_name"],
                        "revision": problem["revision"],
                        "revision_url": _ARCANUM_REVISION_PAGE_URL_TMPL.format(**problem),
                        "revision_comment": problem["test_diff/revision2_info/comment"].encode("utf-8"),
                        "revision_author": problem["test_diff/revision2_info/author"],
                        "owner": problem["owner"],
                        "problem_url": _TE_PROBLEM_URL_TMPL.format(
                            database=testenv_cfg__trunk_db, problem_id=problem["test_diff_id"]
                        ),
                        "problem_task_url": _SANDBOX_TASK_PAGE_URL_TMPL.format(
                            task_id=problem["test_diff/task_id"]
                        ),
                        "owner_url": _STAFF_URL_TMPL.format(
                            person=problem["owner"]
                        ),
                        "revision_author_url": _STAFF_URL_TMPL.format(
                            person=problem["test_diff/revision2_info/author"]
                        ),
                    }
                    report += (
                        "<tr>"

                        "<td style=\"padding: 15px\">{test_name}<br />"
                        "<a href=\"{owner_url}\">{owner}</a></td>"

                        "<td style=\"padding: 15px\"><a href=\"{revision_url}\">{revision}</a> &nbsp; "
                        "<i>{revision_comment}</i><br><a href=\"{revision_author_url}\">{revision_author}</a></td>"

                        "<td style=\"padding: 15px\"><a href=\"{problem_task_url}\">Task</a><br/>"
                        "<a href=\"{problem_url}\">Problem</a></td>"
                        "</tr>\n"
                    ).format(**problem_description)
                report += "</table>"

            if self.ctx.get(_MISSION_COMPLETE):
                report += "\n<br/><br/>\n<i>Mission complete!</i>"

        return report or "Don't panic"


__Task__ = WaitTrunkDB
