import os
import logging
import jinja2

from sandbox import sdk2
from sandbox import common
from sandbox.common.types import resource
from sandbox.common.types import client as ctc
from sandbox.sandboxsdk import process
from sandbox.sdk2.service_resources import SandboxTasksBinary
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.repo.pciexpress.legs.utils import PciExpressArc


class PciExpressJog(sdk2.Resource):  # PCI_EXPRESS_JOG
    auto_backup = False
    restart_policy = resource.RestartPolicy.DELETE

    units = sdk2.parameters.String("Built units")


class PciExpressLegsParameters(sdk2.Task.Parameters):

    with sdk2.parameters.Group("PCI Express Legs parameters"):
        use_last_binary = sdk2.parameters.Bool('use last binary archive', default=True)

        with use_last_binary.value[True]:
            with sdk2.parameters.RadioGroup('binary release type') as release_type:
                release_type.value.stable = release_type.Value('stable', default=True)
                release_type.value.testing = release_type.Value('testing')

        with use_last_binary.value[False]:
            custom_tasks_archive_resource = sdk2.parameters.Resource('task resource', default=None)

        arc_server_url = sdk2.parameters.String('arc_server_url', description='Arc server url', required=False)
        units = sdk2.parameters.String('units', description='units description in protobuf', multiline=True)
        ya_token = sdk2.parameters.YavSecretWithKey("YA token", required=False)
        with sdk2.parameters.RadioGroup("Graph compaction type") as compaction:
            compaction.values.by_package = compaction.Value("by package", default=True)
            compaction.values.by_package_2 = compaction.Value("by package (new)")
            compaction.values.by_platform = compaction.Value("by platform")
            compaction.values.by_platform_2 = compaction.Value("by platform (new)")


class PciExpressLegs(sdk2.Task):  # PCI_EXPRESS_LEGS
    """
    Pci-Express task to get if the commit affect one of the project under the configured pci-dss scope.
    Result is list of affected projects or a list of project errors if they take place for this commit.
    """
    class Requirements(sdk2.Requirements):
        cores = 2
        disk_space = 5 * 1024  # 5 GiB
        ram = 8 * 1024         # 8 GB
        client_tags = ctc.Tag.LXC | ctc.Tag.PORTOD

        class Caches(sdk2.Requirements.Caches):
            pass

    Parameters = PciExpressLegsParameters

    def on_save(self):
        if self.Parameters.use_last_binary:
            binary = SandboxTasksBinary.find(
                attrs={
                    'target': 'repo/pciexpress/legs/bin',
                    'released': self.Parameters.release_type or 'stable'
                }
            ).first()

            if binary:
                self.Requirements.tasks_resource = binary.id
        else:
            self.Requirements.tasks_resource = self.Parameters.custom_tasks_archive_resource

    def on_execute(self):
        # Implement arc client
        from repo.pciexpress.dumper import ya_client
        from repo.pciexpress.legs import legs
        import library.python.json as json
        import library.python.tmp as tmp
        import collections

        class SandboxYaGraphClient(ya_client.YaClientBase):
            YaResult = collections.namedtuple("YaResult", ("returncode", "stderr", "stdout"))

            def __init__(self, arcadia_root):
                super(SandboxYaGraphClient, self).__init__()
                self.arcadia_dir = os.path.join(os.getcwd(), 'fake-arcadia-root')
                os.mkdir(self.arcadia_dir)
                os.symlink(arcadia_root, os.path.join(self.arcadia_dir, 'arcadia'))

            def _is_target_exists(self, directories):
                return all([os.path.isfile(os.path.join(self.arcadia_dir, 'arcadia', directory, "ya.make")) for directory in directories])

            def _is_package_exists(self, package):
                return os.path.isfile(os.path.join(self.arcadia_dir, 'arcadia', package))

            def _check_build_targets(self, targets):
                return all([self._is_target_exists(targets[target]) for target in targets])

            @staticmethod
            def _ya(command, cwd):
                logging.debug("Executing: {}".format(command))
                logging.debug("cwd: {}".format(cwd))
                with open('stdout', 'w') as stdout, open('stderr', 'w') as stderr:
                    proc = process.run_process(command, wait=True, check=False, stdout=stdout, stderr=stderr, work_dir=cwd)
                with open('stdout', 'r') as stdout, open('stderr', 'r') as stderr:
                    logging.debug("Executing completed with code: {}".format(proc.returncode))
                    return SandboxYaGraphClient.YaResult(returncode=proc.returncode, stderr=stderr.read(), stdout=stdout.read())

            def get_inputs(self, package, arcadia_root):
                with tmp.temp_file(prefix="package_report") as tf:
                    if not self._is_package_exists(package):
                        raise legs.UnretryableError("Package does not exist: {}".format(package))
                    sdk.do_package(self.arcadia_dir, [package], dump_inputs_to=tf, publish_logs=False,
                                   check_build_system=False, no_report=True)
                    with open(tf) as f:
                        data = f.read()
                    data = json.loads(data)
                    if not self._check_build_targets(data[package]['build']):
                        raise legs.UnretryableError("Package with invalid targets: {}".format(package))
                return data

            def get_graph(self, arcadia_root, target_platform, targets, flags=None):
                ya = os.path.join(arcadia_root, 'ya')
                target_platform_cmd = self._build_target_platform_cmd(target_platform, flags)
                command = [ya, 'make', '-j0', '--sandboxing', '-G'] + target_platform_cmd + list(targets)
                result = self._ya(command, arcadia_root)

                if result.returncode == 1:
                    raise legs.UnretryableError(
                        'run "{}" failed with exit code {}\nstderr: {}'.format(' '.join(command), result.returncode,
                                                                               result.stderr))
                assert len(result.stdout), '"ya {}" stdout is empty\nstderr {}'.format(result.stdout, result.stderr)
                return json.loads(result.stdout)

            def get_recurses(self, arcadia_root, target_platform, targets, flags=None):
                raise NotImplementedError()

        if self.Parameters.ya_token:
            os.environ['YA_TOKEN'] = self.Parameters.ya_token.data()[self.Parameters.ya_token.default_key]

        if self.Parameters.arc_server_url == 'arc-prestable.sas.yp-c.yandex.net' or self.Parameters.arc_server_url == 'arcd.test.arc-vcs.in.yandex-team.ru':
            result = legs.run(PciExpressArc(use_prestable=True), SandboxYaGraphClient, self.Parameters.units, self.Parameters.compaction)
        else:
            result = legs.run(PciExpressArc(use_prestable=False), SandboxYaGraphClient, self.Parameters.units, self.Parameters.compaction)

        jog = PciExpressJog(self, "PCI Express Legs result", "result")
        jog.units = self.Parameters.units

        data = sdk2.ResourceData(jog)
        data.path.write_bytes(result)
        data.ready()
        self.set_legs_info(result)

    def handle_successful_sandbox_task(self, result_object):
        pci_dss_packages = []
        pci_free_packages = []
        package_errors = {}
        for build_result in result_object.Results:
            if build_result.WhichOneof('Payload') == 'Error':
                package_errors[build_result.Package.Package] = build_result.Error.Text
            if build_result.Info.Files:
                pci_dss_packages.append(build_result.Package.Package)
            else:
                pci_free_packages.append(build_result.Package.Package)
        return pci_dss_packages, pci_free_packages, package_errors

    def set_legs_info(self, result):
        try:
            import google.protobuf.text_format
            from repo.pciexpress.proto import dto_pb2
            output = google.protobuf.text_format.Parse(result, dto_pb2.LegsOutput())
            pci_dss_packages, pci_free_packages, package_errors = self.handle_successful_sandbox_task(output)
            template_path = os.path.join(os.path.dirname(__file__), "result.jinja2")
            template = jinja2.Template(common.fs.read_file(template_path))
            template_context = {
                'errors': package_errors,
                'pcidss': pci_dss_packages,
                'commit': output.CommitId,
            }
            self.set_info(template.render(template_context), do_escape=False)
        except Exception as e:
            self.set_info(e)
