import json

from sandbox import sdk2
from sandbox.sdk2.service_resources import SandboxTasksBinary


class WatchDogCollectorParameters(sdk2.Task.Parameters):

    with sdk2.parameters.Group("watchdog parameters"):
        use_last_binary = sdk2.parameters.Bool('use last binary archive', default=True)

        with use_last_binary.value[True]:
            with sdk2.parameters.RadioGroup('Binary release type') as release_type:
                release_type.value.stable = release_type.Value('stable', default=True)
                release_type.value.test = release_type.Value('test')

        with use_last_binary.value[False]:
            custom_tasks_archive_resource = sdk2.parameters.Resource('task archive resource', default=None)

        repositories_mode = sdk2.parameters.Bool('repositories mode', default=False)
        commits_mode = sdk2.parameters.Bool('commits mode', default=False)
        manifest_mode = sdk2.parameters.Bool('snapshots mode', default=False)
        collect_from_arcadia = sdk2.parameters.Bool('collect data from arcadia', default=False)
        collect_from_github = sdk2.parameters.Bool('collect data from github', default=False)
        collect_from_bb = sdk2.parameters.Bool('collect data from bitbucket', default=False)
        yt_proxy = sdk2.parameters.String('yt proxy', default='hahn')
        solomon_cluster = sdk2.parameters.String('solomon cluster', default='solomon')

        yt_directory = sdk2.parameters.String('YT directory where everything will be stored')

        emails_table = sdk2.parameters.String('emails table')
        repositories = sdk2.parameters.String('repositories settings in JSON', default=None)


class WatchDogCollector(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1 * 1024  # 1 GB
        ram = 10 * 1024  # 10 GB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(WatchDogCollectorParameters):
        pass

    def on_save(self):
        if self.Parameters.use_last_binary:
            self.Requirements.tasks_resource = SandboxTasksBinary.find(
                attrs={
                    'target': 'repo/yeoman/collector/bin',
                    'released': self.Parameters.release_type or 'stable'
                }
            ).first().id
        else:
            self.Requirements.tasks_resource = self.Parameters.custom_tasks_archive_resource

    def on_execute(self):
        secrets = sdk2.yav.Secret('sec-01d3rrchvs6qnza584y5bgq6f9').data()

        tokens = {
            'yt': secrets['token'],
            'yql': secrets['token'],
            'bb': secrets['bb'],
            'github': secrets['github'],
            'staff': secrets['token'],
            'ssh_key': secrets['ssh_key'],
            'ssh_user': secrets['ssh_user'],
        }

        import yt.wrapper as yt
        import repo.yeoman.data.lib.collector

        yt_config = {
            'proxy': {'url': self.Parameters.yt_proxy},
            'token': tokens['yt']}

        client = yt.YtClient(config=yt_config)

        repositories = {}
        if self.Parameters.repositories:
            repositories = json.loads(self.Parameters.repositories)

        repo.yeoman.data.lib.collector.run_data_collector(
            client=client,
            repositories_mode=self.Parameters.repositories_mode,
            commits_mode=self.Parameters.commits_mode,
            snapshots_mode=self.Parameters.manifest_mode,
            yt_directory=self.Parameters.yt_directory,
            emails_table=self.Parameters.emails_table,
            collect_from_arcadia=self.Parameters.collect_from_arcadia,
            collect_from_bb=self.Parameters.collect_from_bb,
            collect_from_github=self.Parameters.collect_from_github,
            solomon_cluster=self.Parameters.solomon_cluster,
            tokens=tokens,
            repositories=repositories,
        )
