# -*- coding: utf-8 -*-

import json
import os
import requests
import tarfile

from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry

from sandbox import sdk2
from sandbox.common import errors
from sandbox.sandboxsdk.paths import make_folder
from sandbox.projects.websearch.begemot.common import add_realtime_version_file
from sandbox.projects.websearch.begemot.resources import BEGEMOT_REALTIME_PACKAGE


class BuildBegemotLogoRealtimePackage(sdk2.Task):
    """
        Обновление бегемотных fresh данных для логотипов серпа
    """

    class Parameters(sdk2.Parameters):
        data_files = sdk2.parameters.List('Logo data files', default=['logo_serp.testing.json', 'logo_serp.json'])
        base_url = sdk2.parameters.String('Base url', default='http://portal-export.yandex.net/pusher_exports/wrk/')
        rule_name = sdk2.parameters.String('Logo Data rule name', default='FlagsProviderLogoData')

    def _validate_logo_data(self, logo_data):
        not_empty_fields = ['domain', 'lang', 'from', 'till', 'url']

        assert('data' in logo_data)
        assert(len(logo_data['data']) > 0)
        for item in logo_data['data']:
            for field in not_empty_fields:
                assert(item[field])

    def on_execute(self):
        fresh_path = 'search/wizard/data/fresh/package/' + self.Parameters.rule_name
        arcadia_fresh_folder = sdk2.svn.Arcadia.get_arcadia_src_dir(sdk2.svn.Arcadia.trunk_url(fresh_path))
        comment = "New Logo data, (sandbox:{}:{}) SKIP_CHECK".format(self.type, self.id)

        data_path = self.Parameters.rule_name
        data_folder = os.path.abspath(data_path)
        make_folder(data_folder)

        session = requests.Session()
        session.mount('http://', HTTPAdapter(max_retries=Retry(total=5, backoff_factor=0.1)))

        for file in self.Parameters.data_files:
            with open(os.path.join(data_folder, file), 'w') as f:
                url = self.Parameters.base_url + file
                r = session.get(url, headers={'Content-Type': 'application/json'}, timeout=10)
                if not r:
                    raise errors.TemporaryError('Could not get logo data from: {}'.format(url))
                received_data = r.json()
                self._validate_logo_data(received_data)
                f.write(json.dumps(received_data))

                with open(os.path.join(arcadia_fresh_folder, file), 'w') as f_arc:
                    f_arc.write(json.dumps(received_data, sort_keys=True, indent=2))

        diff = sdk2.svn.Arcadia.diff(arcadia_fresh_folder)
        if diff:
            self.set_info('Diff to commit:\n{}'.format(diff))
            commit_res = sdk2.svn.Arcadia.commit(arcadia_fresh_folder, comment, 'zomb-sandbox-rw')
            self.set_info(commit_res)
        else:
            self.set_info('No diff to commit to arcadia')

        package_archive = sdk2.ResourceData(BEGEMOT_REALTIME_PACKAGE(
            self, BEGEMOT_REALTIME_PACKAGE.name + ' for RequestInit',
            "realtime_wizard_data.tar", ttl=31, released='stable', shard='RequestInit', version=self.id
        ))

        version_file = add_realtime_version_file('.', self.id)
        with tarfile.open(str(package_archive.path), 'w:gz') as tar:
            tar.add(data_path)
            tar.add(version_file)

        package_archive.ready()
