# -*- coding: utf-8 -*-

import os
import json
import copy

from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxArcadiaUrlParameter, TaskSelector
from sandbox.sandboxsdk.channel import channel
from sandbox.projects import TestReportPerformance as TRP
from sandbox.projects.common.app_host.options import AppHostBundle
from sandbox.projects.common.app_host import AppHostProvider
from sandbox.projects.app_host import resources as app_host_resources
from sandbox.projects import resource_types


class AppHostArcadiaUrlParameter(SandboxArcadiaUrlParameter):
    required = False
    description = 'Svn url for arcadia (AppHost configs)'
    default_value = ''


class AppHostConfigBundle(TaskSelector):
    name = 'app_host_config_bundle'
    description = 'Task with AppHost config bundle:'
    task_type = 'BUILD_APP_HOST_CONFIG_BUNDLE'
    required = False
    default_value = None


class AppHostBackends(TaskSelector):
    name = 'app_host_backends'
    description = 'Task with AppHost backends:'
    task_type = 'BUILD_APP_HOST_BACKENDS_CONFIG'
    required = False
    default_value = None


class AppHostMapping(TaskSelector):
    name = 'app_host_mapping'
    description = 'Task with AppHost mapping:'
    task_type = 'BUILD_APP_HOST_GRAPH_NAME_MAPPING'
    required = False
    default_value = None


class AppHostBackendsType(SandboxStringParameter):
    name = 'app_host_backends_type'
    description = 'AppHost backends type:'
    default_value = "MAN_WEB_APP_HOST_HAMSTER"


class TestMappingPerformance(TRP.TestReportPerformance):
    type = 'TEST_MAPPING_PERFORMANCE'

    input_parameters = TRP.TestReportPerformance.input_parameters + [AppHostBundle, AppHostConfigBundle, AppHostBackends, AppHostBackendsType, AppHostMapping, AppHostArcadiaUrlParameter]

    def app_host_port(self):
        return "9080"

    def app_host_config_bundle_path(self):
        return "app_host_config_bundle"

    def app_host_backends_dir_path(self):
        return "app_host_backends"

    def app_host_backends_path(self):
        return os.path.join(self.app_host_backends_dir_path(), "backends.json")

    def app_host_custom_mapping_dir_path(self):
        return os.path.join(self.abs_path(), "custom_mapping")

    def app_host_custom_mapping_path(self):
        return os.path.join(self.app_host_custom_mapping_dir_path(), "graph_name_mapping.txt")

    def create_and_wait_subtask(self, task, context):
        subtask = self.create_subtask(
            task_type=task.task_type,
            description=self.descr,
            input_parameters=context,
            priority=self.priority,
            inherit_notifications=True
        )

        self.ctx[task.name] = subtask.id

        self.wait_tasks(
            [subtask],
            (self.Status.SUCCESS, self.Status.FAILURE, self.Status.DELETED, self.Status.RELEASED,
             self.Status.EXCEPTION, self.Status.TIMEOUT),
            True
        )

    def get_resource(self, task_id, resource):
        resources = channel.sandbox.list_resources(resource_type=resource, task_id=task_id, status="READY")
        if resources and len(resources) > 0:
            return resources[0].id
        return None

    def sync_app_host_backends(self):
        if self.ctx[AppHostBackends.name] is None:
            context = self.ctx.copy()
            context["choose_app_to_build"] = "app_host"
            context["choose_vertical"] = "WEB"
            context[SandboxArcadiaUrlParameter.name] = self.ctx.get(AppHostArcadiaUrlParameter.name)
            self.create_and_wait_subtask(AppHostBackends, context)

        app_host_backends_resource_id = self.get_resource(self.ctx[AppHostBackends.name], app_host_resources.APP_HOST_BACKENDS_CONFIG_WEB)
        app_host_backends_resource = self.expand_resource(
            res_id=app_host_backends_resource_id,
            dest=self.app_host_backends_dir_path(),
        )

        backends_config = None
        backends_config_path = os.path.join(self.app_host_backends_dir_path(), app_host_backends_resource.file_name)
        with open(backends_config_path, 'r') as backends_source:
            backends_config = json.load(backends_source)

        backends_config_type = self.ctx[AppHostBackendsType.name]
        backends_config = backends_config[backends_config_type]
        with open(self.app_host_backends_path(), 'w') as backends_destination:
            json.dump(backends_config, backends_destination)
        return backends_config_path

    def sync_app_host_graphs(self):
        if self.ctx[AppHostConfigBundle.name] is None:
            context = self.ctx.copy()
            context[SandboxArcadiaUrlParameter.name] = self.ctx.get(AppHostArcadiaUrlParameter.name)
            context["choose_app_to_build"] = "app_host"
            context["choose_vertical"] = "WEB"
            context["build_templates_dir"] = False
            context["run_smoke_test"] = False
            context["no_backends_in_graph"] = True
            context["enrich_graphs_revision"] = True
            context["build_service_resources"] = False
            context["BUILD_APP_HOST_BACKENDS_CONFIG"] = self.ctx[AppHostBackends.name]
            self.create_and_wait_subtask(AppHostConfigBundle, context)

        app_host_config_bundle_resource_id = self.get_resource(self.ctx[AppHostConfigBundle.name], resource_types.APP_HOST_CONFIG_BUNDLE_WEB)
        self.expand_resource(
            res_id=app_host_config_bundle_resource_id,
            dest=self.app_host_config_bundle_path(),
        )

    def sync_app_host_mapping(self):
        if self.ctx[AppHostMapping.name] is None:
            context = self.ctx.copy()
            context[SandboxArcadiaUrlParameter.name] = self.ctx.get(AppHostArcadiaUrlParameter.name)
            context["choose_vertical"] = "WEB"
            self.create_and_wait_subtask(AppHostMapping, context)

        app_host_mapping_resource_id = self.get_resource(self.ctx[AppHostMapping.name],  resource_types.APP_HOST_GRAPH_NAME_MAPPING_WEB)
        resource = self.expand_resource(
            res_id=app_host_mapping_resource_id,
            dest=self.app_host_custom_mapping_dir_path(),
        )
        path = os.path.join(self.app_host_custom_mapping_dir_path(), os.path.basename(resource.file_name))
        if not os.path.samefile(path, self.app_host_custom_mapping_path()):
            os.symlink(path, self.app_host_custom_mapping_path())

    def sync_app_host(self):
        self.app_host = AppHostProvider()
        self.app_host.options = {
            "graphs": {}
        }
        self.app_host.ports = {"main": self.app_host_port()}
        self.app_host.resources = {
                "binary": {"id": self.ctx[AppHostBundle.name], "type": "file"}
        }
        self.app_host.ctx = copy.deepcopy(self.ctx)
        self.app_host.ctx['abs_path'] = self.abs_path()
        self.app_host.use_profiler = False
        self.app_host.use_gperftools = False
        self.app_host.cmd_tpl += " --no-mlock"

        with open(os.path.join(self.app_host_config_bundle_path(), 'hamster_app_host.json'), 'r') as hamster_config:
            self.app_host.options['config'] = json.load(hamster_config)
        self.app_host.options['config']['backends_path'] = self.app_host_backends_path()
        self.app_host.options['config']['conf_dir'] = os.path.join(self.app_host_config_bundle_path(), '_ALL/graphs/')
        self.app_host.options['config']['update_from_fs_config'] = {}
        self.app_host.options['config'].pop('profile_log', None)
        self.app_host.options['config'].pop('fallback_conf_dir', None)
        self.app_host.options['config'].pop('fallback_backends_path', None)
        self.app_host.options['config']['log'] = 'current-eventlog-app_host'
        self.app_host.options['config']["graph_name_mapping_path"] = self.app_host_custom_mapping_path()

        self.app_host.prepare()
        self.app_host.start()

    def prepare_env(self):
        if "app_host" not in self.ctx[TRP.ExtraParams.name]:
            self.sync_app_host_backends()
            self.sync_app_host_graphs()
            self.sync_app_host_mapping()
            self.sync_app_host()

            app_host_address = "localhost:" + self.app_host_port()
            self.ctx[TRP.ExtraParams.name] += " --app_host='"+app_host_address+"'"

        TRP.TestReportPerformance.prepare_env(self)


__Task__ = TestMappingPerformance
