from sandbox import sdk2
import requests
import logging
from datetime import datetime, date
from sandbox.common.errors import TaskFailure


class Changer(object):
    def __init__(self, token):
        self.telegraph = 'https://telegraph.yandex.net/api/v3/cucm/translation_pattern/uuid/1E477BF9-1BD7-5269-3240-1812C54DF69B'
        self.headers = {
            'Authorization': 'OAuth {0}'.format(token),
            'Content-Type': 'application/json'
        }

    def get_number(self):
        data = None
        for i in range(5):
            try:
                res = requests.get(
                    self.telegraph,
                    headers=self.headers,
                    verify=True,
                    timeout=5
                )
                data = res.json() or {}
                if not data.get('error'):
                    break
            except:
                continue

        return data.get('result', {}).get('calledPartyTransformationMask')

    def set_number(self, number):
        for i in range(5):
            try:
                requests.put(
                    self.telegraph,
                    headers=self.headers,
                    json={"calledPartyTransformationMask": str(number)},
                    verify=True,
                    timeout=5
                )
                break
            except:
                continue


class Calendar(object):
    API_TIME_FMT = '%Y-%m-%dT%H:%M:%S'
    API_URL = 'http://calendar-api.tools.yandex.net/internal'

    def __init__(self, calendar_id, change_hour):
        self.calendar_id = calendar_id
        self.change_hour = change_hour

    def choose_current_event(self, events):
        previousEvent = None
        nextEvent = None
        today = datetime.combine(date.today(), datetime.min.time())
        for event in events:
            startTime = datetime.strptime(event['startTs'], self.API_TIME_FMT)
            if startTime < today:
                previousEvent = event
            else:
                nextEvent = event
        if previousEvent is None:
            return nextEvent
        if nextEvent is None:
            return previousEvent
        if datetime.now().hour >= self.change_hour:
            return nextEvent
        return previousEvent

    def get_event(self):
        now = datetime.now()
        params = {
            'from': now.strftime(self.API_TIME_FMT),
            'to': now.strftime(self.API_TIME_FMT),
            'tz': 'Europe/Moscow',
            'layerId': self.calendar_id
        }
        res = requests.get(self.API_URL + '/get-events', params)
        data = res.json()
        logging.info('Calendar response: %s', data)
        if 'events' not in data:
            return None

        return self.choose_current_event(data['events'])


class Staff(object):
    API_URL = "https://staff-api.yandex-team.ru/v3/persons"

    def __init__(self, token):
        self.headers = {
            'Authorization': 'OAuth {0}'.format(token),
            'Content-Type': 'application/json'
        }

    def get_phone(self, login):
        params_staff = {
            "login": login,
            "_fields": "phones.number,phones.type"
        }
        response = requests.get(url=self.API_URL, headers=self.headers, params=params_staff)
        response = response.json()
        for result in response["result"]:
            if result.get("phones"):
                for number in result["phones"]:
                    if number.get("type") and number["type"] == "mobile":
                        return number["number"]
        return None


class ReportDutyChanger(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        calendar_id = sdk2.parameters.Integer("Calendar id with duty schedule", required=True, default=37191)
        change_hour = sdk2.parameters.Integer("Hour when duty is assigned to other person", required=True, default=18)

    def on_execute(self):
        robot_report_duty_token = sdk2.Vault.data('robot-report-duty-token')

        calendar = Calendar(self.Parameters.calendar_id, self.Parameters.change_hour)
        event = calendar.get_event()
        if event is None:
            raise TaskFailure('Could not obtain current event from calendar')
        login = event['name'].replace('duty ', '').replace('@', '')

        staff = Staff(robot_report_duty_token)
        phone = staff.get_phone(login)
        if phone is None:
            raise TaskFailure('User ' + login + ' doesn\'t have phone specified')

        changer = Changer(robot_report_duty_token)

        if changer.get_number() != phone:
            changer.set_number(phone)
