# -*- coding: utf-8 -*-

import copy
import logging
import sys

import sandbox.common.types.client as ctc
import sandbox.projects.sandbox_ci.pulse.resources
from sandbox.projects.resource_types import (
    REPORT_TEMPLATES_PACKAGE,
    REPORT_NEWS_PACKAGE,
    UNISTAT_RESPONSE,
)
from sandbox.projects.report_renderer.parameters import ReportRendererBundlePackage
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.common.BaseTestTask import BaseDolbiloTask
from sandbox.projects.common.report_renderer import ReportRendererProvider
from sandbox.projects.common.report_renderer.utils import params_hash
from sandbox.projects.sandbox_ci.resources import (
    ReportWebTemplatesPackage,
    ReportMmImagesTemplatesPackage,
    ReportMmVideoTemplatesPackage,
    WebMicroPackage,
)


class BenchmarkPlan(parameters.ResourceSelector):
    name = 'benchmark_plan'
    description = 'Dolbilo plan'
    resource_type = sandbox.projects.sandbox_ci.pulse.resources.ReportRendererPlan


class Templates(parameters.ResourceSelector):
    name = 'report_templates_package'
    description = 'templates'
    resource_type = [
        REPORT_TEMPLATES_PACKAGE,
        REPORT_NEWS_PACKAGE,
        ReportMmImagesTemplatesPackage,
        ReportMmVideoTemplatesPackage,
        ReportWebTemplatesPackage,
        WebMicroPackage
    ]


class NUMANode(parameters.SandboxIntegerParameter):
    name = 'numa_node'
    description = 'numa node affinity, -1 for interleave (default)'
    default_value = -1


class SaveHeapDumps(parameters.SandboxBoolParameter):
    """
    Сохранять heap dump в начале и в конце работы бенчмарка
    """
    name = 'save_heap_dumps'
    description = 'Save heap snapshots at the start and at the end of work'
    default_value = False


class HeapDumpInterval(parameters.SandboxFloatParameter):
    """
    Интервалы сохранения heap snapshot'ов
    """
    name = 'heap_dump_interval'
    description = 'Save heap snapshots every %value% seconds. 0 = do not save heap snapshots.'


class DoNotCleanUp(parameters.SandboxBoolParameter):
    """
    Не очищать рабочую директорию
    """
    name = 'do_not_cleanup'
    description = 'Debug mode. Do not cleanup artifacts after the task has been completed'
    default_value = False


class BenchmarkReportRendererBase(BaseDolbiloTask):
    type = "BENCHMARK_REPORT_RENDERER_BASE"

    execution_space = 16384
    client_tags = ctc.Tag.LINUX_BIONIC & ctc.Tag.INTEL_E5_2660V1

    Provider = ReportRendererProvider

    input_parameters = [
        BenchmarkPlan, ReportRendererBundlePackage, Templates, NUMANode, SaveHeapDumps,
        HeapDumpInterval, DoNotCleanUp,
    ] + BaseDolbiloTask.input_parameters

    def initCtx(self):
        self.ctx.update({
            'total_sessions': 1,
            'executor_mode': 'finger',
            'fuckup_mode_max_simultaneous_requests': 1,
            'plan_mode_delay_between_requests_multipliers': 1.0,
        })

    def _start_searcher(self, callbacks=None):
        log_attrs = {param.name: self.ctx[param.name]
                     for param in [BenchmarkPlan, ReportRendererBundlePackage, Templates]}

        app = self.Provider(
            port=None,
            admin_port=None,
            report_renderer_resource_id=self.ctx[ReportRendererBundlePackage.name],
            templates_resource_id=self.ctx[Templates.name],
            task=self,
            numa_node=self.ctx[NUMANode.name],
            dump_start_end_heap_stats=self.ctx[SaveHeapDumps.name],
            heap_dump_interval=self.ctx[HeapDumpInterval.name],
            log_attrs=log_attrs,
        )

        self.renderer = app

        app.start()

        return [self.renderer.process]

    def _stop_searcher(self, subprocess_list, session_id=None, **kvargs):
        self.renderer.stop()

    def _get_searcher_port(self):
        return self.renderer.rr_port

    def benchmark(self):
        self.ctx['requests_per_sec'] = []

        params = copy.deepcopy(self.ctx)
        params['requests_per_sec'] = []
        self.run_dolbilo(params, 'benchmark', [], start_once=True)
        self.ctx['requests_per_sec'] = \
            self.ctx.get('requests_per_sec', []) + params['requests_per_sec']

    def on_execute(self):
        self.ctx['dolbilo_plan_resource_id'] = self.ctx.get(BenchmarkPlan.name)
        self.apps = {}

        logging.info('ctx# ' + str(self.ctx))

        try:
            with self.current_action('benchmarking'):
                self.benchmark()
        except Exception as benchmarking_err:
            traceback = sys.exc_info()[2]

            try:
                self.renderer.save_resources()
            except Exception as saving_err:
                logging.error('Unable to create resources with logs.\n%s', saving_err)

            raise Exception, benchmarking_err, traceback

        self.renderer.save_resources()
        self._save_resource_params_hash()

    def _save_resource_params_hash(self):
        found = channel.sandbox.list_resources(
            resource_type=UNISTAT_RESPONSE,
            task_id=self.id,
            status='READY',
            limit=1
        )[0]
        hash = params_hash(self.ctx)
        channel.sandbox.set_resource_attribute(found.id, 'params_hash', hash)

    def cleanup(self):
        if not self.ctx.get(DoNotCleanUp.name):
            BaseDolbiloTask.cleanup(self)
