# -*- coding: utf-8 -*-

import logging
import json
import requests

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.types.client import Tag

from sandbox.projects.resource_types import EXECUTOR_STAT

from sandbox.projects.report_renderer.resource_types import (
    REPORT_RENDERER_BUNDLE,
    AHPROXY_EXECUTABLE,
    AHPROXY_UNISTAT,
    AHPROXY_LOG
)
from sandbox.projects.common.dolbilka.resources import (
    DEXECUTOR_EXECUTABLE,
    DDUMPER_EXECUTABLE
)
from sandbox.projects.common.report_renderer import ReportRendererProvider
from sandbox.projects.sandbox_ci.resources.template_packages import WebMicroPackage
from sandbox.projects.sandbox_ci.pulse.resources import ReportRendererPlanApphost

AHPROXY_PORT = 13000
RR_ADMIN_PORT = 13001
RR_PORT = 13002
DOLBILO_OUT_BIN = './dolbilo-out.bin'


class BenchmarkReportRendererBaseApphost(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 32768
        client_tags = Tag.LINUX_BIONIC & Tag.INTEL_E5_2660V1

    class Parameters(sdk2.Task.Parameters):
        report_renderer_bundle = sdk2.parameters.Resource(
            'Report-renderer bundle',
            description='report-renderer bundle (ynode + report-renderer)',
            resource_type=REPORT_RENDERER_BUNDLE,
            required=True
        )
        templates = sdk2.parameters.Resource(
            'Templates',
            description='Микропакет с шаблонами',
            resource_type=WebMicroPackage,
            required=True
        )
        dolbilo_plan = sdk2.parameters.Resource(
            'Dolbilo plan',
            description='Ammo for dolbilo',
            resource_type=ReportRendererPlanApphost,
            required=True
        )
        ahproxy = sdk2.parameters.Resource(
            'ahproxy',
            description='ahproxy binary',
            resource_type=AHPROXY_EXECUTABLE,
            required=True
        )
        rr_workers = sdk2.parameters.Integer(
            'Amount of report-renderer workers',
            default=1
        )
        request_limit = sdk2.parameters.Integer(
            'Renderer request count',
            default_value=50000
        )
        save_heap_dumps = sdk2.parameters.Bool(
            'Save heap dumps',
            description='Save heap snapshots at the start and at the end of work',
            default=False
        )
        heap_dump_interval = sdk2.parameters.Integer(
            'Heap dump interval',
            description='Save heap snapshots every N seconds. 0 = do not save heap snapshots.',
            default=0
        )

    def on_execute(self):
        try:
            self._prepare_dolbilo()
            self._prepare_dolbilo_dumper()
            self._prepare_plan()
            self._prepare_ahproxy()
            self._start_ahproxy()
            self._start_renderer()
            self._shoot()
            self._process_shooting_results()
            self._renderer.stop()
            self._save_ahproxy_results()
        finally:
            if hasattr(self, '_renderer'):
                self._renderer.save_resources()

    def _prepare_dolbilo(self):
        resource = self._find_latest_resource(resource_type=DEXECUTOR_EXECUTABLE, attrs={'released': 'stable'})
        resource_data = sdk2.ResourceData(resource)
        self._dolbilo_path = resource_data.path

    def _prepare_dolbilo_dumper(self):
        resource = self._find_latest_resource(resource_type=DDUMPER_EXECUTABLE, attrs={'released': 'stable'})
        resource_data = sdk2.ResourceData(resource)
        self._dolbilo_dumper_path = resource_data.path

    def _prepare_plan(self):
        resource = self.Parameters.dolbilo_plan
        resource_data = sdk2.ResourceData(resource)
        self._plan_path = resource_data.path

    def _prepare_ahproxy(self):
        resource = self.Parameters.ahproxy
        resource_data = sdk2.ResourceData(resource)
        self._ahproxy_path = resource_data.path
        self._ahproxy_conf_path = './ahproxy.json'

        conf = {
            'ServerConfig': {
                'Port': AHPROXY_PORT,
                'Threads': 100,
                'StatsPrefix': 'ahproxy_'
            },
            'Backends': []
        }

        for i in range(0, self.Parameters.rr_workers):
            conf['Backends'].append('post://[::1]:{}'.format(RR_PORT + i))

        logging.debug('ahproxy configuration:')
        logging.debug(conf)

        with open('./ahproxy.json', 'wb') as f:
            f.write(json.dumps(conf))

    def _start_ahproxy(self):
        cmd = '{} {}'.format(self._ahproxy_path, self._ahproxy_conf_path)

        self._start_proc(cmd, 'ahproxy')

    def _save_ahproxy_results(self):
        ahproxy_unistat_path = './ahproxy_unistat.json'
        resource = AHPROXY_UNISTAT(
            self,
            'ahproxy unistat result for rr bundle #{} and ahproxy resource #{}'.format(
                self.Parameters.report_renderer_bundle.id,
                self.Parameters.ahproxy.id
            ),
            ahproxy_unistat_path
        )

        resp = requests.get('http://localhost:{}/remote_admin?action=mstat'.format(AHPROXY_PORT))
        with open(ahproxy_unistat_path, 'w') as f:
            f.write(resp.content)

        sdk2.ResourceData(resource).ready()

        resource = AHPROXY_LOG(
            self,
            'ahproxy unistat log for rr bundle #{} and ahproxy resource #{}'.format(
                self.Parameters.report_renderer_bundle.id,
                self.Parameters.ahproxy.id
            ),
            './ahproxy.out.txt'
        )
        sdk2.ResourceData(resource).ready()

    def _start_renderer(self):
        self._renderer = ReportRendererProvider(
            port=RR_PORT,
            admin_port=RR_ADMIN_PORT,
            report_renderer_resource_id=self.Parameters.report_renderer_bundle.id,
            templates_resource_id=self.Parameters.templates.id,
            task=self,
            dump_start_end_heap_stats=self.Parameters.save_heap_dumps,
            heap_dump_interval=self.Parameters.heap_dump_interval,
            apphost=True,
            workers=self.Parameters.rr_workers,
            log_attrs={
                'benchmark_plan': self.Parameters.dolbilo_plan.id,
                'REPORT_RENDERER_BUNDLE': self.Parameters.report_renderer_bundle.id,
                'report_templates_package': self.Parameters.templates.id,
            },
        )
        self._renderer.start()

    def _shoot(self):
        cmd = '{dolbilo} -H localhost -P {port} -m finger -p {plan} -s {workers} -Q {request_limit} -o {output}'.format(
            dolbilo=self._dolbilo_path,
            port=AHPROXY_PORT,
            plan=self._plan_path,
            # Max simultaneous requests ставим по количеству воркеров
            workers=self.Parameters.rr_workers,
            request_limit=self.Parameters.request_limit,
            output=DOLBILO_OUT_BIN
        )

        proc = self._start_proc(cmd, 'dolbilo')
        proc.wait()

    def _process_shooting_results(self):
        self._start_proc(
            '{dumper} -a -f {out_bin}'.format(
                dumper=self._dolbilo_dumper_path,
                out_bin=DOLBILO_OUT_BIN
            ),
            'd-dumper'
        ).wait()

        resource = EXECUTOR_STAT(
            self,
            'dolbilka result for rr bundle #{} and ahproxy resource #{}'.format(
                self.Parameters.report_renderer_bundle.id,
                self.Parameters.ahproxy.id
            ),
            './d-dumper.out.txt'
        )

        sdk2.ResourceData(resource).ready()

    # ReportRendererProvider is written for sdk1
    def create_resource(self, name, path, res, attributes=None):
        if attributes is None:
            attributes = {}
        return res(self, name, path, **attributes)

    # ReportRendererProvider is written for sdk1
    def mark_resource_ready(self, res):
        sdk2.ResourceData(res).ready()

    def _find_latest_resource(self, resource_type, **kwargs):
        return resource_type.find(**kwargs).order(-sdk2.Resource.id).first()

    def _start_proc(self, cmd, logname):
        with open('{}.out.txt'.format(logname), 'w') as logfile:
            return sp.Popen(cmd, shell=True, stdout=logfile, stderr=logfile)
