# -*- coding: utf-8 -*-
import re
import sys
import os
import logging

from sandbox.common.errors import TaskFailure
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.common.report_renderer import ReportRendererProvider
from sandbox.projects.common import file_utils as fu

CUSTOM_EXTENSION = """
'use strict';
const fs = require('fs');
const http = require('http');
const path = require('path');
module.exports = {
    configure: function(config, proc) {
        if (!proc.isMaster) {
            return;
        }
        const logsDir = proc.config.get('properties.logs.dir');
        const logsPath = path.join(logsDir, 'master-memory.log');
        const fd = fs.openSync(logsPath, 'w');
        let measurementsHandler;

        proc.on('running', () => {
            const start = process.hrtime();
            const measure = () => {
                global.gc();
                global.gc();
                const now = process.hrtime(start);
                fs.write(fd, (now[0] + now[1] / 1e9) + ' ' + process.memoryUsage().heapUsed + '\\n', () => {});
            };
            measure();
            measurementsHandler = setInterval(measure, MEASUREMENT_INTERVAL * 1000);
        });

        proc.on('shutdown', () => {
            clearInterval(measurementsHandler);
        });
    }
};
"""

CUSTOM_CONFIG = """
'use strict';
const config = require('./config.js');
config.extensions['master_memory_benchmark_extension'] = {};
module.exports = config;
"""


class BenchmarkReportRendererMasterMemoryProvider(ReportRendererProvider):
    """
    Запуск report-renderer для бенчмарка памяти мастер-процесса
    Добавляет кастомное расширение мастера
    """

    def _replace_config(self, args):
        matcher = r'[^\s]*config\.js'

        search = re.search(matcher, args)
        if search is None:
            raise TaskFailure('Could not replace config in args {}'.format(args))

        org_path = os.path.dirname(search.group(0))
        config_file = os.path.join(org_path, 'master_memory_benchmark_config.js')
        extension_file = os.path.join(org_path, 'build/src/luster-extensions/master_memory_benchmark_extension.js')
        fu.write_file(config_file, CUSTOM_CONFIG)
        extension_text = re.sub(r'MEASUREMENT_INTERVAL', str(self.task.ctx['measurement_interval']), CUSTOM_EXTENSION)
        fu.write_file(extension_file, extension_text)

        return re.sub(matcher, config_file, args)

    @staticmethod
    def _expose_gc(args):
        args = '--expose_gc {}'.format(args)
        return args

    def get_run_args(self):
        args = super(BenchmarkReportRendererMasterMemoryProvider, self).get_run_args()
        args = self._replace_config(args)
        args = self._expose_gc(args)
        return args

    def get_custom_log_file(self):
        return os.path.join(self.rr_logs, 'master-memory.log')

    def _get_heap_snapshot_url(self):
        return 'http://localhost:{}/admin?action=heapsnapshot&target=master'.format(self.rr_admin_port)

    def start(self):
        super(BenchmarkReportRendererMasterMemoryProvider, self).start()
        requester_path = os.path.join(os.path.dirname(__file__), 'unistat_requester.py')
        logging.info("Starting unistat requester")
        self.unistat_process = run_process(
            [sys.executable, requester_path, str(self.rr_admin_port), str(self.task.ctx['stats_request_interval'])],
            outputs_to_one_file=True,
            log_prefix='unistat_requester',
            wait=False)
        logging.info("Unistat requester has started")

    def stop(self):
        logging.info("Stopping unistat requester")
        self.unistat_process.kill()
        self.unistat_process = None
        logging.info("Unistat requester is stopped")
        super(BenchmarkReportRendererMasterMemoryProvider, self).stop()
