# -*- coding: utf-8 -*-

import logging
import os
import tarfile

from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.nanny import const
from sandbox.projects.report_renderer.parameters import NodeJSPackage, ReportRendererPackage
from sandbox.projects.report_renderer.resource_types import REPORT_RENDERER_BUNDLE
from sandbox.projects.resource_types import ARCADIA_PROJECT

TAR_NAME = 'report-renderer.tar'

EXTENSIONS = [
    {
        "name": "apphost",
        "build_target": "serp/ynode/extensions/apphost/dynamic",
        "package_target": "report-renderer/node_modules/apphost",
    },
    {
        "name": "highlighter",
        "build_target": "serp/ynode/extensions/highlighter/dynamic",
        "package_target": "report-renderer",
    },
    {
        "name": "safeappend",
        "build_target": "serp/ynode/extensions/safeappend/dynamic",
        "package_target": "report-report-renderer/node_modules/safeappend",
    },
    {
        "name": "signurl",
        "build_target": "serp/ynode/extensions/signurl/dynamic",
        "package_target": "report-renderer",
    },
    {
        "name": "svnversion",
        "build_target": "serp/ynode/extensions/svnversion/dynamic",
        "package_target": "report-renderer",
    },
]

# param value must match output param name
PLATFORMS = [
    {
        "target_arch": "linux",
        "lto": True,
        "build_filename": lambda e: "lib{}.node.so".format(e["name"]),
        "package_filename": lambda e: "{}-x64-linux.node".format(e["name"]),
    },
    {
        "target_arch": "darwin",
        "lto": False,
        "build_filename": lambda e: "lib{}.node.dylib".format(e["name"]),
        "package_filename": lambda e: "{}-x64-darwin.node".format(e["name"]),
    }
]


class BuildReportRendererBundleSdk2(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
        Таск сборки бандла report-renderer. Содержит аркадийную node.js и сам report-renderer
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 3 * 1024

    class Parameters(sdk2.Task.Parameters):
        renderer_package = ReportRendererPackage(
            required=True,
        )
        nodejs_package = NodeJSPackage(
            required=True,
        )
        build_dso = sdk2.parameters.Bool(
            'Extensions as DSO',
            description='Собирать нативные расширения Node.js как DSO',
            default=False,
        )
        release_to = ['report-renderer']
        release_ticket_key = sdk2.parameters.String(
            'Startrek release ticket key',
            description='Will be linked to nanny release',
            required=False,
        )

    @staticmethod
    def create_tar(filenames, tarname=TAR_NAME):
        with tarfile.open(tarname, 'w') as tar:
            for fname in filenames:
                tar.add(fname)

    @staticmethod
    def untar_archive(filename):
        filename = str(filename)
        with tarfile.open(filename) as tar:
            tar.extractall()
        os.remove(filename)

    def create_bundle(self):
        nodejs_resource = self.Parameters.nodejs_package
        rr_resource = self.Parameters.renderer_package

        for resource in [
            rr_resource,
            nodejs_resource,
        ]:
            resource_path = sdk2.ResourceData(resource).path
            sdk2.paths.copy_path(str(resource_path), str(self.path()))
            local_path = resource_path.name
            self.untar_archive(local_path)

        if self.Parameters.build_dso:
            for extension_subtask_id in self.Context.build_extension_subtask_ids:
                extension_task = sdk2.Task[extension_subtask_id]
                if extension_task is None:
                    raise "Subtask {} not found".format(extension_subtask_id)
                extension_resource = ARCADIA_PROJECT.find(
                    task_id=extension_subtask_id,
                ).first()
                if extension_resource is None:
                    raise "{} resource not found in subtask {}".format(ARCADIA_PROJECT.name, extension_subtask_id)

                # extension_task is YA_PACKAGE on sdk1, all parameters is in Context
                extension_task_target = extension_task.Context.targets
                extension_task_platform = extension_task.Context.target_platform

                logging.info('Found DSO resource %s for extension %s arch %s', extension_resource.id,
                             extension_task_target, extension_task_platform)

                extension = next(e for e in EXTENSIONS if e["build_target"] == extension_task_target)
                if extension is None:
                    raise "Extension not found for task {} resource {}".format(extension_task.id,
                                                                               extension_resource.id)

                platform = next(p for p in PLATFORMS if p["target_arch"] == extension_task_platform)
                if platform is None:
                    raise "Platform not found for task {} resource {} (arch {})".format(extension_task.id,
                                                                                        extension_resource.id,
                                                                                        extension_task_platform)

                package_target = extension["package_target"]
                platform_target = platform["package_filename"](extension)
                target_path = str(self.path() / package_target / platform_target)
                logging.info('Saving resource %s to %s', extension_resource.id, target_path)

                resource_path = sdk2.ResourceData(extension_resource).path
                sdk2.paths.copy_path(str(resource_path), str(self.path() / target_path))

        self.create_tar(['report-renderer', 'ynode'])

        nodejs_version = nodejs_resource.resource_version
        self.Context.nodejs_version = nodejs_version

        rr_version = rr_resource.version
        self.Context.rr_version = rr_version

        # TODO use new report_renderer_version attribute and remove this one
        legacy_attrs = {
            'report-renderer_version': rr_version
        }

        bundle_resource = REPORT_RENDERER_BUNDLE(
            self,
            'Bundle (report-renderer: {}, node.js: {}, DSO: {})'.format(
                rr_version,
                nodejs_version,
                self.Parameters.build_dso,
            ),
            TAR_NAME,
            arch=nodejs_resource.arch,
            platform=nodejs_resource.platform,
            nodejs_version=nodejs_version,
            report_renderer_version=rr_version,
            dso=self.Parameters.build_dso,
            **legacy_attrs
        )

        sdk2.ResourceData(bundle_resource).ready()

    def on_execute(self):
        self.create_bundle()

    def on_release(self, additional_parameters):
        if self.Parameters.release_ticket_key:
            setattr(self.Context, const.STARTREK_TICKET_IDS_KEY, [self.Parameters.release_ticket_key])
        additional_parameters[const.RELEASE_SUBJECT_KEY] = 'Report renderer {} with ynode {}'.format(
            self.Context.rr_version, self.Context.nodejs_version
        )
        additional_parameters[const.RELEASE_COMMENTS] = self.Parameters.description
        super(BuildReportRendererBundleSdk2, self).on_release(additional_parameters)
