import sys
import json
import logging
import os.path

from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
import sandbox.projects.common.constants as consts
from sandbox.projects.common.nanny import nanny
from sandbox.projects import resource_types
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter


def _make_url(base_url, subpath, rev):
    return "%s/%s@%s" % (base_url, subpath, rev)


class BuildReportRendererConfigs(nanny.ReleaseToNannyTask, ArcadiaTask):
    """
        Builds a report-renderer configs (loop.conf's, logmux and push-client)
    """

    type = "BUILD_REPORT_RENDERER_CONFIGS"

    execution_space = 512

    input_parameters = [
        SandboxArcadiaUrlParameter
    ]

    result_configs = {
        "production_web_loop.conf": resource_types.REPORT_RENDERER_PRODUCTION_LOOP_CONFIG,
        "testing_web_loop.conf": resource_types.REPORT_RENDERER_TESTING_LOOP_CONFIG,
        "production_web_push-client.yml": resource_types.REPORT_RENDERER_PUSH_CLIENT_CONFIG,
    }

    release_to = ['report-renderer']

    def do_execute(self):
        url = self.ctx[consts.ARCADIA_URL_KEY]
        parsed_url = Arcadia.parse_url(url)
        assert parsed_url.revision
        base_url = url.split("@")[0] + "/serp/admin"

        pva_dir = os.path.abspath("pva_dir")
        Arcadia.export(
            _make_url(base_url, ".", parsed_url.revision),
            pva_dir)

        generated_dir = os.path.abspath("generated")
        make_folder(generated_dir)
        os.chdir(pva_dir)

        pva_path = os.path.join(pva_dir, "pva.py")
        pva_config = os.path.join(pva_dir, "config.report-renderer.json")

        logging.debug("Config: {}".format(json.load(open(pva_config))))
        try:
            run_process(
                [
                    sys.executable,
                    pva_path,
                    pva_config,
                    generated_dir],
                outputs_to_one_file=False,
                log_prefix="pva")
        except:
            logging.error("Unexpected error:", sys.exc_info()[0])
            raise

        for config_name, config_type in self.result_configs.iteritems():
            self.create_resource(
                description=self.descr,
                resource_path="{}/{}".format(generated_dir, config_name),
                resource_type=config_type,
                arch="any")


__Task__ = BuildReportRendererConfigs
