# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.common.types import (
    task as ctt,
)
from sandbox.projects.common.build import YaPackage
from sandbox.projects.common.build.YaPackage import PackageResourceAttrsParameter, TargetPlatformParameter
from sandbox.projects.common.build.parameters import (
    ArcadiaPatch,
    BuildType,
    UseArcadiaApiFuse,
    DoNotRemoveResources
)
from sandbox.projects.common.constants import CHECKOUT_MODE, CHECKOUT_MODE_AUTO
from sandbox.projects.common.utils import check_subtasks_fails
from sandbox.projects.report_renderer.common.arcadia import fix_arcadia_url
from sandbox.projects.report_renderer.resource_types import REPORT_RENDERER_NODEJS_PACKAGE
from sandbox.sdk2.vcs.svn import Arcadia

# param value must match output param name
PLATFORMS = [
    {
        "target_arch": "linux",
        "param": "ynode_linux",
    },
    {
        "target_arch": "darwin",
        "param": "ynode_macos",
    }
]


class BuildYnode(sdk2.Task):
    """
        Таск сборки ynode под все необходимые платфоры
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 3 * 1024

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            required=True,
        )
        arcadia_patch = ArcadiaPatch()
        build_type = BuildType()

        with sdk2.parameters.Output():
            ynode_linux = sdk2.parameters.Resource('ynode package for Linux')
            ynode_macos = sdk2.parameters.Resource('ynode package for macOS')

        release_to = ['report-renderer']

    def create_build_ynode_package_subtask(self, arcadia_url, target_arch):
        parsed_arcadia_url = Arcadia.parse_url(arcadia_url)

        return sdk2.Task["YA_PACKAGE"](
            self,
            description="{} #{} subtask (building ynode for {} from {})".format(
                self.type,
                self.id,
                target_arch,
                arcadia_url
            ),
            **{
                sdk2.parameters.ArcadiaUrl.name: arcadia_url,
                ArcadiaPatch.name: self.Parameters.arcadia_patch,
                UseArcadiaApiFuse.name: True,
                CHECKOUT_MODE: CHECKOUT_MODE_AUTO,

                DoNotRemoveResources.name: True,
                YaPackage.PackagesParameter.name: 'serp/ynode/packages/ynode.json',
                YaPackage.ResourceTypeParameter.name: REPORT_RENDERER_NODEJS_PACKAGE.name,
                YaPackage.PackageTypeParameter.name: 'tarball',
                BuildType.name: self.Parameters.build_type,
                TargetPlatformParameter.name: target_arch,

                YaPackage.PublishPackageParameter.name: False,
                YaPackage.UseNewFormatParameter.name: True,
                YaPackage.CompressPackageArchiveParameter.name: True,
                YaPackage.SaveBuildOutputParameter.name: True,
                PackageResourceAttrsParameter.name: {
                    'svn_path': arcadia_url,
                    'arcadia_patch': self.Parameters.arcadia_patch,
                    'branch': parsed_arcadia_url.branch,
                    'tag': parsed_arcadia_url.tag,
                    'revision': parsed_arcadia_url.revision,
                    # TODO check if needed
                    'ttl': 'inf',
                    # TODO remove this attr when YA_PACKAGE will set proper attrs and templar will not use it for filter
                    'arch': target_arch,
                    'target_arch': target_arch,
                    'build_type': self.Parameters.build_type,
                }
            }
        )

    def on_execute(self):
        with self.memoize_stage.build(commit_on_entrance=False):
            fixed_arcadia_url = fix_arcadia_url(self.Parameters.arcadia_url)
            logging.info('Fixing arcadia URL at %s', fixed_arcadia_url)

            build_subtasks = self.Context.build_subtasks = []
            for target_arch in map(lambda p: p["target_arch"], PLATFORMS):
                logging.info('Building %s for %s', REPORT_RENDERER_NODEJS_PACKAGE.name, target_arch)
                build_subtask = self.create_build_ynode_package_subtask(fixed_arcadia_url, target_arch)
                build_subtask.enqueue()
                build_subtasks.append(build_subtask.id)

            raise sdk2.WaitTask(
                tasks=build_subtasks,
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
            )

        check_subtasks_fails()

        for subtask_id in self.Context.build_subtasks:
            ynode_resource = REPORT_RENDERER_NODEJS_PACKAGE.find(
                task_id=subtask_id,
            ).first()
            if ynode_resource is None:
                raise "{} resource not found in subtask {}".format(REPORT_RENDERER_NODEJS_PACKAGE.name, subtask_id)

            logging.info('Found ynode resource %s for arch %s', ynode_resource.id, ynode_resource.target_arch)

            platform = next(p for p in PLATFORMS if p["target_arch"] == ynode_resource.target_arch)
            if platform is None:
                raise "Platform not found for resource {} (arch {})".format(ynode_resource.id, ynode_resource.target_arch)

            output_param = platform["param"]
            logging.info('Saving resource %s to output param %s', ynode_resource.id, output_param)
            setattr(self.Parameters, output_param, ynode_resource)
