# -*- coding: utf-8 -*-

from collections import namedtuple
from functools import partial
import json
import os
import logging

import jinja2
from sandbox import sdk2
from sandbox.common.types import (
    task as ctt,
)
from sandbox.common.types.client import Tag
from sandbox.common.utils import singleton_property
from sandbox.projects.common.dolbilka import stats_parser
from sandbox.projects.common.utils import check_subtasks_fails
from sandbox.projects.common.BaseTestTask import BaseDolbiloTask
from sandbox.projects.report_renderer.parameters import ReportRendererBundlePackage
from sandbox.projects.report_renderer.BenchmarkReportRendererBase.task import (
    BenchmarkReportRendererBase,
    BenchmarkPlan,
    Templates,
)
from sandbox.projects.report_renderer.BenchmarkReportRendererBaseApphost import BenchmarkReportRendererBaseApphost
from sandbox.projects.resource_types import (
    UNISTAT_RESPONSE,
    REPORT_RENDERER_BENCHMARK_COMPARISON_RESULT,
    EXECUTOR_STAT
)
from sandbox.projects.report_renderer.resource_types import (
    AHPROXY_UNISTAT,
    AHPROXY_EXECUTABLE,
)
from sandbox.projects.sandbox_ci.pulse.resources import (
    ReportRendererPlan,
    ReportRendererPlanApphost
)
from sandbox.projects.sandbox_ci.resources.template_packages import WebMicroPackage

from sandbox.sandboxsdk.environments import PipEnvironment


def read_unistat(filename):
    with open(str(filename)) as uni_data:
        return dict(json.load(uni_data))

SignalDescription = namedtuple('SignalDescription', ['title', 'xlabel', 'ylabel'])

APPHOST_SUFFIX = '_apphost'


def get_signal_description(name):
    if "render" in name:
        return SignalDescription(
            title='Number of responses by response time ({})'.format(name),
            xlabel='Response time, us',
            ylabel='# of requests',
        )
    elif "bemhtml_template" in name:
        return SignalDescription(
            title='Number of responses by BEMHTML time ({})'.format(name),
            xlabel='BEMHTML time, us',
            ylabel='# of requests',
        )
    elif "priv_template" in name:
        return SignalDescription(
            title='Number of responses by priv time ({})'.format(name),
            xlabel='priv time, us',
            ylabel='# of requests',
        )
    elif "request_body_parse" in name:
        return SignalDescription(
            title='Number of responses by body parse time ({})'.format(name),
            xlabel='body parse time, us',
            ylabel='# of requests',
        )
    else:
        return SignalDescription(
            title=name,
            xlabel='value',
            ylabel='# of samples',
        )


def make_plot(name, base, compared, root_path):
    import matplotlib.pyplot as plt

    out_filename = str(root_path / "comparison_{}.svg".format(name))

    fig, ax = plt.subplots()

    plot_base = ax.plot([v[0] for v in base], [v[1] for v in base], color='r')
    plot_compared = ax.plot([v[0] for v in compared], [v[1] for v in compared], color='b')

    descr = get_signal_description(name)
    ax.set_title(descr.title)
    ax.set_xlabel(descr.xlabel)
    ax.set_ylabel(descr.ylabel)
    ax.legend((plot_base[0], plot_compared[0]), ("base", "compared"))

    plt.savefig(out_filename, format='svg')


BundleProps = namedtuple('BundleProps', ['bundle', 'ctx_task_key'])


class CompareReportRendererBenchmarkSdk2(sdk2.Task):
    class Requirements(sdk2.Requirements):
        # Требуемые для построения графика wheel-пакеты есть только для ubuntu 12.04
        client_tags = Tag.LINUX_PRECISE
        environments = (
            PipEnvironment("matplotlib", '1.5.1', use_wheel=True),
        )

    class Parameters(sdk2.Task.Parameters):
        base_bundle = ReportRendererBundlePackage(
            'Base report-renderer bundle (ynode + report-renderer)',
            required=True,
        )
        compared_bundle = ReportRendererBundlePackage(
            label='Compared report-renderer bundle (ynode + report-renderer)',
            required=True,
        )

        request_limit = sdk2.parameters.Integer(
            label=BaseDolbiloTask.RequestsLimit.description,
            description="Renderer request count",
            default_value=50000,
        )

        rr_workers = sdk2.parameters.Integer(
            'Renderer workers (apphost mode)',
            description="Renderer workers amount for apphost shooting",
            default_value=8,
        )

    @property
    def bundles(self):
        return [
            BundleProps(
                bundle=self.Parameters.base_bundle,
                ctx_task_key='base_bundle_task_id',
            ),
            BundleProps(
                bundle=self.Parameters.compared_bundle,
                ctx_task_key='compared_bundle_task_id',
            ),
        ]

    @singleton_property
    def _web4_templates(self):
        return WebMicroPackage.find(
            attrs={
                'type': 'web4-micropackage',
                'released': 'stable',
                'YENV': 'production',
            }
        ).first()

    plan_common_attrs = {
        'project': 'web4',
        'platform': 'desktop',
        'released': 'stable',
    }

    def get_web4_plan(self):
        return ReportRendererPlan.find(
            attrs=self.plan_common_attrs
        ).first()

    def get_web4_plan_apphost(self):
        return ReportRendererPlanApphost.find(
            attrs=self.plan_common_attrs
        ).first()

    def get_ahproxy(self):
        return AHPROXY_EXECUTABLE.find(
            attrs={'released': 'stable'}
        ).first()

    def get_dolbilo_shooting_params(self, rr_bundle, plan, templates):
        return {
            # BenchmarkReportRendererBase params
            BenchmarkPlan.name: plan.id,
            ReportRendererBundlePackage.name: rr_bundle.id,
            Templates.name: templates.id,

            # BaseDolbiloTask
            BaseDolbiloTask.RequestsLimit.name: self.Parameters.request_limit,
            BaseDolbiloTask.ExecutorMode.name: 'finger',
            BaseDolbiloTask.FuckupModeMaxSimultaneousRequests.name: 4,
            BaseDolbiloTask.Circular.name: True,
        }

    def get_apphost_shooting_params(self, rr_bundle, plan, templates, ahproxy):
        return {
            # BenchmarkReportRendererBaseApphost params
            'report_renderer_bundle': rr_bundle.id,
            'dolbilo_plan': plan.id,
            'templates': templates.id,
            'rr_workers': self.Parameters.rr_workers,
            'ahproxy': ahproxy.id,
            'request_limit': self.Parameters.request_limit
        }

    def start_shooting(self, params):
        task = sdk2.Task[BenchmarkReportRendererBase.type](
            self,
            description="Benchmark of bundle {}".format(
                params[ReportRendererBundlePackage.name],
            ),
            **params
        )
        task.enqueue()
        return task

    def start_shooting_apphost(self, params):
        task = BenchmarkReportRendererBaseApphost(
            self,
            description="Benchmark of bundle {}".format(
                params['report_renderer_bundle'],
            ),
            **params
        )
        task.enqueue()
        return task

    def create_shooting_tasks(self):
        plan = self.get_web4_plan()  # TODO implement custom plan
        logging.info('Using plan {}'.format(plan))
        templates = self._web4_templates  # TODO implement custom templates
        logging.info('Using templates {}'.format(templates))

        tasks = []
        for rr_bundle_props in self.bundles:
            rr_bundle = rr_bundle_props.bundle
            ctx_key = rr_bundle_props.ctx_task_key
            shooting_params = self.get_dolbilo_shooting_params(rr_bundle, plan, templates)
            logging.info('Creating tasks with {}'.format(str(shooting_params)))
            task = self.start_shooting(shooting_params)
            tasks.append(task)
            setattr(self.Context, ctx_key, task.id)
        return tasks

    def create_apphost_shooting_tasks(self):
        plan = self.get_web4_plan_apphost()
        ahproxy = self.get_ahproxy()
        tasks = []
        for rr_bundle_props in self.bundles:
            rr_bundle = rr_bundle_props.bundle
            ctx_key = rr_bundle_props.ctx_task_key + APPHOST_SUFFIX
            shooting_params = self.get_apphost_shooting_params(rr_bundle, plan, self._web4_templates, ahproxy)
            logging.info('Creating tasks with {}'.format(str(shooting_params)))
            task = self.start_shooting_apphost(shooting_params)
            tasks.append(task)
            setattr(self.Context, ctx_key, task.id)
        return tasks

    def get_shooting_task_id(self, bundle_props, suffix):
        ctx_key = bundle_props.ctx_task_key + suffix
        return getattr(self.Context, ctx_key)

    def get_shooting_result(self, bundle_props, suffix, resource_type):
        shooting_task_id = self.get_shooting_task_id(bundle_props, suffix)
        return resource_type.find(
            task_id=shooting_task_id,
        ).first()

    def get_shooting_result_or_raise(self, bundle_props, suffix, resource_type):
        res = self.get_shooting_result(bundle_props, suffix, resource_type)

        if res is None:
            shooting_task_id = self.get_shooting_task_id(bundle_props, suffix)
            raise Exception("Resource of type {} not found for task {}".format(resource_type, shooting_task_id))

        return res

    @staticmethod
    def get_screen_name(rr_bundle):
        return '{}_{}'.format(
            rr_bundle.report_renderer_version,
            rr_bundle.nodejs_version,
        )

    def on_execute(self):
        with self.memoize_stage.shoot:
            shooting_tasks = self.create_shooting_tasks()
            shooting_tasks += self.create_apphost_shooting_tasks()
            raise sdk2.WaitTask(
                tasks=shooting_tasks,
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
            )

        with self.memoize_stage.compare:
            check_subtasks_fails()

            if not os.path.isdir('result'):
                os.mkdir('result')

            [base_result, compared_result] = map(
                partial(self.get_shooting_result_or_raise, suffix='', resource_type=UNISTAT_RESPONSE),
                self.bundles
            )
            self._make_unistat_report(base_result, compared_result, mode='http')

            [apphost_base_result, apphost_compared_result] = map(
                partial(self.get_shooting_result_or_raise, suffix=APPHOST_SUFFIX, resource_type=UNISTAT_RESPONSE),
                self.bundles
            )
            self._make_unistat_report(apphost_base_result, apphost_compared_result, mode='apphost')

            [ahproxy_base_result, ahproxy_compared_result] = map(
                partial(self.get_shooting_result_or_raise, suffix=APPHOST_SUFFIX, resource_type=AHPROXY_UNISTAT),
                self.bundles
            )
            self._make_unistat_report(ahproxy_base_result, ahproxy_compared_result, mode='ahproxy', required_signals=[])

            result = REPORT_RENDERER_BENCHMARK_COMPARISON_RESULT(
                self,
                'Result of comparison {} vs {}'.format(
                    self.get_screen_name(self.Parameters.base_bundle),
                    self.get_screen_name(self.Parameters.compared_bundle),
                ),
                'result/',
            )
            sdk2.ResourceData(result).ready()

            [http_base_dolbilka_result, http_compared_dolbilka_result] = map(
                partial(self.get_shooting_result_or_raise, suffix='', resource_type=EXECUTOR_STAT),
                self.bundles
            )
            self._make_dolbilka_report('http', http_base_dolbilka_result, http_compared_dolbilka_result)

            [apphost_base_dolbilka_result, apphost_compared_dolbilka_result] = map(
                partial(self.get_shooting_result_or_raise, suffix=APPHOST_SUFFIX, resource_type=EXECUTOR_STAT),
                self.bundles
            )
            self._make_dolbilka_report('apphost', apphost_base_dolbilka_result, apphost_compared_dolbilka_result)

            logging.info('Finished successfully')

    def _make_unistat_report(self, base_result, compared_result, mode, required_signals=('render_dhhh',)):
        logging.info('[{}] Comparing base result {} vs compared result {}'.format(
            mode,
            base_result.id,
            compared_result.id
        ))

        [base_result_data, compared_result_data] = map(
            lambda r: read_unistat(sdk2.ResourceData(r).path),
            [base_result, compared_result]
        )

        # TODO build more detailed comparison via profile log
        missing = []
        for name in required_signals:
            for (data, target) in [(base_result_data, 'base'), (compared_result_data, 'compared')]:
                if name not in data:
                    missing.append("{} in {}".format(name, target))

        all_signals = set(base_result_data.keys() + compared_result_data.keys())

        logging.info('Found signals: {}'.format(all_signals))

        all_hist_names = filter(
            lambda n: n.endswith('hhh'),
            set(base_result_data.keys() + compared_result_data.keys())
        )

        logging.info('Found histograms: {}'.format(all_hist_names))

        if len(all_hist_names) == 0:
            raise Exception("No histograms were found in unistat responses")

        logging.info('Generating graphs')
        for name in all_hist_names:
            if name not in base_result_data:
                logging.info('Histogram {} not found in base results, skipping'.format(name))
                continue
            if name not in compared_result_data:
                logging.info('Histogram {} not found in compared results, skipping'.format(name))
                continue
            make_plot(
                '{}_{}'.format(mode, name),
                base_result_data[name],
                compared_result_data[name], self.path() / "result"
            )

        if len(missing) > 0:
            raise Exception("Missing signals:\n" + "\n".join(missing))

    def _make_dolbilka_report(self, kind, base_dolbilka_result, compared_dolbilka_result):
        if 'dolbilka_status_reports' not in self.Context:
            self.Context.dolbilka_status_reports = {}

        context_statuses = self.Context.dolbilka_status_reports
        [parsed_base, parsed_compared] = map(
            lambda r: stats_parser.StatsParser(sdk2.ResourceData(r).path).response_statuses,
            [base_dolbilka_result, compared_dolbilka_result]
        )

        if parsed_base != parsed_compared:
            statuses_data = {
                'compare_status': 'error',
                'blocks': {
                    'base': parsed_base,
                    'compared': parsed_compared,
                }
            }
        elif parsed_base.keys() != ['Ok']:
            statuses_data = {
                'compare_status': 'warning',
                'blocks': {'': parsed_base}
            }
        else:
            statuses_data = {
                'compare_status': 'ok',
                'blocks': {'': parsed_base}
            }
        context_statuses[kind] = statuses_data

    @property
    def footer(self):
        def render_bundle_result(rr_bundle_props, suffix):
            executor_stat_resource = self.get_shooting_result(rr_bundle_props, suffix, EXECUTOR_STAT)
            bundle_name = self.get_screen_name(rr_bundle_props.bundle)

            if executor_stat_resource is None:
                return '<span>Executor stat for {name} is not ready</span>'.format(
                    name=bundle_name,
                )
            return '<div><a href="{href}">Executor stat for {name}</a></div>'.format(
                href=executor_stat_resource.http_proxy,
                name=bundle_name,
            )

        def render_dolbilka_status_reports():
            template_path = os.path.dirname(os.path.abspath(__file__))
            env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path), extensions=["jinja2.ext.do"])
            return env.get_template("dolbilka_statuses.html").render({'reports': self.Context.dolbilka_status_reports})

        def render_charts():
            result = REPORT_RENDERER_BENCHMARK_COMPARISON_RESULT.find(
                task_id=self.id,
            ).first()

            if not result:
                return 'Comparison charts not ready'

            return """
            <div><img alt="Время рендера http" src="{http_proxy}/comparison_http_render_dhhh.svg"/></div>
            <div><img alt="Время рендера apphost" src="{http_proxy}/comparison_apphost_render_dhhh.svg"/></div>
            <div><img alt="Время ответа, измеренное apphost-библиотекой" src="{http_proxy}/comparison_apphost_context.total_dhhh.svg"/></div>
            <div><img alt="Время ответа ahproxy" src="{http_proxy}/comparison_ahproxy_ahproxy_answer_time_dhhh.svg"/></div>
            <div><a href="{http_proxy}">All comparison charts</a></div>
            """.format(
                http_proxy=result.http_proxy,
            )

        def render_bundle_results():
            http_bundle_results = map(
                partial(render_bundle_result, suffix=''),
                self.bundles,
            )
            apphost_bundle_results = map(
                partial(render_bundle_result, suffix=APPHOST_SUFFIX),
                self.bundles,
            )
            return """
                <h4>Http</h4>
                {http_bundle_results}
                <h4>Apphost</h4>
                {apphost_bundle_results}
            """.format(
                http_bundle_results='\n'.join(http_bundle_results),
                apphost_bundle_results='\n'.join(apphost_bundle_results),
            )

        try:
            return """
                <h3>Response statuses</h3>
                {dolbilka_status_reports}
                <h3>Response time comparison charts</h3>
                {charts}
                <h3>Executor stats</h3>
                {bundle_results}
            """.format(
                dolbilka_status_reports=render_dolbilka_status_reports(),
                charts=render_charts(),
                bundle_results=render_bundle_results(),
            )

        except Exception as e:
            return "Exception in footer: {}".format(e)
