# -*- coding: utf-8 -*-

import re
import logging

from sandbox import sdk2

from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.projects.report_renderer.BuildReportRendererBundleSdk2 import BuildReportRendererBundleSdk2
from sandbox.projects.report_renderer.CompareReportRendererBenchmarkSdk2 import CompareReportRendererBenchmarkSdk2
from sandbox.projects.report_renderer.parameters import ReportRendererBundlePackage
from sandbox.projects.report_renderer.common.tasks import get_all_ancestors

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.logs.TestReportLogs import LogName, InputData

from sandbox.projects.report_renderer.resource_types import (
    REPORT_RENDERER_BUNDLE,
    REPORT_RENDERER_NODEJS_PACKAGE,
)
from sandbox.projects.resource_types import \
    REPORT_RENDERER_BENCHMARK_COMPARISON_RESULT, \
    REPORT_RENDERER_MEMORY_BENCHMARK_RESULT, \
    REPORT_RENDERER_MASTER_MEMORY_BENCHMARK_RESULT, \
    REPORT_RENDERER_BLOCKSTAT_LOG, REPORT_RENDERER_PACKAGE
from sandbox.projects.sandbox_ci.search_interfaces_integration_tests import SearchInterfacesIntegrationTests

GENERATED_DESCRIPTION_START = u'!!Release tasks will update following text. Do not edit or remove it or this marker.!!\n'
RELEASE_URL_BASE = '{}/tags/report-renderer'.format(Arcadia.ARCADIA_BASE_URL)


class ReleaseReportRenderer(sdk2.Task):
    """
    Шаблонные действия при релизе report-renderer
    Описание будет скопировано в создаваемый тикет
    """
    RELEASE_URL_BASE = RELEASE_URL_BASE

    class Requirements(sdk2.Task.Requirements):
        environments = [PipEnvironment('startrek_client', use_wheel=True)]

    class Parameters(sdk2.Task.Parameters):
        renderer_package = sdk2.parameters.LastResource(
            'Renderer to include in bundle',
            resource_type=REPORT_RENDERER_PACKAGE,
            required=True,
            attrs={
                'released': 'stable',
            },
        )
        nodejs_package = sdk2.parameters.LastResource(
            'Ynode to include in bundle',
            resource_type=REPORT_RENDERER_NODEJS_PACKAGE,
            required=True,
            attrs={
                'released': 'stable',
                'platform': 'linux',
            },
        )
        compare_vs_bundle = sdk2.parameters.LastReleasedResource(
            'Bundle to compare response times',
            resource_type=REPORT_RENDERER_BUNDLE,
            required=True,
        )
        ticket_followers = sdk2.parameters.String('Comma-separated list of users to be added as ticket followers')

    @classmethod
    def _get_next_release_number(cls):
        releases = []
        num_pattern = r'^r(\d+)/$'
        cur_releases = Arcadia.list(cls.RELEASE_URL_BASE, 'immediates', True)
        logging.debug('Found releases {}'.format(cur_releases))
        for rev in cur_releases:
            res = re.search(num_pattern, rev)
            if res:
                releases.append(int(res.group(1)))
        return sorted(releases)[-1] + 1

    @classmethod
    def _create_release(cls, source, release_number):
        logging.debug('Creating release {} from {}'.format(release_number, source))
        target_url = '{}/r{}/arcadia'.format(cls.RELEASE_URL_BASE, release_number)
        Arcadia.create_tag(
            source,
            target_url,
            'zomb-sandbox-rw',
            'report-renderer/{}'.format(release_number),
            parents=True,
        )
        return target_url

    def _start_build(self):
        logging.debug('Creating subtask BUILD_REPORT_RENDERER_BUNDLE_SDK_2')
        build_subtask = BuildReportRendererBundleSdk2(
            self,
            description=self.Parameters.description,
            renderer_package=self.Parameters.renderer_package.id,
            nodejs_package=self.Parameters.nodejs_package.id,
            release_ticket_key=self.Context.ticket_key,
        )
        logging.debug('Subtask {} created'.format(build_subtask.id))
        build_subtask.enqueue()
        return build_subtask

    def _release_build(self):
        self.server.release(task_id=self.Context.build_subtask_id, type='testing', subject='')
        logging.debug('Subtask {} is set for release'.format(self.Context.build_subtask_id))

    def _get_package_description(self):
        return u'''
Сборка релиза: https://sandbox.yandex-team.ru/task/{}/view
Собранный пакет: https://sandbox.yandex-team.ru/resource/{}/view
'''.format(self.Context.build_subtask_id, self.Context.bundle_id)

    def _get_compare_description(self):
        status = self.Context.subtask_statuses['compare']
        start = u'(({} Сравнение бенчмарков)) времени ответа'.format(
            self._subtask_href(self.Context.compare_subtask_id))
        if status != ctt.Status.SUCCESS:
            return start + u' в статусе {}'.format(status)
        comparison_result = REPORT_RENDERER_BENCHMARK_COMPARISON_RESULT.find(
            task_id=self.Context.compare_subtask_id,
        ).first()
        return u'''{start}:
{proxydir}/comparison_http_render_dhhh.svg
{proxydir}/comparison_apphost_render_dhhh.svg
{proxydir}/comparison_apphost_context.total_dhhh.svg
{proxydir}/comparison_ahproxy_ahproxy_answer_time_dhhh.svg
'''.format(start=start, proxydir=comparison_result.http_proxy)

    def _get_worker_benchmark_description(self):
        status = self.Context.subtask_statuses['worker_benchmark']
        start = u'(({} Бенчмарк памяти воркера))'.format(
            self._subtask_href(self.Context.worker_benchmark_subtask_id))
        if status != ctt.Status.SUCCESS:
            return start + u' в статусе {}'.format(status)
        benchmark_result = REPORT_RENDERER_MEMORY_BENCHMARK_RESULT.find(
            task_id=self.Context.worker_benchmark_subtask_id,
        ).first()
        return start + u':\n{}/memory_usage.svg'.format(benchmark_result.http_proxy)

    def _get_master_benchmark_description(self):
        status = self.Context.subtask_statuses['master_benchmark']
        start = u'(({} Бенчмарк памяти мастера))'.format(
            self._subtask_href(self.Context.master_benchmark_subtask_id))
        if status != ctt.Status.SUCCESS:
            return start + u' в статусе {}'.format(status)
        benchmark_result = REPORT_RENDERER_MASTER_MEMORY_BENCHMARK_RESULT.find(
            task_id=self.Context.master_benchmark_subtask_id,
        ).first()
        return start + u':\n{}/memory_usage.svg'.format(benchmark_result.http_proxy)

    def _get_blockstat_checks_description(self):
        statuses = self.Context.subtask_statuses['blockstat_checks']
        if statuses is None:
            return u'Проверки blockstat log не запущена'
        return u'\n'.join(map(
            lambda args: self._get_blockstat_check_description(*args),
            zip(self.Context.blockstat_check_subtask_ids, self.Context.subtask_statuses['blockstat_checks'])
        ))

    def _get_blockstat_check_description(self, task_id, status):
        start = u'(({} Проверка blockstat log))'.format(
            self._subtask_href(task_id))
        if status != ctt.Status.SUCCESS:
            return start + u' в статусе {}'.format(status)
        return start + u' завершена успешно'

    def _get_integration_check_description(self):
        status = self.Context.subtask_statuses['integration_check']
        if status is None:
            return u'Интеграционные тесты не запущены'
        start = u'(({} Интеграционные тесты))'.format(
            self._subtask_href(self.Context.integration_check_subtask_id))
        if status != ctt.Status.SUCCESS:
            return start + u' в статусе {}'.format(status)
        return start + u' завершены успешно'

    def _get_ticket_description(self):
        return u'\n'.join([
            GENERATED_DESCRIPTION_START,
            self._get_package_description(),
            self._get_compare_description(),
            self._get_worker_benchmark_description(),
            self._get_master_benchmark_description(),
            self._get_blockstat_checks_description(),
            self._get_integration_check_description(),
        ])

    @staticmethod
    def _get_startrek_client():
        from startrek_client import Startrek
        return Startrek(useragent='sandbox-task', token=sdk2.Vault.data('robot-serp-bot_startrek_token'))

    def _create_ticket(self, rr_version):
        logging.debug('Creating release ticket in startrek')
        issue_name = 'report-renderer/{}'.format(rr_version)
        followers = [name.strip() for name in self.Parameters.ticket_followers.split(',') if name != '']
        description = u'''
{description}

----

Пакет содержит report-renderer версии {rr_version} и ynode версии {ynode_version}
Приёмка:
https://nanny.yandex-team.ru/ui/#/services/catalog/renderer_loadtest/
https://nanny.yandex-team.ru/ui/#/services/catalog/renderer_loadtest_apphost/
https://nanny.yandex-team.ru/ui/#/services/catalog/priemka__templates__news_renderer_yp/
Тикет создан задачей https://sandbox.yandex-team.ru/task/{self_id}/view

----

Информация о тестировании релиза будет отображена здесь
'''.format(
            description=self.Parameters.description,
            rr_version=rr_version,
            ynode_version=self.Parameters.nodejs_package.resource_version,
            self_id=self.id,
        )
        ticket = self._get_startrek_client().issues.create(
            queue='SEAREL',
            summary=issue_name,
            description=description,
            type='task',
            tags=['report-renderer'],
            followers=followers,
            assignee=self.author,
        )
        self.Context.ticket_key = ticket.key
        logging.debug('Created ticket {}'.format(ticket.key))

    def _update_ticket_description(self):
        ticket = self._get_startrek_client().issues[self.Context.ticket_key]
        cut_start = ticket.description.find(GENERATED_DESCRIPTION_START)
        if cut_start == -1:
            prev_description = ticket.description + u'\n'
        else:
            prev_description = ticket.description[:cut_start]
        new_description = prev_description + self._get_ticket_description()
        ticket.update(description=new_description)

    def _get_bundle_id_and_revisions(self):
        logging.debug('Finding bundle to release')
        bundle = REPORT_RENDERER_BUNDLE.find(
            task_id=self.Context.build_subtask_id,
        ).first()
        self.Context.bundle_id = bundle.id
        self.Context.full_rr_version = bundle.report_renderer_version

    def _start_compare(self):
        logging.debug('Starting compare task COMPARE_REPORT_RENDERER_BENCHMARK')

        compare_subtask = CompareReportRendererBenchmarkSdk2(
            self,
            description="Release subtask {} vs {}".format(
                self.Context.bundle_id, self.Parameters.compare_vs_bundle.id
            ),
            base_bundle=self.Parameters.compare_vs_bundle.id,
            compared_bundle=self.Context.bundle_id,
        )
        self.Context.compare_subtask_id = compare_subtask.id
        compare_subtask.enqueue()
        logging.debug('Compare task {} created'.format(compare_subtask.id))

    def _start_worker_memory_benchmark(self):
        logging.debug('Starting benchmark task BENCHMARK_REPORT_RENDERER_MEMORY')

        benchmark_subtask = sdk2.Task["BENCHMARK_REPORT_RENDERER_MEMORY"](
            self,
            description="Benchmark subtask for {}".format(self.Context.bundle_id),
            **{
                # Остальные параметры оставим по умолчанию
                ReportRendererBundlePackage.name: self.Context.bundle_id,
            }
        )
        self.Context.worker_benchmark_subtask_id = benchmark_subtask.id
        benchmark_subtask.enqueue()
        logging.debug('Worker memory benchmark task {} created'.format(benchmark_subtask.id))

    def _start_master_memory_benchmark(self):
        logging.debug('Starting benchmark task BENCHMARK_REPORT_RENDERER_MASTER_MEMORY')

        benchmark_subtask = sdk2.Task["BENCHMARK_REPORT_RENDERER_MASTER_MEMORY"](
            self,
            description="Benchmark subtask for {}".format(self.Context.bundle_id),
            **{
                # Остальные параметры оставим по умолчанию
                ReportRendererBundlePackage.name: self.Context.bundle_id,
            }
        )
        self.Context.master_benchmark_subtask_id = benchmark_subtask.id
        benchmark_subtask.enqueue()
        logging.debug('Master memory benchmark task {} created'.format(benchmark_subtask.id))

    def _start_blockstat_check(self):
        logging.debug('Starting blockstat check tasks TEST_REPORT_LOGS')
        task_tree = get_all_ancestors([self.id])
        logging.debug('Found tasks {} in self task tree'.format(task_tree))
        blockstat_logs = list(REPORT_RENDERER_BLOCKSTAT_LOG.find(
            task_id=task_tree,
            attrs={'REPORT_RENDERER_BUNDLE': str(self.Context.bundle_id)},
        ).limit(100))
        logging.debug('Found block logs {} matching bundle {}'.format(blockstat_logs, self.Context.bundle_id))
        if not blockstat_logs:
            logging.error('Could not find any REPORT_RENDERER_BLOCKSTAT_LOG resource with report_renderer_bundle={}'
                          .format(self.Context.bundle_id))
            return
        subtask_ids = []
        for blockstat_log in blockstat_logs:
            blockstat_subtask = sdk2.Task["TEST_REPORT_LOGS"](
                self,
                description="Blockstat check for {}".format(self.Context.bundle_id),
                **{
                    LogName.name: 'blockstat_log',
                    InputData.name: blockstat_log.id,
                }
            )
            subtask_ids.append(blockstat_subtask.id)
            blockstat_subtask.enqueue()
            logging.debug('Blockstat check task {} for log {} created'.format(blockstat_subtask.id, blockstat_log.id))
        self.Context.blockstat_check_subtask_ids = subtask_ids

    def _start_integration_tests(self):
        logging.debug('Starting integration tests task SEARCH_INTERFACES_INTEGRATION_TESTS')

        integration_tests_subtask = SearchInterfacesIntegrationTests(
            self,
            owner=self.owner,
            description='Search interfaces integration tests subtask for {}'.format(self.Context.bundle_id),
            priority={
                'class': 'SERVICE',
                'subclass': 'NORMAL',
            },
            project_build_context='dev',
            report_renderer_package=self.Parameters.renderer_package.id,
            report_renderer_nodejs_package=self.Parameters.nodejs_package.id,
        )
        logging.debug('Integration tests task {} created'.format(integration_tests_subtask.id))

        integration_tests_subtask.enqueue()
        self.Context.integration_check_subtask_id = integration_tests_subtask.id
        return integration_tests_subtask

    def _get_subtask_status(self, task_id):
        if not task_id:
            return None
        logging.debug('Checking status of task {}'.format(task_id))
        return self.server.task.read(
            id=task_id,
            children=True,
            fields='status',
            limit=1)['items'][0]['status']

    def _update_subtask_statuses(self):
        self.Context.subtask_statuses = {
            'compare': self._get_subtask_status(self.Context.compare_subtask_id),
            'worker_benchmark': self._get_subtask_status(self.Context.worker_benchmark_subtask_id),
            'master_benchmark': self._get_subtask_status(self.Context.master_benchmark_subtask_id),
            'blockstat_checks':
                map(self._get_subtask_status, self.Context.blockstat_check_subtask_ids)
                if self.Context.blockstat_check_subtask_ids
                else [],
            'integration_check': self._get_subtask_status(self.Context.integration_check_subtask_id),
        }

    def _has_failed_subtasks(self):
        result = False
        for name, value in self.Context.subtask_statuses.iteritems():
            if name == 'blockstat_checks':
                bad_statuses = filter(lambda s: s != ctt.Status.SUCCESS, value)
                result = result or bool(bad_statuses)
            else:
                result = result or value != ctt.Status.SUCCESS
        return result

    def on_execute(self):
        with self.memoize_stage.start_build_subtask:
            rr_version = self.Parameters.renderer_package.version
            self._create_ticket(rr_version)
            build_subtask = self._start_build()
            self.Context.build_subtask_id = build_subtask.id
            raise sdk2.WaitTask(
                tasks=[build_subtask],
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True
            )

        with self.memoize_stage.start_benchmark_subtasks:
            build_status = self._get_subtask_status(self.Context.build_subtask_id)
            if build_status != ctt.Status.SUCCESS:
                raise TaskFailure('Build task status is {}'.format(build_status))
            self._release_build()
            self._get_bundle_id_and_revisions()

            # Эти задачи генерируют blockstat-лог, их будем дожидаться для следующего этапа
            self._start_compare()
            self._start_master_memory_benchmark()

            # Эти задачи запускаем сейчас, но дожидаться их завершения будем на следующем этапе
            self._start_worker_memory_benchmark()
            self._start_integration_tests()

            raise sdk2.WaitTask(
                [
                    self.Context.compare_subtask_id,
                    self.Context.master_benchmark_subtask_id,
                ],
                list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK)
            )

        with self.memoize_stage.start_blockstat_subtask:
            self._update_subtask_statuses()
            self._start_blockstat_check()
            subtasks_to_wait = [
                self.Context.worker_benchmark_subtask_id,
                self.Context.integration_check_subtask_id
            ] + self.Context.blockstat_check_subtask_ids
            self._update_ticket_description()
            raise sdk2.WaitTask(subtasks_to_wait, list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK))

        self._update_subtask_statuses()
        self._update_ticket_description()

        if self._has_failed_subtasks():
            raise TaskFailure('Some of subtasks failed')

    @staticmethod
    def _subtask_href(task_id):
        return 'https://sandbox.yandex-team.ru/task/{}/view'.format(task_id)

    def _subtask_link(self, task_id, name):
        if not task_id:
            return name
        return '<a href="{}">{}</a>'.format(self._subtask_href(task_id), name)

    def _formatted_subtask_status(self, task_id):
        status = self._get_subtask_status(task_id)
        if not status:
            return 'not created'
        return '<span class="status status_{}">{}</span>'.format(status.lower(), status)

    @property
    def footer(self):
        def render_subtask(name, task_id):
            return '{} {}'.format(self._subtask_link(task_id, name), self._formatted_subtask_status(task_id))

        def render_ticket():
            if not self.Context.ticket_key:
                return 'Release ticket not created'
            return '<a href="https://st.yandex-team.ru/{}">Release ticket</a>'.format(self.Context.ticket_key)

        res = [render_subtask('Build', self.Context.build_subtask_id), render_ticket(),
               render_subtask('Compare response times', self.Context.compare_subtask_id),
               render_subtask('Benchmark worker memory', self.Context.worker_benchmark_subtask_id),
               render_subtask('Benchmark master memory', self.Context.master_benchmark_subtask_id),
               render_subtask('Integration with projects', self.Context.integration_check_subtask_id)] + map(
            lambda task_id: render_subtask('Check blockstat', task_id), self.Context.blockstat_check_subtask_ids)
        return '<br>'.join(res)
