import calendar
import json
import logging
import os

from sandbox import sdk2
import sandbox.common.errors as ce
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.projects.review as review


class AutocheckReview(sdk2.Task):
    """
    Base review pre-commit check class
    """

    LAUNCHER_LOG_NAME = 'launcher'

    launcher_resource_type = None
    patch_resource_type = None
    zipatch_resource_type = None

    def check_params(self):
        pass

    def extend_args(self, args):
        pass

    class Parameters(sdk2.Task.Parameters):
        rev = sdk2.parameters.Integer('Base Arcadia SVN revision', required=True)
        author = sdk2.parameters.String('Author username', required=True)
        project_list = sdk2.parameters.List('List of changed projects')
        zipatch_url = sdk2.parameters.String('Zipatch URL (Arcadia root relative)')
        patch_url = sdk2.parameters.String('Human-readable patch URL')
        zipatch = sdk2.parameters.String('Zipatch data (Arcadia root relative, base64, deprecated)')
        patch = sdk2.parameters.String('Human-readable patch data (compressed + base64, deprecated)')
        report = sdk2.parameters.String('Post report to review (any value)')
        message = sdk2.parameters.String('Commit message')
        branch = sdk2.parameters.String('Arcadia SVN branch to use for checking')
        gsid = sdk2.parameters.String('GSID')
        recheck = sdk2.parameters.Bool('Recheck all affected builds and tests')

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1 * 1024  # 1 GiB
        client_tags = ctc.Tag.SSD
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Task.Context):
        testenv_check_id = None

    def on_execute(self):
        if self.Parameters.report and not self.check_params():
            raise ce.TaskFailure('Report required credentials are not specified')

        launcher_path = review.fetch(self.launcher_resource_type)
        if not launcher_path:
            raise ce.TaskError('Checker executable ({}) release not found'.format(self.launcher_resource_type.name))
        logging.info('Fetched launcher: %s', launcher_path)

        project_list_path = self._prepare_project_list(self.Parameters.project_list)

        if self.Parameters.zipatch_url:
            zipatch_url = self.Parameters.zipatch_url
        elif self.Parameters.zipatch:
            _, zipatch_url = self._prepare_zipatch(self.Parameters.zipatch)
        else:
            raise ce.TaskError('No zipatch specified')

        patch_url = None
        if self.Parameters.patch_url:
            patch_url = self.Parameters.patch_url
        elif self.Parameters.patch:
            _, patch_url = self._prepare_patch(self.Parameters.patch)

        gsid = []
        if self.Parameters.gsid:
            gsid.extend(self.Parameters.gsid.split(' '))
        if os.environ.get('GSID'):
            gsid.extend(os.environ.get('GSID').split(' '))
        logging.info('GSID: %s', os.environ.get('GSID'))

        args = [
            launcher_path, str(self.Parameters.rev), project_list_path,
            '--author', self.Parameters.author,
            '--zipatch-url', zipatch_url,
            '--time', str(int(calendar.timegm(self.created.timetuple()))),
        ]

        if self.Parameters.report:
            args.extend(['--report', 'true'])
        if patch_url:
            args.extend(['--patch-url', patch_url])
        if self.Parameters.message:
            args.extend(['--message', review.escape_text(self.Parameters.message)])
            args.extend(['--message-base64'])
        if self.Parameters.branch:
            args.extend(['--branch', self.Parameters.branch])
        if gsid:
            args.extend(['--gsid', ' '.join(gsid)])

        if self.Parameters.recheck:
            args.extend(['--recheck'])

        self.extend_args(args)
        logging.info('Launcher args: %s', args)

        try:
            with sdk2.helpers.ProcessLog(self, self.LAUNCHER_LOG_NAME) as pl:
                check_info = sdk2.helpers.subprocess.check_output(args, stderr=pl.stderr)
        except sdk2.helpers.subprocess.CalledProcessError, e:
            with open(str(self.log_path('{}.err.log'.format(self.LAUNCHER_LOG_NAME))), 'r') as log:
                self.set_info('Checker error log:\n{}'.format(log.read()))
            if e.returncode == 2:
                raise ce.TemporaryError('Error creating check (retriable)')
            raise ce.TaskError('Error creating check, exit code {}'.format(e.returncode))

        check = json.loads(check_info)
        logging.info('Check: %s', check)

        self.set_info('CI check created: <a href="{url}">{id}</a>'.format(**check), do_escape=False)

        self.Context.testenv_check_id = check['id']

    def _prepare_project_list(self, project_list):
        project_list_path = str(self.path('project_list.txt'))
        with open(project_list_path, 'wb') as project_list_file:
            project_list_file.write(''.join(('{}\n'.format(path) for path in project_list)))
        return project_list_path

    def _prepare_zipatch(self, zipatch):
        return review.share_as_resource(self, zipatch, self.zipatch_resource_type, 'review_autocheck.zipatch', 'Review autocheck zipatch', compressed=False)

    def _prepare_patch(self, patch):
        if patch is ctm.NotExists:
            return None, None
        return review.share_as_resource(self, patch, self.patch_resource_type, 'review_autocheck.diff', 'Review autocheck human-readable patch')
