import json

from sandbox import sdk2

import sandbox.projects.common.arcadia.sdk as asdk
import sandbox.projects.review.base as base


class AutocheckCrucible(sdk2.resource.AbstractResource):
    """
    Crucible review pre-commit autocheck tool
    """

    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    releasers = ['nslus', 'albazh']


class AutocheckCrucibleZipatch(sdk2.resource.Resource):
    """
    Crucible review zipatch
    """

    pass


class AutocheckCruciblePatch(sdk2.resource.Resource):
    """
    Crucible review human-readable patch
    """

    pass


class AutocheckFromCrucible(base.AutocheckReview):
    """
    Crucible review pre-commit autocheck
    https://wiki.yandex-team.ru/arcadia/hooks/crucible
    """

    class Parameters(base.AutocheckReview.Parameters):
        crucible_review = sdk2.parameters.String('Crucible review ID', required=True)
        crucible_url = sdk2.parameters.String('Crucible URL')
        crucible_creds = sdk2.parameters.String('Crucible credentials (Vault key, format: two lines, username and password)')

    launcher_resource_type = AutocheckCrucible
    patch_resource_type = AutocheckCruciblePatch
    zipatch_resource_type = AutocheckCrucibleZipatch

    def check_params(self):
        if self.Parameters.report:
            if not self.Parameters.crucible_url or not self.Parameters.crucible_creds:
                return False
        return True

    def extend_args(self, args):
        args.extend(['--crucible-review', self.Parameters.crucible_review])
        if self.Parameters.report:
            args.extend(['--crucible-url', self.Parameters.crucible_url])
            args.extend(['--crucible-auth', asdk.save_vault(self, self.Parameters.crucible_creds, 'auth-crucible.txt', postprocess=self._jsonize_creds)])

    def _jsonize_creds(self, creds):
        lines = [line.strip() for line in creds.split('\n')]
        lines = [line for line in lines if line]
        assert len(lines) == 2
        username, password = lines
        return json.dumps({
            'user_name': username,
            'password': password,
        })
