#!/usr/bin/env python
# coding: utf-8

import glob
import logging
import os
import shlex
import subprocess

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox.projects import resource_types
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import environments


class RevisionParameter(parameters.SandboxIntegerParameter):
    name = "revision"
    description = "Revision, using HEAD if empty"
    default_value = None


class ReviewBoardSvnUrlParameter(parameters.SandboxArcadiaUrlParameter):
    name = "review_board_svn_url"
    default_value = "arcadia:/arc/trunk/arcadia/devtools/reviewboard"
    description = "Review Board SVN URL"


class ReviewBoardDepsSvnUrlParameter(parameters.SandboxArcadiaUrlParameter):
    name = "review_board_deps_svn_url"
    default_value = "arcadia:/arc/trunk/arcadia/junk/deniskuzin/reviewboard-deps"
    description = "Review Board Dependencies SVN URL"


class BuildReviewBoard(task.SandboxTask):

    type = "BUILD_REVIEW_BOARD"

    privileged = True

    dns = ctm.DnsType.DNS64

    execution_space = 5 * 1024

    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = [
        RevisionParameter,
        ReviewBoardSvnUrlParameter,
        ReviewBoardDepsSvnUrlParameter,
    ]

    def on_execute(self):
        self._get_source_files()
        self.package_path = self.abs_path("package")
        paths.make_folder(self.package_path)
        paths.make_folder(os.path.join(self.package_path, "linux"))
        self._add_samogon_plugin()
        self._add_venv()
        self._make_package()

    def _get_source_files(self):
        self.deps_src_path = self.abs_path("reviewboard-deps")
        svn.Arcadia.checkout(self.ctx[ReviewBoardDepsSvnUrlParameter.name], self.deps_src_path, revision=self.ctx[RevisionParameter.name])
        self.src_path = self.abs_path("reviewboard")
        svn.Arcadia.checkout(self.ctx[ReviewBoardSvnUrlParameter.name], self.src_path, revision=self.ctx[RevisionParameter.name])
        self.commit_revision = svn.Arcadia.info(self.src_path)["commit_revision"]

    def _add_samogon_plugin(self):
        paths.copy_path(os.path.join(self.src_path, "samogon-plugin"), os.path.join(self.package_path, "plugin"))

    def _add_venv(self):
        self._run('apt-get update')
        self._run('apt-get install -y libffi-dev libgmp3-dev libsvn1 libsvn-dev node-uglify libmemcached-dev')
        with environments.VirtualEnvironment('/tmp') as venv:
            self.venv = venv
            venv.pip('pip==9.0.3')
            self._install_blackbox()
            venv.pip('pip-accel')
            venv.pip('requests[security]')
            venv.pip('startrek-client==1.6.0')
            self._run('{} -m pip install --allow-external PIL --allow-unverified PIL --index-url https://pypi.yandex-team.ru/simple PIL'.format(venv.executable))
            self._run('{} -m pip install --allow-external Django --allow-unverified Django --index-url https://pypi.yandex-team.ru/simple Django==1.4.22'.format(venv.executable))
            venv.pip('django-elliptics==1.6')
            venv.pip('django-pipeline==1.2.24')
            self._install_djblets()
            self._run('{} -m pip install --index-url https://pypi.yandex-team.ru/simple/ django-evolution==0.6.9'.format(venv.executable))
            self._run('{} -m pip install --index-url https://pypi.yandex-team.ru/simple/ django_yauth==4.0'.format(venv.executable))
            self._run('{} -m pip install --index-url https://pypi.yandex-team.ru/simple/ ylog'.format(venv.executable))
            self._run('{} -m pip install --index-url https://pypi.yandex-team.ru/simple/ jsmin'.format(venv.executable))
            venv.pip('docutils')
            venv.pip('markdown')
            venv.pip('mimeparse')
            self._install_pycparser()
            venv.pip('paramiko')
            venv.pip('gunicorn')
            venv.pip('MySQL-python')
            venv.pip('Pygments')
            venv.pip('python-memcached')
            venv.pip('pylibmc')
            venv.pip('recaptcha-client')
            venv.pip('mercurial')
            venv.pip('python-dateutil==1.5')
            self._install_pysvn()
            self._install_reviewboard()
            self._install_reviewboard_yandex()
            venv.pack_bundle(os.path.join(self.src_path, "venv.tgz"))

    def _install_blackbox(self):
        blackbox_path = os.path.join(self.deps_src_path, 'python-blackbox-ipv6')
        subprocess.check_call(['python', 'setup.py', 'sdist'], cwd=blackbox_path)
        dist_file = glob.glob('{}/dist/blackbox-*.tar.gz'.format(blackbox_path))
        self.venv.pip(dist_file[0])

    def _install_djblets(self):
        djblets_path = glob.glob('{}/Djblets-*'.format(self.deps_src_path))[0]
        self._run('python setup.py sdist', cwd=djblets_path)
        dist_file = glob.glob('{}/dist/Djblets-*.tar.gz'.format(djblets_path))
        self.venv.pip(dist_file[0])

    def _install_pysvn(self):
        pysvn_path = os.path.join(self.deps_src_path, 'pysvn-1.8.0.tar.gz')
        self._run('tar -xf {}'.format(pysvn_path), cwd=self.package_path)
        command_template = '{} setup.py configure --svn-inc-dir=/usr/include/subversion-1 --svn-lib-dir=/usr/lib/x86_64-linux-gnu --apr-inc-dir=/usr/include/apr-1.0 --apu-inc-dir=/usr/include/apr-1.0'
        self._run(command_template.format(self.venv.executable), cwd=os.path.join(self.package_path, 'pysvn-1.8.0', 'Source'))
        self._run('make', cwd=os.path.join(self.package_path, 'pysvn-1.8.0', 'Source'))
        self._run('mv {} {}'.format(
            os.path.join(self.package_path, 'pysvn-1.8.0', 'Source', 'pysvn'),
            os.path.join(self.venv.root_dir, 'lib', 'python2.7', 'site-packages', 'pysvn')
        ))

    def _install_pycparser(self):
        pycparser_path = os.path.join(self.deps_src_path, 'pycparser-2.14.tar.gz')
        self.venv.pip(pycparser_path)

    def _install_reviewboard(self):
        review_board_path = os.path.join(self.src_path, 'reviewboard')
        self._run('python setup.py sdist', cwd=review_board_path)
        dist_file = glob.glob('{}/dist/ReviewBoard-*.tar.gz'.format(review_board_path))
        self._run('pip install --pre --no-index --find-links=file://{} ReviewBoard'.format(dist_file[0]))

    def _install_reviewboard_yandex(self):
        yandex_path = os.path.join(self.src_path, 'reviewboard-yandex')
        self._run('python setup.py sdist', cwd=yandex_path)
        dist_file = glob.glob('{}/dist/reviewboard_yandex-*.tar.gz'.format(yandex_path))
        self._run('pip install --pre --no-index --find-links=file://{} reviewboard_yandex'.format(dist_file[0]))

    def _run(self, command, cwd=None):
        logging.debug(command)
        command = shlex.split(command)
        popen = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, cwd=cwd)
        stdout, stderr = popen.communicate()
        if popen.returncode:
            logging.error('stdout: %s', stdout)
            logging.error('stderr: %s', stderr)
            raise Exception("non zero return code")

    def _make_package(self):
        self.review_board_tarball = os.path.join(self.package_path, "linux", "reviewboard.tar.gz")
        process.run_process(
            ["tar", "czf", self.review_board_tarball, "reviewboard", "reviewboard-yandex", "samogon-plugin", "venv.tgz"],
            log_prefix="rb_tar",
            work_dir=self.src_path,
        )
        self.tarball_path = os.path.join(self.package_path, "reviewboard.tgz")
        process.run_process(
            ["tar", "czf", self.tarball_path, "plugin", "linux"],
            log_prefix="tar",
            work_dir=self.package_path,
        )
        self.create_resource(
            description="Review Board Samogon package r%s" % self.commit_revision,
            resource_path=self.tarball_path,
            resource_type=resource_types.REVIEW_BOARD_PACKAGE,
            attributes={
                'ttl': 180,
            },
        )


__Task__ = BuildReviewBoard
