# -*- encoding: utf8 -*-

import requests
from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common import link_builder as lb
from sandbox.sandboxsdk.environments import PipEnvironment


class RmpBackendSetActiveTicketSprint(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        st_token_vault = sdk2.parameters.String('Tracker token vault', default='startrek_token')
        st_board = sdk2.parameters.String('Board', required=True)
        ignored_statuses = sdk2.parameters.List('Ignored ticket statuses', default=[])
        dry_run = sdk2.parameters.Bool('Dry run', default=False)

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        environments = (
            PipEnvironment('startrek_client', use_wheel=True),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    @staticmethod
    def _get_st_api_headers(token):
        return {'Authorization': 'OAuth {}'.format(token)}

    def _get_board(self, st_token):
        response = requests.get(
            'https://st-api.yandex-team.ru/v2/boards/{}'.format(self.Parameters.st_board),
            headers=self._get_st_api_headers(st_token),
        )
        response.raise_for_status()
        return response.json()

    def _get_current_sprint(self, st_token, board):
        response = requests.get(
            'https://st-api.yandex-team.ru/v2/boards/{}/sprints'.format(board['id']),
            headers=self._get_st_api_headers(st_token),
        )
        response.raise_for_status()
        for sprint in response.json():
            if sprint['status'] == 'in_progress':
                return sprint

    def _get_query(self, board):
        query = [
            '(({board_query}) OR Boards: {board_id})'.format(board_query=board['query'], board_id=board['id']),
            'Sprint: empty()'
        ]
        for status in self.Parameters.ignored_statuses:
            query.append('Status: ! "{}"'.format(status))
        query = ' AND '.join(query)
        return query

    def _set_sprint(self, st_client, ticket, board, sprint):
        try:
            issue = st_client.issues[ticket]
            issue.update(sprint=sprint['id'])
            issue.comments.create(
                text=('Автоматика обнаружила тикет на доске "{board_name}" в рабочем статусе, но не в спринте, '
                      'и проставила ему текущий активный спринт.\n'
                      '<{{Техническая информация:\n'
                      'Sandbox task: {task_link}\n'
                      'Ticket find query: %%\n'
                      '{query}\n'
                      '%%}}>').format(board_name=board['name'],
                                      task_link=lb.task_link(self.id, plain=True),
                                      query=self._get_query(board))
            )
        except Exception as e:
            self.set_info('Failed to update ticket {}:\n{}'.format(ticket, e))
            return [ticket]
        else:
            return []

    def on_execute(self):
        from startrek_client import Startrek

        st_token = sdk2.Vault.data(self.owner, self.Parameters.st_token_vault)
        st_client = Startrek(useragent="RmpBackendSetActiveTicketSprint", token=st_token)
        board = self._get_board(st_token)
        current_sprint = self._get_current_sprint(st_token, board)
        tickets = [ticket['key'] for ticket in st_client.issues.find(self._get_query(board))]

        self.set_info('Set sprint "{sprint}" for tickets:\n{tickets}'.format(
            sprint=current_sprint['name'],
            tickets='\n'.join(tickets)
        ))

        if self.Parameters.dry_run:
            return

        failed_tickets = []
        for ticket in tickets:
            failed_tickets += self._set_sprint(st_client, ticket, board, current_sprint)
        if failed_tickets:
            raise errors.TaskFailure(
                'Failed to update tickets {}'.format(', '.join(failed_tickets))
            )
