import collections
import json
import logging
import requests
from six.moves.urllib import parse as urlparse

from sandbox import sdk2
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common.nanny import nanny


ALERT_PREFIX_TEMPLATE = '{yasm_prefix}.{service_id}'
ALERT_NAME_TEMPLATE = ALERT_PREFIX_TEMPLATE + '.{resource}'


def get_resources():
    resources = {
        'cpu-throttled-cores': {
            'signal': 'portoinst-cpu_throttled_cores_tmmv',
            'warn': [1, 2],
            'crit': [2, None],
        },
    }
    for res, sig in (
        ('cpu', 'cpu_guarantee'),
        ('memory', 'memory_anon_unevict_limit')
    ):
        resources['{res}-usage-perc'.format(res=res)] = {
            'signal': 'quant(portoinst-{sig}_usage_perc_hgram, 99)'.format(sig=sig),
            'warn': [60, 75],
            'crit': [75, None],
        }
    for op in ('read', 'write'):
        for res in ('bps', 'ops'):
            for disk_mnt in (
                {'sig': '/place', 'name': 'place'},
                {'sig': '/ssd', 'name': 'ssd'},
                {'sig': '/', 'name': 'root'}
            ):
                resources['io-{op}-{res}-{disk_mnt}-usage-perc'.format(
                    op=op,
                    res=res,
                    disk_mnt=disk_mnt['name'],
                )] = {
                    'signal': 'quant(portoinst-io_{op}_{res}_limit_{disk_mnt}_usage_perc_hgram, 99)'.format(
                        op=op,
                        res=res,
                        disk_mnt=disk_mnt['sig'],
                    ),
                    'warn': [75, 90],
                    'crit': [90, None],
                }
    # TODO: volume usage, net
    return resources


RESOURCES = get_resources()


class RmpGenerateResourceUsageAlerts(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        service_ids = sdk2.parameters.List('Service ids', required=True)
        dry_run = sdk2.parameters.Bool('Do not make actual requests', default=True)

        with sdk2.parameters.Group('Yasm and Juggler parameters'):
            abc_service_slug = sdk2.parameters.String('ABC service', required=True)
            yasm_prefix = sdk2.parameters.String('Yasm prefix', required=True)
            yasm_api_endpoint = sdk2.parameters.String('Yasm api endpoint', default='https://yasm.yandex-team.ru')
            juggler_namespace = sdk2.parameters.String('Juggler namespace', required=True)
            juggler_tags = sdk2.parameters.List('Juggler tags', default=[])

        with sdk2.parameters.CheckGroup('Resources') as resources:
            for resource in RESOURCES:
                resources.values[resource] = resources.Value(resource, checked=True)

        with sdk2.parameters.Group('Nanny parameters'):
            nanny_api_endpoint = sdk2.parameters.String('Nanny api url', default='https://nanny.yandex-team.ru/')
            nanny_token_vault = sdk2.parameters.String('Nanny token vault', default='nanny_oauth_token')

    def on_execute(self):
        nanny_oauth_token = sdk2.Vault.data(self.owner, self.Parameters.nanny_token_vault)
        nanny_client = nanny.NannyClient(api_url=self.Parameters.nanny_api_endpoint, oauth_token=nanny_oauth_token)

        all_alerts = collections.defaultdict(list)
        for service_id in self.Parameters.service_ids:
            all_alerts[ALERT_PREFIX_TEMPLATE.format(
                yasm_prefix=self.Parameters.yasm_prefix,
                service_id=service_id,
            )].extend(self._generate_alerts_for_nanny_service(nanny_client, service_id))
        for prefix, service_alerts in all_alerts.iteritems():
            logging.info(
                'Going to replace alerts on prefix %s\n%s',
                prefix,
                json.dumps(service_alerts, indent=4),
            )
        if self.Parameters.dry_run:
            logging.info('Dry run - not really replacing')
            return

        for prefix, service_alerts in all_alerts.iteritems():
            response = requests.post(
                urlparse.urljoin(self.Parameters.yasm_api_endpoint, '/srvambry/alerts/replace'),
                json={
                    'prefix': prefix,
                    'alerts': service_alerts,
                }
            )
            logging.info(
                json.dumps(response.json(), indent=4)
            )
            response.raise_for_status()

    def _get_instance_tags(self, nanny_client, service_id):
        instances = nanny_client.get_service_instances(service_id)
        tags = {}
        for key in ('itype', 'ctype', 'prj'):
            tags[key] = [instances['content']['yp_pod_ids']['orthogonal_tags'][key]]
        tags['geo'] = list(set([pod['cluster'].lower() for pod in instances['content']['yp_pod_ids']['pods']]))
        return tags

    def _generate_alerts_for_nanny_service(self, nanny_client, service_id):
        alerts = []
        for resource in self.Parameters.resources:
            alerts.append({
                'name': ALERT_NAME_TEMPLATE.format(
                    yasm_prefix=self.Parameters.yasm_prefix,
                    service_id=service_id,
                    resource=resource,
                ),
                'description': 'Generated by sandbox task {}'.format(lb.task_link(self.id, plain=True)),
                'signal': RESOURCES[resource]['signal'],
                'tags': self._get_instance_tags(nanny_client, service_id),
                'crit': RESOURCES[resource]['crit'],
                'warn': RESOURCES[resource]['warn'],
                'mgroups': ['ASEARCH'],
                'abc': self.Parameters.abc_service_slug,
                'juggler_check': {
                    'host': service_id,
                    'namespace': self.Parameters.juggler_namespace,
                    'service': resource,
                    'tags': self.Parameters.juggler_tags,
                    'flaps': {
                        'critical': 120,
                        'boost': 0,
                        'stable': 30
                    },
                    'aggregator_kwargs': {
                        'nodata_mode': 'force_ok',
                    },
                },
            })
        return alerts
