# -*- encoding: utf8 -*-

import logging
import os
import json
import copy
import re

from sandbox.common.errors import TaskFailure
import sandbox.common.types.misc as common_types
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.common.vcs import arc as arc_client


RELATIVE_PROJECT_PATH = os.path.join('direct', 'frontend', 'services', 'yfa')
YAV_ROBOT_RMP_SECRET = 'sec-01f7v789a9ta11ynp35txd6b5k'
ROBOT_RMP_NAME = 'robot-rmp'
DEPENDENCIES_TO_INSTALL = [
    'tanker-kit',
    'ts-node',
    'typograf',
    'typescript',
    '@yandex-int/tokenator',
    'chalk',
    'express',
    '@babel/core',
    '@babel/preset-env',
    'prettier'
]


class YandexForAppsTankerManagementTask(sdk2.Task):

    name = 'YANDEX_FOR_APPS_TANKER_MANAGEMENT_TASK'

    class Requirements(sdk2.Requirements):
        dns = common_types.DnsType.DNS64
        environments = [
            environments.NodeJS('10.20.1'),
            environments.GCCEnvironment('5.3.0'),
            environments.ArcEnvironment()
        ]

    class Parameters(sdk2.Parameters):
        description = "Задача для загрузки/выгрузки переводов для проекта YFA"

        with sdk2.parameters.RadioGroup("Операция") as tanker_operation:
            tanker_operation.values["push"] = tanker_operation.Value(
                "Загрузить в Танкер", default=True)
            tanker_operation.values["pull"] = tanker_operation.Value(
                "Выгрузить из Танкера")

        tanker_branch = sdk2.parameters.String(
            "Ветка в Танкере",
            default="master",
            description='Например: YFA-51',
            required=True
        )

        arc_branch = sdk2.parameters.String(
            "Ветка в Аркадии",
            default="trunk",
            description='Например: users/fomasha/feat_simplified_translation_process.YFA-51',
            required=True
        )

        translate_sync_ticket = sdk2.parameters.String(
            "Номер тикета синхронизации переводов в Аркадии",
            default="YFA-70",
            description='Например: YFA-70, используется для создания бранчи и пр-а',
            required=True
        )

        secret_data = sdk2.parameters.YavSecret(
            label='ID секрета в Секретнице',
            description='Обязательные поля: ARC_TOKEN, TANKER_TOKEN, SSH_PRIVATE_KEY',
            default_value=YAV_ROBOT_RMP_SECRET,
            required=True
        )

    def _prepare_env(self):
        """
        Calling prepare for environments
        :return:
        """
        for env in self.Requirements.environments:
            env.prepare()

    def _get_yav_secret_data(self):
        return self.Parameters.secret_data.data()

    def _get_arc_oauth_token(self):
        secret_data = self._get_yav_secret_data()
        arc_oauth_token = secret_data['ARC_TOKEN']

        return arc_oauth_token

    def _get_tanker_oauth_token(self):
        secret_data = self._get_yav_secret_data()
        tanker_oauth_token = secret_data['TANKER_TOKEN']

        return tanker_oauth_token

    def _get_private_ssh_key(self):
        secret_data = self._get_yav_secret_data()
        private_ssh_key = secret_data['SSH_PRIVATE_KEY']

        return private_ssh_key

    def _get_ya_token(self):
        secret_data = self._get_yav_secret_data()
        ya_token = secret_data['YA_TOKEN']

        return ya_token

    def _remove_redundant_dependencies_from_package_json(self, project_path):
        logging.info("Redundant dependencies removal started")

        package_json_path = os.path.join(project_path, 'package.json')
        modified_package_json_data = None

        with open(package_json_path, 'r') as package_json_read:
            package_json_data = json.load(package_json_read)
            modified_package_json_data = copy.deepcopy(package_json_data)

            # Удаляем проверку на использование pnpm в качестве пакетного менеджера, так как здесь он не нужен
            del modified_package_json_data['scripts']['preinstall']

            modified_package_json_data['dependencies'] = {}
            modified_package_json_data['devDependencies'] = {}

            for dependency_name in DEPENDENCIES_TO_INSTALL:
                if dependency_name in package_json_data['devDependencies']:
                    modified_package_json_data['devDependencies'][
                        dependency_name] = package_json_data['devDependencies'][dependency_name]
                elif dependency_name in package_json_data['dependencies']:
                    modified_package_json_data['dependencies'][dependency_name] = package_json_data['dependencies'][dependency_name]
                else:
                    raise TaskFailure(
                        '{} is not specified in package.json'.format(dependency_name))

        with open(package_json_path, 'w') as package_json_write:
            json.dump(modified_package_json_data, package_json_write)

        logging.info("Redundant dependencies removal finished")
        logging.info("Here modified package.json:")

        with sdk2.helpers.ProcessLog(self, logger=logging) as process_log:
            subprocess.check_call(
                ['node', '-e',
                    'console.log(JSON.stringify(require("{}"), null, 4))'.format(package_json_path)],
                cwd=project_path,
                stdout=process_log.stdout,
                stderr=process_log.stderr,
            )

    def _install_npm_dependencies(self, project_path):
        self._remove_redundant_dependencies_from_package_json(project_path)

        logging.info("Dependencies installation started")

        with sdk2.helpers.ProcessLog(self, logger=logging) as process_log:
            subprocess.check_call(
                ['npm', 'install', '--registry=https://npm.yandex-team.ru/'],
                cwd=project_path,
                stdout=process_log.stdout,
                stderr=process_log.stderr,
            )

            # undo changes in package.json
            package_json_path = os.path.join(project_path, 'package.json')
            subprocess.check_call(
                ['arc', 'checkout', 'HEAD', package_json_path],
                cwd=project_path,
                stdout=process_log.stdout,
                stderr=process_log.stderr,
            )

        logging.info("Dependencies installation finished")

    def _tanker_push(self, project_path):
        tanker_branch = self.Parameters.tanker_branch

        with sdk2.helpers.ProcessLog(self, logger=logging) as process_log:
            env = os.environ
            env["TANKER_OAUTH_TOKEN"] = self._get_tanker_oauth_token()
            env["TANKER_BRANCH"] = tanker_branch

            logging.info('Tanker Branch to push {}'.format(tanker_branch))
            logging.info("Tanker-kit configuration:")

            subprocess.check_call(
                ['npx', 'tanker', 'conf'],
                cwd=project_path,
                stdout=process_log.stdout,
                stderr=process_log.stderr,
                env=env
            )

            private_ssh_key = self._get_private_ssh_key()

            with sdk2.ssh.Key(private_part=private_ssh_key):
                logging.info("Tanker push subprocess started")

                subprocess.check_call(
                    ['npm', 'run', 'tanker:push'],
                    cwd=project_path,
                    stdout=process_log.stdout,
                    stderr=process_log.stderr,
                    env=env
                )

        logging.info("Tanker push subprocess finished")

    def _tanker_pull(self, project_path, arcadia_root):
        arc = arc_client.Arc()

        with sdk2.helpers.ProcessLog(self, logger=logging) as process_log:
            env = os.environ
            env["TANKER_OAUTH_TOKEN"] = self._get_tanker_oauth_token()
            env["TANKER_BRANCH"] = self.Parameters.tanker_branch
            env["YA_TOKEN"] = self._get_ya_token()
            env["YA_USER"] = ROBOT_RMP_NAME

            logging.info('Sync ticket {}'.format(self.Parameters.translate_sync_ticket))
            logging.info("Tanker-kit configuration:")

            subprocess.check_call(
                ['npx', 'tanker', 'conf'],
                cwd=project_path,
                stdout=process_log.stdout,
                stderr=process_log.stderr,
                env=env
            )

            private_ssh_key = self._get_private_ssh_key()

            with sdk2.ssh.Key(private_part=private_ssh_key):
                logging.info("Tanker pull subprocess started")

                subprocess.check_call(
                    ['npm', 'run', 'tanker:pull'],
                    cwd=project_path,
                    stdout=process_log.stdout,
                    stderr=process_log.stderr,
                    env=env
                )

            status = arc.status(arcadia_root, False)
            logging.info('Arc status: {}'.format(status))

            status = arc.status(arcadia_root, True)
            if status.get('status'):
                arc_branch = 'users/{author}/{ticket}'.format(
                    author=ROBOT_RMP_NAME,
                    ticket=self.Parameters.translate_sync_ticket
                )

                arc.checkout(arcadia_root, arc_branch, True)

                arc.commit(
                    arcadia_root,
                    "feat(yfa): sync translate",
                    all_changes=True
                )

                arc.push(
                    arcadia_root,
                    upstream=arc_branch,
                    force=True,
                )

                pr = arc.pr_create(
                    arcadia_root,
                    message='{}: sync translate'.format(self.Parameters.translate_sync_ticket),
                    publish=True,
                    auto=True
                )
                logging.info(pr)

                # workaround to force merge pr, more info https://st.yandex-team.ru/DEVTOOLSSUPPORT-13443
                match_id = re.search(r'(?<=a\.yandex\-team\.ru\/review\/)\d+', pr)
                if match_id:
                    pr_id = match_id.group()
                    subprocess.check_call(
                        [
                            os.path.join(arcadia_root, 'ya'),
                            'pr',
                            'merge',
                            '--wait',
                            '--force',
                            '--now',
                            pr_id
                        ],
                        cwd=project_path,
                        stdout=process_log.stdout,
                        stderr=process_log.stderr,
                        env=env
                    )

        logging.info("Tanker pull subprocess finished")

    def on_execute(self):
        self._prepare_env()

        arc_oauth_token = self._get_arc_oauth_token()
        arc_branch_url = 'arcadia-arc:/#{}'.format(self.Parameters.arc_branch)

        with arcadia_sdk.mount_arc_path(arc_branch_url, arc_oauth_token=arc_oauth_token) as arcadia_root:
            project_path = os.path.join(arcadia_root, RELATIVE_PROJECT_PATH)

            logging.info('Changing path to {}'.format(project_path))
            os.chdir(project_path)

            self._install_npm_dependencies(project_path)

            if self.Parameters.tanker_operation == "push":
                self._tanker_push(project_path)
            elif self.Parameters.tanker_operation == "pull":
                self._tanker_pull(project_path, arcadia_root)
            else:
                raise TaskFailure(
                    'Tanker operation {} does not have implementation'.format(self.Parameters.tanker_operation)
                )
