import collections
import json
import logging
import os
import re

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common.backend_build_pipeline.BuildAndReleaseOnCommitNotifier import Action, with_notify_parameters, send_notification
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.svn import Arcadia


ARCADIA_FILES = 'arcadia_files'
ARCADIA_RMP_BACKEND_ROOT = 'yabs/rmp/backend'
ARCADIA_SETTINGS_FILE = os.path.join(ARCADIA_RMP_BACKEND_ROOT, 'src/uac/settings/settings.py')
ARCADIA_CONF_LOCAL_DIR = os.path.join(ARCADIA_RMP_BACKEND_ROOT, 'conf_local')
CONF_LOCAL_NAME = 'Conf.local'

CODE_PATTERN = re.compile(r'\s*check_environment_variable\s*\(\s*[\'\"](?P<var>[^\'\"]+)[\'\"]')
CONFIG_PATTERN = re.compile(r'\s*ENV_(?P<var>[^=]+)=')


def list_variables(lines, pattern):
    envs = set()
    for line in lines:
        match = re.match(pattern, line)
        if match:
            envs.add(match.group('var'))
    return envs


class ValidateUacServicesEnvSpec(sdk2.Task):
    class Parameters(with_notify_parameters(sdk2.Parameters)):
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl()
        dashboard_name = sdk2.parameters.String('Services dashboard name', default='uac_backend')
        dashboard_groups = sdk2.parameters.List('Services dashboard groups', default=['production'])
        container_names = sdk2.parameters.List('Service container names to validate', default=['uac'])
        nanny_api_endpoint = sdk2.parameters.String('Nanny api url', default='https://nanny.yandex-team.ru/')
        nanny_token_vault = sdk2.parameters.String('Nanny token vault', default='nanny_oauth_token')

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        os.mkdir(ARCADIA_FILES)

        arcadia_path = Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url).path
        base_settings_file_name = os.path.basename(ARCADIA_SETTINGS_FILE)
        settings_file_name = os.path.join(ARCADIA_FILES, base_settings_file_name)
        Arcadia.export(
            Arcadia.replace(
                self.Parameters.checkout_arcadia_from_url,
                path=os.path.join(arcadia_path, ARCADIA_SETTINGS_FILE),
            ),
            settings_file_name,
        )

        with open(settings_file_name) as f:
            required_vars = list_variables(f.readlines(), CODE_PATTERN)

        base_conf_local_dir_name = os.path.basename(ARCADIA_CONF_LOCAL_DIR)
        conf_local_dir_name = os.path.join(ARCADIA_FILES, base_conf_local_dir_name)
        Arcadia.export(
            Arcadia.replace(
                self.Parameters.checkout_arcadia_from_url,
                path=os.path.join(arcadia_path, ARCADIA_CONF_LOCAL_DIR),
            ),
            conf_local_dir_name,
        )

        nanny_oauth_token = sdk2.Vault.data(self.owner, self.Parameters.nanny_token_vault)
        nanny_client = nanny.NannyClient(api_url=self.Parameters.nanny_api_endpoint, oauth_token=nanny_oauth_token)
        nanny_services = nanny_client.get_dashboard_services_groups(
            self.Parameters.dashboard_name,
            self.Parameters.dashboard_groups
        )
        logging.debug(
            'Dashboard {} groups {} services {}',
            self.Parameters.dashboard_name,
            ', '.join(self.Parameters.dashboard_groups),
            ', '.join(nanny_services)
        )
        total_diff = collections.defaultdict(lambda: dict())  # service -> container -> envs

        for nanny_service in nanny_services:
            runtime_attrs = nanny_client.get_service_runtime_attrs(nanny_service)
            logging.debug(
                'Runtime attr for {}:\n{}',
                nanny_service,
                json.dumps(runtime_attrs, indent=4)
            )
            resource_type = None
            conf_local_vars = set()
            for sandbox_file in runtime_attrs.get('content', {}).get('resources', {}).get('sandbox_files', []):
                if sandbox_file.get('local_path') == CONF_LOCAL_NAME:
                    resource_type = sandbox_file.get('resource_type')
                    break
            if resource_type:
                resource_cls = sdk2.Resource[resource_type]
                conf_local_file_name = os.path.join(conf_local_dir_name, resource_cls.arcadia_build_name)
                with open(conf_local_file_name) as f:
                    conf_local_vars = list_variables(f.readlines(), CONFIG_PATTERN)

            for container in runtime_attrs.get('content', {}).get('instance_spec', {}).get('containers', []):
                container_name = container.get('name')
                if not container_name or container_name not in self.Parameters.container_names:
                    logging.debug('Skipping container {} of service {}', container_name, nanny_service)
                    continue
                container_vars = set([env.get('name') for env in container.get('env', [])])
                diff = required_vars - conf_local_vars - container_vars
                if diff:
                    total_diff[nanny_service][container_name] = diff

        if total_diff:
            lines = ['Missing env variables in services:']
            for service_name, containers in total_diff.items():
                for container, envs in containers.items():
                    lines.append('\t{service_name} ({container}): {envs}'.format(
                        service_name=service_name,
                        container=container,
                        envs=', '.join(envs)
                    ))
            self.set_info('\n'.join(lines))
            send_notification(
                self,
                'Missing env variables in services. More details in {}'.format(
                    lb.task_link(self.id, 'Sandbox task info')
                ),
                actions=[Action(
                    text=nanny_service,
                    url=lb.nanny_service_link(nanny_service, link_type=lb.LinkType.plain)
                ) for nanny_service in total_diff.keys()]
            )
            raise errors.TaskFailure('Missing env variables')
