# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from datetime import datetime, date
from multiprocessing.dummy import Pool as ThreadPool

import sandbox.common.types.client as ctc
import sandbox.sandboxsdk.environments as sdk_environments
from sandbox import sdk2, common
from sandbox.projects.robot_adapter.achievements.achievery_api import AchieveryApi, AchievementIds
from sandbox.projects.robot_adapter.achievements.common import ApiMode
from sandbox.projects.robot_adapter.achievements.staff_api import StaffApi
from sandbox.projects.robot_adapter.common.email_utils import send_email

DATE_FORMAT = '%Y-%m-%dT%H:%M:%S'

QUEUE = "ADAPTINTERN"
ROBOT_LOGIN = "robot-adapter"
STAFF_TOKEN_NAME = "staff_token"
STARTREK_TOKEN_NAME = "startrek_token"


TICKET_STATUS = [
    'addedToStaff',
    'probationOne',
    'probationTwo',
]


def token(token_name):
    try:
        return sdk2.Vault.data(ROBOT_LOGIN, token_name)
    except common.errors.VaultError as err:
        logging.error(err)
        raise common.errors.TaskFailure("Error occurred on attempt to get the OAuth token for Staff")


def auth_header():
    return {'Authorization': 'OAuth {}'.format(token(STAFF_TOKEN_NAME))}


class RobotAdapterAchievements(sdk2.Task):

    client = None
    skipped_achievements = None

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        environments = [sdk_environments.PipEnvironment('startrek_client')]

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.CheckGroup('Ticket status') as status:
            for st in TICKET_STATUS:
                status.values[st] = st

    @staticmethod
    def achievement_is_given(args):
        login, city = args
        info = StaffApi.person_full_info(
            login=login,
            headers=auth_header())
        try:
            join_at = date.fromisoformat(info['official']['join_at'])
        except Exception as e:
            logging.info("{}".format(e))
            return False, (login, city)
        achievement_info = AchieveryApi.person_achievement(
            login=login,
            achievement_id=AchievementIds.Intern,
            headers=auth_header())
        if achievement_info is None or "events" not in achievement_info or not achievement_info["events"]:
            return False, (login, city)
        try:
            given_at = achievement_info["events"][0]["happened_at"]
            given_at = datetime.strptime(given_at, DATE_FORMAT).date()
        except (TypeError, KeyError, IndexError) as e:
            logging.error("{}".format(e))
            return False, (login, city)
        return join_at <= given_at, (login, city)

    @staticmethod
    def current_level(login):
        achievement_info = AchieveryApi.person_achievement(
            login=login,
            achievement_id=AchievementIds.Intern,
            headers=auth_header()
        )
        if achievement_info is None:
            return 0
        return achievement_info["level"]

    def give_achievement(self, args):
        i, (login, city) = args
        new_level = self.current_level(login=login) + 1
        year = datetime.now().year
        logging.info("{}: {}".format(login, new_level))
        if new_level != 1:
            self.skipped_achievements.append(login)
            logging.info("skipping...")
            return
        self.set_info("{}: {}".format(login, new_level))
        AchieveryApi.give_achievement(
            login=login,
            achievement_id=AchievementIds.Intern,
            level=new_level,
            comment="{}, {}".format(city, year),
            headers=auth_header()
        )

    def extract_interns_data(self):
        from startrek_client import Startrek
        if self.client is None:
            self.client = Startrek(useragent="robot-adapter", token=token(STARTREK_TOKEN_NAME))
        interns = self.client.issues.find(filter=self.filters())

        logins = []
        for intern in interns:
            logins.append(self.parse_login(intern.summary))
        return logins

    @staticmethod
    def parse_login(summary):
        return summary.split('(')[-1].split(')')[0].rstrip('@')

    def filters(self):
        return dict(
            queue=QUEUE,
            status=list(self.Parameters.status)
        )

    def on_execute(self):
        self.skipped_achievements = []

        StaffApi.API_MODE = ApiMode.Production
        AchieveryApi.API_MODE = ApiMode.Production

        logins_data = self.extract_interns_data()
        pool = ThreadPool(5)

        total = 0
        without_achievement = []
        for is_given, (login, city) in pool.map(self.achievement_is_given, logins_data):
            logging.info("{}: {}".format(login, is_given))
            if not is_given:
                without_achievement.append((login, city))
            total += is_given
        logging.info('total without achievement: {}'.format(total))

        for _ in pool.map(lambda args: self.give_achievement(args),
                          enumerate([(login, city) for (login, city) in without_achievement])):
            pass

        if self.skipped_achievements:
            template = 'Привет, Даша!\n\nПомоги, пожалуйста! Я не умею, а этим ребятам нужно апнуть ачивку:\n{}'
            send_email(
                subject='Помоги апнуть ачивку стажерам',
                to_emails=['daryapal@yandex-team.ru'],
                text=template.format('\n'.join('{}@'.format(login) for login in self.skipped_achievements))
            )
            logging.info('Achievements awaiting manual up: {}'.format(self.skipped_achievements))
        logging.info("Done")
