# -*- coding: utf-8 -*-
from __future__ import print_function
from __future__ import unicode_literals

import logging
import sandbox.common.types.client as ctc
import sandbox.sandboxsdk.environments as sdk_environments
from collections import namedtuple
from datetime import timedelta, datetime
from sandbox import sdk2, common

try:
    from typing import List, AnyStr, Optional  # noqa
except ImportError:
    pass


DRY_RUN = True
ROBOT_LOGIN = "robot-adapter"
SANDBOX_SECRET_NAME = "startrek_token"
QUEUE = 'ADAPTINTERN'
TEMPLATE1 = "Привет!\nУ тебя в команде уже месяц стажируется {intern}. " \
            "Расскажи, как идут дела? Справляется ли с задачами? Как складываются отношения с коллективом? " \
            "Есть ли какие-нибудь проблемы?\n" \
            "Если куратором является кто-то из коллег, то сориентируй пожалуйста, кто именно?\nСпасибо!\n"
TEMPLATE2 = "Привет!\n" \
            "\n" \
            "У твоего стажёра скоро заканчивается стажировка.\n" \
            "Нам важно понять, рекомендуешь ли ты его на позицию младшего специалиста в своей команде или в другой, " \
            "чтобы запланировать собеседования для перевода в штат.\n" \
            "\n" \
            "Пожалуйста, **сегодня** напиши о своем решении в формате:\n" \
            "\n" \
            "* я **оставляю** стажёра **у себя** в команде \n" \
            "* я рекомендую его **в другую команду**\n" \
            "* я хочу, чтобы стажёр остался в моей команде, но **не знаю, есть ли для него вакансия**\n" \
            "\n" \
            "\n" \
            "* я **не рекомендую** этого стажёра на другие вакансии в компании \n" \
            "\n" \
            "Эта информация нужна нам в первую очередь для назначения АА-секции.\n" \
            "__Сейчас много ребят заканчивают стажировку одновременно, " \
            "поэтому мы планируем интервью по алгоритмам заранее.__\n" \
            "\n" \
            "Напиши, какой именно вариант ты выбрал уже сейчас. \n" \
            "Через 3 дня мы попросим тебя поделиться подробной обратной связью о результатах стажировки.\n" \
            "\n" \
            "Спасибо!"
TEMPLATE2_REMINDER = "Привет!\n" \
                     "\n" \
                     "У твоего стажера скоро заканчивается стажировка.\n" \
                     "Поделись, пожалуйста, впечатлениями от совместной работы:\n" \
                     "\n" \
                     "1. Как стажёр справлялся с задачами? Что успел сделать за время стажировки? \n" \
                     "2. Какие сильные и слабые стороны ты можешь отметить?\n" \
                     "\n" \
                     "**Если ты еще не написал нам о том, готов ли рекомендовать стажера на вакансию младшего " \
                     "специалиста, пожалуйста, напиши об этом сейчас.**\n" \
                     "\n" \
                     "Без твоего решения мы не сможем назначить АА-секцию для перевода в штат.\n" \
                     "\n" \
                     "В ближайшее время мы пригласим твоего стажера на встречу, где расскажем о дальнейших шагах в " \
                     "последний месяц стажировки.\n" \
                     "\n" \
                     "Спасибо!\n"
DATE_FORMAT = '%Y-%m-%d'
INDENT = "   |"


InvalidIssue = namedtuple('InvalidIssue', ['key', 'reason'])
RelativeDate = namedtuple('RelativeDate', ['relation', 'delta'])
NameKey = namedtuple('NameKey', ['name', 'key'])

# Special value status.
NoChange = NameKey('No change', None)


class DateRelation:
    START = 'start'
    END = 'end'


class Tags:

    def __init__(self, add_tags=None, del_tags=None, search_tags=None):
        # type: (Optional[list], Optional[list], Optional[list]) -> ...
        self.add_tags = add_tags or []
        self.del_tags = del_tags or []
        self.search_tags = search_tags or []

    def __repr__(self):
        return ("Tags[" +
                ", ".join("+'{}'".format(tag) for tag in self.add_tags) +
                (", " if self.add_tags and self.del_tags else "") +
                ", ".join("-'{}'".format(tag) for tag in self.del_tags) +
                (", " if (self.del_tags or self.add_tags) and self.search_tags else "") +
                ", ".join("?'{}'".format(tag) for tag in self.search_tags) +
                "]")

    def __nonzero__(self):
        return bool(self.add_tags or self.del_tags or self.search_tags)

    def tags(self):
        return self.add_tags


class State:

    def __init__(self, status, tags=None):
        # type: (NameKey, Optional[Tags]) -> ...
        self.status = status
        self.tags = tags or Tags()

    def __repr__(self):
        return ("State(status={}".format(self.status) +
                (", tags={}".format(self.tags) if self.tags else "") + ")")


class IssueTransition:

    def __init__(self, from_state, to_state, template, rel_date):
        # type: (State, State, AnyStr, RelativeDate) -> ...
        self.from_state = from_state
        self.to_state = to_state
        self.template = template
        self.rel_date = rel_date

    def __repr__(self):
        return "IssueTransition({indent}from_sate={},{indent}to_state={},{indent}rel_date={}\n)" \
               .format(self.from_state, self.to_state, self.rel_date, indent="\n    ")


# List of transitions.
PROBATION1 = IssueTransition(
    from_state=State(status=NameKey('Added to Staff', 'addedToStaff')),
    to_state=State(status=NameKey('Probation 1', 'probationOne')),
    template=TEMPLATE1,
    rel_date=RelativeDate(DateRelation.START, timedelta(days=31))
)
PROBATION2 = IssueTransition(
    from_state=State(status=NameKey('Probation 1', 'probationOne')),
    to_state=State(status=NameKey('Probation 2', 'probationTwo'),
                   tags=Tags(add_tags=['awaitingReminder'])),
    template=TEMPLATE2,
    rel_date=RelativeDate(DateRelation.END, timedelta(days=38))
)
PROBATION2_REMINDER = IssueTransition(
    from_state=State(status=NameKey('Probation 2', 'probationTwo'),
                     tags=Tags(search_tags=['awaitingReminder'])),
    to_state=State(status=NoChange,
                   tags=Tags(del_tags=['awaitingReminder'])),
    template=TEMPLATE2_REMINDER,
    rel_date=RelativeDate(DateRelation.END, timedelta(days=35))
)


class AIBot(sdk2.Task):  # AdaptInternBot :)
    """Update issues in ADAPTINTERN Startrek queue using robot-adapter bot."""

    description = 'Update issues in {} queue using robot-adapter bot.'.format(QUEUE)
    client = None
    dry_run = DRY_RUN

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        environments = [sdk_environments.PipEnvironment('startrek_client'), ]

    class Parameters(sdk2.Task.Parameters):
        dry_run = sdk2.parameters.Bool(
            "Dry run mode",
            required=True,
            description="Dry run mode: only prints actions in sandbox task logs, instead of executing them",
            default=DRY_RUN
        )
        transitions = [PROBATION1, PROBATION2, PROBATION2_REMINDER]

    def on_execute(self):
        from startrek_client import Startrek
        self.dry_run = self.Parameters.dry_run

        transitioned_issues = []
        dryrunned_issues = []
        invalid_issues = []

        self.client = Startrek(useragent='robot-adapter', token=self.get_token())
        self.set_info("Performing task as {}@".format(self.client.myself.login))
        self.set_info("Dry run mode is {}".format("ON" if self.dry_run else "OFF"))
        self.set_info("Processing queue {}".format(QUEUE))

        for transition in self.Parameters.transitions:
            self.set_info("Applying {}".format(transition))
            issues = list(self.issues(transition))
            self.set_info("Found {} issues with filters {}".format(len(issues), self.get_filters(transition)))
            for issue in issues:
                try:
                    self.set_info(INDENT + "Validating {} issue".format(issue.key))
                    is_valid = self.validate_issue_for_transition(issue, transition)
                    if not is_valid:
                        self.set_info(INDENT + "Skipping {} issue...".format(issue.key))
                        continue
                    names = self.extract_names(issue)
                    manager_login = names['manager']  # issue description contains only login for manager :/
                    intern_login, intern_name = names['intern']
                    self.set_info(INDENT + "Manager(login='{}'), Intern(login='{}', name='{}')"
                                  .format(manager_login, intern_login, intern_name))

                    time_diff = self.now() - AIBot.transition_date(issue, transition)
                    if self.dry_run:
                        self.set_info(INDENT + "DRYRUN: Would apply transition '{}' for {} issue"
                                      .format(transition, issue.key))
                        dryrunned_issues.append({'transition': transition, 'key': issue.key, 'time_diff': time_diff})
                    else:
                        self.set_info(INDENT + "Applying transition to '{}' for {} issue..."
                                      .format(transition.to_state.status.name, issue.key))
                        issue.comments.create(text=transition.template.format(intern="(({}@ {}))"
                                                                              .format(intern_login, intern_name)),
                                              summonees=[manager_login])
                        if transition.to_state.status is not NoChange:
                            issue.transitions[transition.to_state.status.key].execute()
                        self.update_tags(issue, transition.to_state.tags)
                        transitioned_issues.append({'transition': transition, 'key': issue.key, 'time_diff': time_diff})
                except Exception as e:
                    invalid_issues.append(InvalidIssue(issue.key, e))

        processed_issues = transitioned_issues or dryrunned_issues
        processed_issues.sort(key=lambda issue: issue['time_diff'], reverse=True)
        if processed_issues:
            self.set_info("Processed {dryrun}{count} issues:\n    {issues}"
                          .format(dryrun="(dry run mode) " if self.dry_run else "",
                                  count=len(processed_issues),
                                  issues="\n    ".join("{key} as '{transition}' as {time_diff} passed"
                                                       .format(**processed_issues)
                                                       for processed_issues in processed_issues)))
        else:
            self.set_info("Queue has no issues to process")

        if invalid_issues:
            self.set_info("Also found {count} invalid issues:\n    {issues}"
                          .format(count=len(invalid_issues),
                                  issues="\n    ".join("{0.key}: {0.reason}".format(invalid_issue)
                                                       for invalid_issue in invalid_issues)))

        self.set_info('Task successfully done.')

    def issues(self, transition):
        # type: (IssueTransition) -> ...
        return self.client.issues.find(filter=self.get_filters(transition))

    def validate_issue_for_transition(self, issue, transition):
        # type: (..., IssueTransition) -> ...
        """
        Checks issue transitions, current state, title format and issue has
        manager field.
        :param issue: issue to validate
        :param transition: transition to validate for
        :return: True if issue fits criteria, false otherwise
        """
        transitions = issue.transitions.get_all()

        has_transition = any(transition_i.to.key == transition.to_state.status.key for transition_i in transitions)
        has_transition = has_transition or transition.to_state.status is NoChange
        correct_status = issue.status.key == transition.from_state.status.key
        transition_time_diff = self.now() - AIBot.transition_date(issue, transition)

        if not has_transition:
            self.set_info(INDENT * 2 + "Issue has no transition '{}'".format(transition.to_state.status.name))
            return False
        if not correct_status:
            self.set_info(INDENT * 2 + "Issue isn't in status '{}' for transition '{}'"
                          .format(transition.from_state.status.name, transition.to_state.status.name))
            return False
        if transition_time_diff < timedelta(seconds=0):
            self.set_info(INDENT * 2 + "Time left for transition: {}".format(-transition_time_diff))
            return False
        self.set_info(INDENT * 2 + "Transition can be done already as: {}".format(transition_time_diff))
        return True

    def update_tags(self, issue, tags):
        # type: (..., Tags) -> ...
        if not tags:
            return
        current_tags = self.client.issues[issue.key].tags
        new_tags = (set(current_tags) - set(tags.del_tags)) | set(tags.add_tags)
        self.client.issues[issue.key].update(tags=list(new_tags))

    @staticmethod
    def now():
        return datetime.now().replace(microsecond=0)

    @staticmethod
    def get_filters(transition):
        # type: (IssueTransition) -> dict
        return dict(queue=QUEUE,
                    status=transition.from_state.status.key,
                    tags=transition.from_state.tags.search_tags)

    @staticmethod
    def get_token():
        try:
            token = sdk2.Vault.data(ROBOT_LOGIN, SANDBOX_SECRET_NAME)
        except common.errors.VaultError as err:
            logging.error(err)
            raise common.errors.TaskFailure("Error occurred on attempt to get the OAuth token for Startrek")
        return token

    @staticmethod
    def transition_date(issue, transition):
        # type: (..., IssueTransition) -> ...
        if issue.start is None:
            raise ValueError("Issue has empty 'Start Date' field")
        if issue.end is None:
            raise ValueError("Issue has empty 'End Date' field")
        if transition.rel_date.relation == DateRelation.START:
            return datetime.strptime(issue.start, DATE_FORMAT) + transition.rel_date.delta
        if transition.rel_date.relation == DateRelation.END:
            return datetime.strptime(issue.end, DATE_FORMAT) - transition.rel_date.delta
        raise ValueError("transition.rel_date.relation has unknown value {}".format(transition.rel_date.relation))

    @staticmethod
    def extract_names(issue):
        MANAGER = 'Руководитель'
        INTERN_NAME = "ФИО"
        desc = issue.description
        manager_login = None
        intern_login = None
        intern_name = None
        for table_line in desc.split('\n'):
            if MANAGER in table_line:
                manager_login = AIBot.extract_second_col(table_line)
            if INTERN_NAME in table_line:
                name_and_login = AIBot.extract_second_col(table_line)
                intern_login = name_and_login.split()[-1].strip()
                intern_name = ' '.join(name_and_login.split()[:-1]).strip()
        if manager_login is None:
            raise ValueError("couldn't retrieve manager login from issue description")
        if intern_login is None:
            raise ValueError("couldn't retrieve intern login from issue description")
        manager_login = manager_login.replace('@', '')
        intern_login = intern_login.replace('@', '')
        return {'manager': manager_login, 'intern': (intern_login, intern_name)}

    @staticmethod
    def extract_second_col(table_line):
        tokens = [token for token in table_line.split('|') if token.strip()]
        return tokens[1].strip()
