# -*- coding: utf-8 -*-
from __future__ import print_function
from __future__ import unicode_literals

import logging
import sandbox.common.types.client as ctc
import sandbox.sandboxsdk.environments as sdk_environments
from datetime import timedelta, datetime
from sandbox import sdk2, common

QUEUE = "ADAPTINTERN"
DATE_FORMAT = "%Y-%m-%d"
ROBOT_LOGIN = "robot-adapter"
SANDBOX_SECRET_NAME = "startrek_token"
EPIC_TASK_KEY = "ADAPTINTERN-2594"
ASSIGNEE_LOGIN = "daryapal"
FOLLOWERS = [
    "daryapal",
    "kpirogova",
    "agrebenyuk",
    "nastia-i",
    "juliasorokina",
    "gg0sha",
    "robot-help",
    "yumatyushina",
    "rozochka1",
    "letenkov",
    "polyakova-hr",
    "stribog",
    "kozymasha",
    "sing1etone",
    "fn-555",
    "robot-adapter"
]
TAGS = ["царь"]
SUMMARY_TEMPLATE = "Выход стажеров {day} {month}"
months = ["", "января", "февраля",
          "марта", "апреля", "мая",
          "июня", "июля", "августа",
          "сентября", "октября", "ноября",
          "декабря"]


class Relationship:
    """
      https://wiki.yandex-team.ru/tracker/api/issues/links/create/#relationship
    """
    relates = "relates"
    is_subtask_for = "is subtask for"
    is_parent_task_for = "is parent task for"


class FixVersions:
    """
      Numbers are taken from ADAPTINTERN queue. Can be extracted from url
      params on filter requests.
    """
    Moscow = 80019
    SaintPetersburg = 80012


class AIStartScheduler(sdk2.Task):
    client = None

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        environments = [sdk_environments.PipEnvironment('startrek_client')]

    class Parameters(sdk2.Task.Parameters):
        interval = sdk2.parameters.Integer(
            "Interval to search interns (in days)",
            required=True,
            description="Interval to search intern tickets from now",
            default=6
        )

    def on_execute(self):
        from startrek_client import Startrek
        # Initializing.
        self.client = Startrek(useragent="robot-adapter", token=self.token())
        self.set_info("Performing task as {}@"
                      .format(self.client.myself.login))
        self.set_info("Processing queue {}".format(QUEUE))

        # Searching for intern tickets.
        new_interns = list(self.find_new_interns(
            interval=timedelta(days=self.Parameters.interval)
        ))
        # Print number of found issues.
        self.set_info("Found {} issues".format(len(new_interns)))
        # Filter interns by common start date.
        common_start = self.extract_common_start_date(new_interns)
        filtered_interns = self.filter_issues_by(start_date=common_start,
                                                 issues=new_interns)
        skipped_interns = self.find_skipped(filtered=filtered_interns,
                                            unfiltered=new_interns)
        # Logging skipped filters.
        self.set_info(
            "Skipping issues: {}".format(
                "\n".join(
                    "{key} ({start})".format(key=issue.key, start=issue.start)
                    for issue in skipped_interns
                )
            )
        )
        # Replace new_interns with filtered.
        new_interns = filtered_interns
        # Sort for convenient logging.
        new_interns.sort(key=lambda issue: self.str_to_date(issue.start))
        self.set_info("Found {} intern tickets".format(len(new_interns)))

        # Generating new intern group ticket.
        summary = self.generate_summary(new_interns)
        description = self.generate_description(new_interns)
        self.set_info("Creating new issue:\n\nSummary: {}\n\nDescription:\n{}"
                      .format(summary, description))
        group_ticket = self.client.issues.create(
            **self.make_interns_ticket(summary, description)
        )

        # Linking created ticket to epic.
        self.set_info("Created {}. Linking it to {}"
                      .format(group_ticket.key, EPIC_TASK_KEY))
        group_ticket.links.create(relationship=Relationship.is_subtask_for,
                                  issue=EPIC_TASK_KEY)

        # Linking intern tickets to group ticket.
        self.set_info("Linking intern tickets:")
        for intern_issue in new_interns:
            self.set_info("  {key}".format(key=intern_issue.key))
            group_ticket.links.create(
                relationship=Relationship.is_parent_task_for,
                issue=intern_issue.key)
        self.set_info("Task successfully done.")

    def find_new_interns(self, interval=None):  # type: (timedelta) -> ...
        if interval is None:
            interval = timedelta(days=6)
        return self.client.issues.find(filter=self.filters(interval))

    @staticmethod
    def generate_summary(new_interns):
        common_start = AIStartScheduler.extract_common_start_date(new_interns)
        start_date = AIStartScheduler.str_to_date(common_start)
        return SUMMARY_TEMPLATE.format(day=start_date.day,
                                       month=months[start_date.month])

    @staticmethod
    def generate_description(new_interns):
        return "\n".join(
            "{}. {}".format(i + 1, intern_issue.summary)
            for i, intern_issue in enumerate(new_interns)
        )

    @staticmethod
    def filter_issues_by(start_date, issues):
        return [issue for issue in issues if issue.start == start_date]

    @staticmethod
    def find_skipped(filtered, unfiltered):
        skipped = []
        filtered_keys = set(issue.key for issue in filtered)
        for issue in unfiltered:
            if issue.key not in filtered_keys:
                skipped.append(issue)
        return skipped

    @staticmethod
    def str_to_date(s):
        return datetime.strptime(s, DATE_FORMAT)

    @staticmethod
    def make_interns_ticket(summary, description):
        # type: (str, str) -> dict
        return dict(
            queue=QUEUE,
            type=dict(name="Task"),
            assignee=ASSIGNEE_LOGIN,
            summary=summary,
            description=description,
            followers=FOLLOWERS,
            tags=TAGS
        )

    @staticmethod
    def extract_common_start_date(issues):
        start_dates = [issue.start for issue in issues]
        return max(set(start_dates), key=start_dates.count)

    @staticmethod
    def now():
        return datetime.now().replace(microsecond=0)

    @staticmethod
    def filters(interval):
        # type: (timedelta) -> dict
        return dict(
            queue=QUEUE,
            status="approvedByHead",
            start={"from": AIStartScheduler.now().strftime(DATE_FORMAT),
                   "to": (AIStartScheduler.now() + interval).strftime(DATE_FORMAT)}
        )

    @staticmethod
    def token():
        try:
            token = sdk2.Vault.data(ROBOT_LOGIN, SANDBOX_SECRET_NAME)
        except common.errors.VaultError as err:
            logging.error(err)
            raise common.errors.TaskFailure(
                "Couldn't get OAuth token for Startrek")
        return token
