# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from sandbox.projects.robot_adapter.common.email_utils import (
    YANDEX_TEAM_EMAIL, send_email, read_email_template, make_attachment
)
import sandbox.common.types.client as ctc
import sandbox.sandboxsdk.environments as sdk_environments
from sandbox import sdk2, common


QUEUE = "ADAPTINTERN"
ROBOT_LOGIN = "robot-adapter"
SANDBOX_SECRET_NAME = "startrek_token"


class Tags:
    awaitingWelcomeEmail = 'awaitingWelcomeEmail'


class RobotAdapterWelcomeEmail(sdk2.Task):

    client = None

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        environments = [sdk_environments.PipEnvironment('startrek_client')]

    def on_execute(self):
        from startrek_client import Startrek
        # Initializing.
        self.client = Startrek(useragent="robot-adapter", token=self.token())
        self.set_info("Performing task as {}@"
                      .format(self.client.myself.login))
        self.set_info("Processing queue {}".format(QUEUE))
        # Finding interns.
        interns = list(self.find_awaiting_issues())
        self.set_info('Found {} interns awaiting welcome email.'
                      .format(len(interns)))
        self.send_email_to_awaiting(interns=interns)
        self.set_info('Deleting {} tags'.format(Tags.awaitingWelcomeEmail))
        self.delete_tags(interns)
        self.set_info("Task successfully done.")

    def delete_tags(self, interns):
        for intern in interns:
            tags = list(set(intern.tags) - set([Tags.awaitingWelcomeEmail]))
            self.client.issues[intern.key].update(tags=tags)

    def send_email_to_awaiting(self, interns):
        for intern in interns:
            login = self.parse_login(intern.summary)
            email = YANDEX_TEAM_EMAIL.format(login=login)
            self.set_info('Sending email to {}'.format(email))
            send_email(
                subject='Добро пожаловать в Яндекс',
                to_emails=[email],
                html=read_email_template(),
                attachments=[
                    make_attachment(data=self.load_presentation(),
                                    filename='Вводная презентация.pdf')
                ]
            )

    def find_awaiting_issues(self):
        return self.client.issues.find(filter=self.filters())

    @staticmethod
    def load_presentation():
        presentation = sdk2.Resource.find(
            attrs=dict(robot_adapter="introduction.pdf")
        ).first()
        if not presentation:
            raise RuntimeError('Presentation resource wasn\'t found')
        resource_path = str(sdk2.ResourceData(presentation).path)
        logging.info('Resource path: {}'.format(resource_path))
        with open(resource_path, 'rb') as f:
            return f.read()

    @staticmethod
    def parse_login(summary):
        return summary.split('(')[-1].split(')')[0].rstrip('@')

    @staticmethod
    def filters():
        return dict(
            queue=QUEUE,
            status='addedToStaff',
            tags=[Tags.awaitingWelcomeEmail]
        )

    @staticmethod
    def token():
        try:
            token = sdk2.Vault.data(ROBOT_LOGIN, SANDBOX_SECRET_NAME)
        except common.errors.VaultError as err:
            logging.error(err)
            raise common.errors.TaskFailure("Error occurred on attempt to get the OAuth token for Startrek")
        return token
